% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/alpha_fd_multidim.R
\name{alpha.fd.multidim}
\alias{alpha.fd.multidim}
\title{Compute a set of alpha functional indices for a set of assemblages}
\usage{
alpha.fd.multidim(
  sp_faxes_coord,
  asb_sp_w,
  ind_vect = c("fide", "fdis", "fmpd", "fnnd", "feve", "fric", "fdiv", "fori", "fspe"),
  scaling = TRUE,
  check_input = TRUE,
  details_returned = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{sp_faxes_coord}{a matrix of species coordinates in a chosen
functional space. Species coordinates have been retrieved thanks to
\code{\link{tr.cont.fspace}} or \code{\link{quality.fspaces}}.}

\item{asb_sp_w}{a matrix linking weight of species (columns) and a
set of assemblages (rows).}

\item{ind_vect}{a vector of character string of the name of
functional indices to compute. \strong{Indices names must be written in
lower case letters}. Possible indices to compute are: 'fide', fdis',
'fmpd', 'fnnd', 'feve', 'fric', 'fdiv', 'fori' and 'fspe'. Default: all
the indices are computed.}

\item{scaling}{a logical value indicating if scaling is to be done
(TRUE) or not (FALSE) on functional indices. Scaling is used to be able to
compare indices values between assemblages. Default: scaling = TRUE.}

\item{check_input}{a logical value indicating whether key features the
inputs are checked (e.g. class and/or mode of objects, names of rows
and/or columns, missing values). If an error is detected, a detailed
message is returned. Default: \code{check.input = TRUE}.}

\item{details_returned}{a logical value indicating whether the user
want to store details. Details are used in graphical functions and thus
must be kept if the user want to have graphical outputs for the computed
indices.}

\item{verbose}{a logical value indicating whether progress details should be
printed in the console. If \code{FALSE} does not provide percent progress when
computing diversity indices.}
}
\value{
The following list is returned: \itemize{

\item \emph{functional_diversity_indices} matrix containing indices values
(columns) for each assemblage (rows)

\item \emph{details} list: a \strong{asb_sp_occ} data.frame of species
occurrences in each assemblage ; a \strong{asb_sp_relatw} matrix of
relative weight of species in each assemblage ; a \strong{sp_coord_all_asb}
list of matrices of species coordinates along functional axes for species
present in each assemblage ; a \strong{vert_nm_all_asb} list of vectors of
species names being vertices of the convex hull for each assemblage ; a
\strong{mst_all_asb} list of data.frames summarizing link between species
in the minimum spanning tree of each assemblage ; a
\strong{grav_center_vert_coord_all_asb} list of vectors of coordinates of
the vertices gravity center for each assemblage ; a
\strong{mean_dtogravcenter_all_asb} list of vectors containing mean
distance to the species gravity center for each assemblage ; a
\strong{dist_gravcenter_global_pool} vector containing the distance of each
species to the gravity center of all species from the global pool ; a
\strong{dist_nn_global_pool} data.frame showing the distances of each
species from the global pool to its nearest neighbor ; a
\strong{nm_nn_all_asb} data.frame containing the name of each nearest
neighbor of each species present in a given assemblage ; a
\strong{dist_nn_all_asb} data.frame containing distance of each species
present in a given assemblage to its nearest neighbor.}
}
\description{
This function computes a set of multidimensional space based indices of
alpha functional diversity. The user can choose which functional indices to
compute.
}
\examples{
# Load Species*Traits dataframe:
data('fruits_traits', package = 'mFD')

# Load Assemblages*Species dataframe:      
data('baskets_fruits_weights', package = 'mFD')

# Load Traits categories dataframe:
data('fruits_traits_cat', package = 'mFD')  
 
# Compute functional distance 
sp_dist_fruits <- mFD::funct.dist(sp_tr         = fruits_traits,
                                  tr_cat        = fruits_traits_cat,
                                  metric        = "gower",
                                  scale_euclid  = "scale_center",
                                  ordinal_var   = "classic",
                                  weight_type   = "equal",
                                  stop_if_NA    = TRUE)
 
# Compute functional spaces quality to retrieve species coordinates matrix:
fspaces_quality_fruits <- mFD::quality.fspaces(
  sp_dist             = sp_dist_fruits, 
  maxdim_pcoa         = 10,
  deviation_weighting = 'absolute',
  fdist_scaling       = FALSE,
  fdendro             = 'average')
  
# Retrieve species coordinates matrix:
sp_faxes_coord_fruits <- fspaces_quality_fruits$details_fspaces$sp_pc_coord

# Compute alpha diversity indices
alpha_fd_indices_fruits <- mFD::alpha.fd.multidim(
  sp_faxes_coord   = sp_faxes_coord_fruits[, c('PC1', 'PC2', 'PC3', 'PC4')],
  asb_sp_w         = baskets_fruits_weights, 
  ind_vect         = c('fdis', 'fmpd', 'fnnd', 'feve', 'fric', 'fdiv', 
                       'fori', 'fspe'),
  scaling          = TRUE, 
  check_input      = TRUE, 
  details_returned = TRUE)
  
# Retrieve alpha diversity indices table
fd_ind_values_fruits <- alpha_fd_indices_fruits$functional_diversity_indices
fd_ind_values_fruits
}
\author{
Camille Magneville and Sebastien Villeger
}
