% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/maps.R
\name{get_static_tiles}
\alias{get_static_tiles}
\title{Get static tiles from a Mapbox style for use as a basemap}
\usage{
get_static_tiles(
  location,
  zoom,
  style_id,
  username,
  scaling_factor = c("1x", "2x"),
  buffer_dist = 5000,
  crop = TRUE,
  access_token = NULL
)
}
\arguments{
\item{location}{An input location for which you would like to request tiles.
Can be a length-4 vector representing a bounding box, or an sf object.
If an input sf object is supplied, use the \code{buffer_dist} argument to
control how much area you want to capture around the layer.
While the input sf object can be in an arbitrary coordinate reference system,
if a length-4 bounding box vector is supplied instead it must represent
WGS84 longitude/latitude coordinates and be in the order
\code{c(xmin, ymin, xmax, ymax)}.}

\item{zoom}{The zoom level for which you'd like to return tiles.}

\item{style_id}{A Mapbox style ID; retrieve yours from your Mapbox account.}

\item{username}{A Mapbox username.}

\item{scaling_factor}{The scaling factor to use; one of \code{"1x"} or \code{"2x"}.}

\item{buffer_dist}{The distance to buffer around an input sf object for determining tile extent, specified in meters.  Defaults to 5000.}

\item{crop}{Whether or not to crop the result to the specified bounding box or buffer area.
Defaults to \code{TRUE}; \code{FALSE} will return the extent of the overlapping
tiles.}

\item{access_token}{Your Mapbox access token.  Supply yours here or set globally with the \code{mb_access_token()} function.}
}
\value{
A raster layer of tiles from the requested Mapbox style representing the area around the input location.  The raster layer is projected in the Web Mercator coordinate reference system.
}
\description{
This function queries the Mapbox Static Tiles API and composites the tiles as a
raster suitable for use as a basemap in tmap or ggplot2 (with the \code{layer_spatial()}
function in ggspatial).  It returns a raster layer that corresponds either to
an input bounding box or a buffered area around an input shape.
}
\examples{
\dontrun{

library(mapboxapi)
library(tigris)
library(tmap)
library(ggspatial)
library(ggplot2)

ny_tracts <- tracts("NY", "New York", cb = TRUE)

ny_tiles <- get_static_tiles(
  location = ny_tracts,
  zoom = 10,
  style_id = "light-v9",
  username = "mapbox"
)

# tmap usage:
tm_shape(ny_tiles) +
  tm_rgb() +
  tm_shape(ny_tracts) +
  tm_polygons(alpha = 0.5, col = "navy") +
  tm_credits("Basemap (c) Mapbox, (c) OpenStreetMap",
             position = c("RIGHT", "BOTTOM"))

# ggplot2 usage:
ggplot() +
  layer_spatial(ny_tiles) +
  geom_sf(data = ny_tracts, fill = "navy", alpha = 0.5) +
  theme_void() +
  labs(caption = "Basemap (c) Mapbox, (c) OpenStreetMap")

}

}
