% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/showEqn.R
\name{showEqn}
\alias{showEqn}
\title{Show Matrices (A, b) as Linear Equations}
\usage{
showEqn(
  A,
  b,
  vars,
  simplify = FALSE,
  reduce = FALSE,
  fractions = FALSE,
  latex = FALSE
)
}
\arguments{
\item{A}{either the matrix of coefficients of a system of linear equations, or the matrix \code{cbind(A,b)}.
Alternatively, can be of class \code{'lm'} to print the equations for the design matrix in a linear
regression model}

\item{b}{if supplied, the vector of constants on the right hand side of the equations. When omitted
the values \code{b1, b2, ..., bn} will be used as placeholders}

\item{vars}{a numeric or character vector of names of the variables.
If supplied, the length must be equal to the number of unknowns in the equations.
The default is \code{paste0("x", 1:ncol(A)}.}

\item{simplify}{logical; try to simplify the equations?}

\item{reduce}{logical; only show the unique linear equations}

\item{fractions}{logical; express numbers as rational fractions, using the \code{\link[MASS]{fractions}} 
function; if you require greater accuracy, you can set the \code{cycles} (default 10)
and/or \code{max.denominator} (default 2000) arguments to \code{fractions} as a global option, e.g.,
\code{options(fractions=list(cycles=100, max.denominator=10^4))}.}

\item{latex}{logical; print equations in a form suitable for LaTeX output?}
}
\value{
a one-column character matrix, one row for each equation
}
\description{
Shows what matrices \eqn{A, b} look like as the system of linear equations, \eqn{A x = b}, but written out
as a set of equations.
}
\examples{
  A <- matrix(c(2, 1, -1,
               -3, -1, 2,
               -2,  1, 2), 3, 3, byrow=TRUE)
  b <- c(8, -11, -3)
  showEqn(A, b)
  # show numerically
  x <- solve(A, b)
  showEqn(A, b, vars=x)

  showEqn(A, b, simplify=TRUE)
  showEqn(A, b, latex=TRUE)

  # lower triangle of equation with zeros omitted (for back solving)
  A <- matrix(c(2, 1, 2,
               -3, -1, 2,
               -2,  1, 2), 3, 3, byrow=TRUE)
  U <- LU(A)$U
  showEqn(U, simplify=TRUE, fractions=TRUE)
  showEqn(U, b, simplify=TRUE, fractions=TRUE)

  ####################
  # Linear models Design Matricies
  data(mtcars)
  ancova <- lm(mpg ~ wt + vs, mtcars)
  summary(ancova)
  showEqn(ancova)
  showEqn(ancova, simplify=TRUE)
  showEqn(ancova, vars=round(coef(ancova),2))
  showEqn(ancova, vars=round(coef(ancova),2), simplify=TRUE)

  twoway_int <- lm(mpg ~ vs * am, mtcars)
  summary(twoway_int)
  car::Anova(twoway_int)
  showEqn(twoway_int)
  showEqn(twoway_int, reduce=TRUE)
  showEqn(twoway_int, reduce=TRUE, simplify=TRUE)

  # Piece-wise linear regression
  x <- c(1:10, 13:22)
  y <- numeric(20)
  y[1:10] <- 20:11 + rnorm(10, 0, 1.5)
  y[11:20] <- seq(11, 15, len=10) + rnorm(10, 0, 1.5)
  plot(x, y, pch = 16)

  x2 <- as.numeric(x > 10)
  mod <- lm(y ~ x + I((x - 10) * x2))
  summary(mod)
  lines(x, fitted(mod))
  showEqn(mod)
  showEqn(mod, vars=round(coef(mod),2))
  showEqn(mod, simplify=TRUE)

}
\references{
Fox, J. and Friendly, M. (2016). "Visualizing Simultaneous Linear Equations, Geometric Vectors, and
Least-Squares Regression with the matlib Package for R". \emph{useR Conference}, Stanford, CA, June 27 - June 30, 2016.
}
\seealso{
\code{\link{plotEqn}}, \code{\link{plotEqn3d}}
}
\author{
Michael Friendly, John Fox, and Phil Chalmers
}
