% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/concordance_corr.R
\name{ccc}
\alias{ccc}
\alias{print.ccc}
\alias{plot.ccc}
\title{Pairwise Lin's concordance correlation coefficient}
\usage{
ccc(data, ci = FALSE, conf_level = 0.95, verbose = FALSE)

\method{print}{ccc}(x, digits = 4, ...)

\method{plot}{ccc}(
  x,
  title = "Lin's Concordance Correlation Heatmap",
  low_color = "indianred1",
  high_color = "steelblue1",
  mid_color = "white",
  value_text_size = 4,
  ...
)
}
\arguments{
\item{data}{A numeric matrix or data frame with at least two numeric columns.
Non-numeric columns will be ignored.}

\item{ci}{Logical; if TRUE, return lower and upper confidence bounds}

\item{conf_level}{Confidence level for CI, default = 0.95}

\item{verbose}{Logical; if TRUE, prints how many threads are used}

\item{x}{An object of class \code{"ccc"} (either a matrix or a list with
confidence intervals).}

\item{digits}{Integer; number of decimal places to print in the concordance
matrix (default is 4).}

\item{...}{Additional arguments passed to underlying functions
(like \code{theme} or \code{print}).}

\item{title}{Title for the plot.}

\item{low_color}{Color for low CCC values.}

\item{high_color}{Color for high CCC values.}

\item{mid_color}{Color for mid CCC values (typically near 0).}

\item{value_text_size}{Text size for numbers in the heatmap.}
}
\value{
A symmetric numeric matrix with class \code{"ccc"} and attributes:
\itemize{
\item \code{method}: The method used ("Lin's concordance")
\item \code{description}: Description string
}
If \code{ci = FALSE}, returns matrix of class \code{"ccc"}.
If \code{ci = TRUE}, returns a list with elements: \code{est},
\code{lwr.ci}, \code{upr.ci}.
}
\description{
Computes all pairwise Lin's Concordance Correlation Coefficients (CCC)
from the numeric columns of a matrix or data frame. CCC measures both
precision (Pearson correlation) and accuracy (closeness to the 45-degree line).
This function is backed by a high-performance 'C++' implementation.
}
\details{
Lin's CCC is defined as:
\deqn{
\rho_c = \frac{2 \cdot \mathrm{cov}(X, Y)}{\sigma_X^2 + \sigma_Y^2 +
(\mu_X - \mu_Y)^2}
}{
rho_c = 2 * cov(X, Y) / [var(X) + var(Y) + (mean(X) - mean(Y))^2]
}

This formula combines the Pearson correlation coefficient
\eqn{r = \mathrm{cov}(X, Y) / (\sigma_X \sigma_Y)}
with a bias correction factor:
\deqn{
C_b = \frac{2 \sigma_X \sigma_Y}{\sigma_X^2 + \sigma_Y^2 + (\mu_X - \mu_Y)^2}
}

Confidence intervals are not provided in the matrix version to retain speed,
but can be computed separately for individual variable pairs using the
scalar form of Lin's CCC.

Missing values are not allowed; input must be clean numeric data.
}
\examples{
# Example with multivariate normal data
Sigma <- matrix(c(1, 0.5, 0.3,
                  0.5, 1, 0.4,
                  0.3, 0.4, 1), nrow = 3)
mu <- c(0, 0, 0)
set.seed(123)
mat_mvn <- MASS::mvrnorm(n = 100, mu = mu, Sigma = Sigma)
result_mvn <- ccc(mat_mvn)
print(result_mvn)
plot(result_mvn)

}
\references{
Lin L (1989). A concordance correlation coefficient to evaluate
reproducibility. Biometrics 45: 255-268.

Lin L (2000). A note on the concordance correlation coefficient.
Biometrics 56: 324-325.

Bland J, Altman D (1986). Statistical methods for assessing agreement
between two methods of clinical measurement. The Lancet 327: 307-310.
}
\seealso{
\code{\link{print.ccc}}, \code{\link{plot.ccc}}
}
\author{
Thiago de Paula Oliveira
}
