% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/krige.R
\name{krige.mcgf_rs}
\alias{krige.mcgf_rs}
\title{Obtain kriging forecasts for an \code{mcgf_rs} object.}
\usage{
\method{krige}{mcgf_rs}(
  x,
  newdata = NULL,
  newlabel = NULL,
  soft = FALSE,
  prob,
  model = c("all", "base", "empirical"),
  interval = FALSE,
  level = 0.95,
  ...
)
}
\arguments{
\item{x}{An \code{mcgf_rs} object.}

\item{newdata}{A data.frame with the same column names as \code{x}. If \code{newdata}
is missing the forecasts at the original data points are returned.}

\item{newlabel}{A vector of new regime labels.}

\item{soft}{Logical; if true, soft forecasts (and bounds) are produced.}

\item{prob}{Matrix with simplex rows. Number of columns must be the same as
unique labels in \code{x}.}

\item{model}{Which model to use. One of \code{all}, \code{base}, or \code{empirical}.}

\item{interval}{Logical; if TRUE, prediction intervals are computed.}

\item{level}{A numeric scalar between 0 and 1 giving the confidence level for
the intervals (if any) to be calculated. Used when \code{interval = TRUE}}

\item{...}{Additional arguments. Give \code{lag} and \code{horizon} if they are not
defined in \code{x} for the \code{empirical} model.}
}
\value{
A list of kriging forecasts (and prediction intervals).
}
\description{
Obtain kriging forecasts for an \code{mcgf_rs} object.
}
\details{
It produces simple kriging forecasts for a zero-mean mcgf. It supports
kriging for the \code{empirical} model, the \code{base} model, and the \code{all} model
which is the general stationary model with the base and Lagrangian model
from \code{x}.

When \code{soft = TRUE}, \code{prob} will be used to compute the soft forecasts
(weighted forecasts). The number of columns must match the number of unique
levels in \code{x}. The column order must be the same as the order of regimes as
in \code{levels(attr(x, "label", exact = TRUE))}. If not all regimes are seen in
\code{newlabel}, then only relevant columns in \code{prob} are used.

When \code{interval = TRUE}, confidence interval for each forecasts and each
horizon is given. Note that it does not compute confidence regions.
}
\examples{
data(sim2)
sim2_mcgf <- mcgf_rs(sim2$data, dists = sim2$dists, label = sim2$label)
sim2_mcgf <- add_acfs(sim2_mcgf, lag_max = 5)
sim2_mcgf <- add_ccfs(sim2_mcgf, lag_max = 5)

# Fit a regime-switching separable model
fit_sep <- fit_base(
    sim2_mcgf,
    lag_ls = 5,
    model_ls = "sep",
    par_init_ls = list(list(
        c = 0.00005,
        gamma = 0.5,
        a = 0.5,
        alpha = 0.5
    )),
    par_fixed_ls = list(c(nugget = 0))
)

# Store the fitted separable models to 'sim2_mcgf'
sim2_mcgf <- add_base(sim2_mcgf, fit_base_ls = fit_sep)

# Calculate the simple kriging predictions and intervals
sim2_krige <- krige(sim2_mcgf, model = "base", interval = TRUE)

# Calculate RMSE for each location
rmse <- sqrt(colMeans((sim2_mcgf - sim2_krige$fit)^2, na.rm = TRUE))
rmse

# Calculate MAE for each location
mae <- colMeans(abs(sim2_mcgf - sim2_krige$fit), na.rm = TRUE)
mae

# Calculate POPI for each location
popi <- colMeans(
    sim2_mcgf < sim2_krige$lower | sim2_mcgf > sim2_krige$upper,
    na.rm = TRUE
)
popi
}
\seealso{
Other functions on fitting an mcgf_rs: 
\code{\link{add_base.mcgf_rs}()},
\code{\link{add_lagr.mcgf_rs}()},
\code{\link{fit_base.mcgf_rs}()},
\code{\link{fit_lagr.mcgf_rs}()},
\code{\link{krige_new.mcgf_rs}()}
}
\concept{functions on fitting an mcgf_rs}
