% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sloss_week.R
\name{sloss_week}
\alias{sloss_week}
\title{Compute MCTQ weekly sleep loss}
\usage{
sloss_week(sd_w, sd_f, wd)
}
\arguments{
\item{sd_w}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration on workdays} from a standard or micro version of the MCTQ
questionnaire. You can use \code{\link[=sdu]{sdu()}} to compute it.}

\item{sd_f}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration on work-free days} from a standard or micro version of
the MCTQ questionnaire. You can use \code{\link[=sdu]{sdu()}} to compute it.}

\item{wd}{An \link[checkmate:checkIntegerish]{integerish}
\code{\link[base:numeric]{numeric}} object or an \code{\link[base:integer]{integer}}
object corresponding to the \strong{number of workdays per week} from a standard
or micro version of the MCTQ questionnaire.}
}
\value{
A \code{\link[lubridate:duration]{Duration}} object corresponding to the
weekly sleep loss.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{sloss_week()} computes the \strong{weekly sleep loss} for the standard and micro
versions of the Munich ChronoType Questionnaire (MCTQ).
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Roenneberg, Allebrandt, Merrow, & Vetter (2012) and The Worldwide
Experimental Platform (n.d.) guidelines for \code{sloss_week()}
(\eqn{SLoss_{week}}{SLoss_week}) computation are as follows.
\subsection{Notes}{
\itemize{
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{Computation}{

\deqn{\textrm{If } SD_{week} > SD_W \; , \; SLoss_{week} =
(SD_{week} - SD_W) \times WD}
\deqn{\textrm{Else } \; , \; SLoss_{week} = (SD_{week} - SD_F) \times FD}

Where:
\itemize{
\item \eqn{SLoss_{week}}: Weekly sleep loss.
\item \eqn{SD_W} = Sleep duration on workdays.
\item \eqn{SD_F} = Sleep duration on work-free days.
\item \eqn{SD_{week}} = Average weekly sleep duration.
\item \eqn{WD} = Number of workdays per week ("I have a regular work schedule and
work ___ days per week").
\item \eqn{FD} = Number of work-free days per week.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.
}
}

\examples{
## Scalar example

sd_w <- lubridate::dhours(6.5)
sd_f <- lubridate::dhours(7)
wd <- 4
sloss_week(sd_w, sd_f, wd)
#> [1] "3085.71428571429s (~51.43 minutes)" # Expected

sd_w <- lubridate::dhours(7)
sd_f <- lubridate::dhours(8)
wd <- 5
sloss_week(sd_w, sd_f, wd)
#> [1] "5142.85714285714s (~1.43 hours)" # Expected

sd_w <- lubridate::dhours(NA)
sd_f <- lubridate::dhours(9.45)
wd <- 7
sloss_week(sd_w, sd_f, wd)
#> [1] NA # Expected

## Vector example

sd_w <- c(lubridate::dhours(7), lubridate::dhours(8))
sd_f <- c(lubridate::dhours(6.5), lubridate::dhours(8))
wd <- c(2, 0)
sloss_week(sd_w, sd_f, wd)
#> [1] "2571.42857142857s (~42.86 minutes)" "0s" # Expected

## Converting the output to 'hms'

sd_w <- lubridate::dhours(4)
sd_f <- lubridate::dhours(5)
wd <- 3
sloss_week(sd_w, sd_f, wd)
#> [1] "6171.42857142858s (~1.71 hours)" # Expected

hms::as_hms(as.numeric(sloss_week(sd_w, sd_f, wd)))
#> 01:42:51.428571 # Expected

## Rounding the output at the seconds level

sd_w <- lubridate::dhours(5.8743)
sd_f <- lubridate::dhours(7.4324)
wd <- 6
sloss_week(sd_w, sd_f, wd)
#> [1] "4807.85142857144s (~1.34 hours)" # Expected

round_time(sloss_week(sd_w, sd_f, wd))
#> [1] "4808s (~1.34 hours)" # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}.

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}.

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}.

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}.

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
