% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sdu.R
\name{sdu}
\alias{sdu}
\title{Compute MCTQ sleep duration}
\usage{
sdu(so, se)
}
\arguments{
\item{so}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
sleep onset} from a standard, micro, or shift version of the MCTQ
questionnaire. You can use \code{\link[=so]{so()}} to compute it for the
standard or shift version.}

\item{se}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
sleep end} from a standard, micro, or shift version of the MCTQ
questionnaire.}
}
\value{
A \code{\link[lubridate:duration]{Duration}} object corresponding to the
vectorized difference between \code{se} and \code{so} in a circular time frame of 24
hours.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{sdu()} computes the \strong{sleep duration} for standard, micro, and shift
versions of the Munich ChronoType Questionnaire (MCTQ).

Please note that, although we tried to preserve the original authors' naming
pattern for the MCTQ functions, the name \code{sd} provokes a dangerous name
collision with the widely used \code{\link[stats:sd]{sd()}} function (standard
deviation). That's why we named it \code{sdu}. \code{sdu()} and \code{\link[=msl]{msl()}}
are the only exceptions, all the other \code{mctq} functions maintain a strong
naming resemblance with the original authors' naming pattern.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Roenneberg, Allebrandt, Merrow, & Vetter (2012), Ghotbi et al. (2020), Juda,
Vetter, & Roenneberg (2013), and The Worldwide Experimental Platform (n.d.)
guidelines for \code{sdu()} (\eqn{SD}) computation are as follows.
\subsection{Notes}{
\itemize{
\item This computation must be applied to each section of the questionnaire.
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{For standard and micro versions of the MCTQ}{

\deqn{SD_{W/F} = SE_{W/F} - SO_{W/F}}

Where:
\itemize{
\item \eqn{SD_{W/F}} = Sleep duration on work \strong{or} work-free days.
\item \eqn{SE_{W/F}} = Local time of sleep end on work \strong{or} work-free days.
\item \eqn{SO_{W/F}}  = Local time of sleep onset on work \strong{or} work-free days.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.
}

\subsection{For the shift version of the MCTQ}{

\deqn{SD_{W/F}^{M/E/N} = SE_{W/F}^{M/E/N} - SO_{W/F}^{M/E/N}}

Where:
\itemize{
\item \eqn{SD_{W/F}^{M/E/N}} = Sleep duration between two days in a particular
shift \strong{or} between two free days after a particular shift.
\item \eqn{SE_{W/F}^{M/E/N}} = Local time of sleep end between two days in a
particular shift \strong{or} between two free days after a particular shift.
\item \eqn{SO_{W/F}^{M/E/N}} = Local time of sleep onset between two days in a
particular shift \strong{or} between two free days after a particular shift.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.
}
}

\examples{
## Scalar example

so <- hms::parse_hm("23:00")
se <- hms::parse_hm("08:00")
sdu(so, se)
#> [1] "32400s (~9 hours)" # Expected

so <- hms::parse_hm("02:00")
se <- hms::parse_hm("12:30")
sdu(so, se)
#> [1] "37800s (~10.5 hours)" # Expected

so <- hms::parse_hm("03:15")
se <- hms::as_hms(NA)
sdu(so, se)
#> [1] NA # Expected

## Vector example

so <- c(hms::parse_hm("04:12"), hms::parse_hm("21:20"))
se <- c(hms::parse_hm("14:30"), hms::parse_hm("03:45"))
sdu(so, se)
#> [1] "37080s (~10.3 hours)" "23100s (~6.42 hours)" # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}.

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}.

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}.

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}.

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msf_sc}()},
\code{\link{msl}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_overall}()},
\code{\link{sd_week}()},
\code{\link{sjl_sc}()},
\code{\link{sjl_weighted}()},
\code{\link{sjl}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
