% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom_streamline.R
\docType{data}
\name{geom_streamline}
\alias{geom_streamline}
\alias{stat_streamline}
\alias{StatStreamline}
\alias{GeomStreamline}
\title{Streamlines}
\usage{
geom_streamline(
  mapping = NULL,
  data = NULL,
  stat = "streamline",
  position = "identity",
  ...,
  L = 5,
  min.L = 0,
  res = 1,
  S = NULL,
  dt = NULL,
  xwrap = NULL,
  ywrap = NULL,
  skip = 1,
  skip.x = skip,
  skip.y = skip,
  n = NULL,
  nx = n,
  ny = n,
  jitter = 1,
  jitter.x = jitter,
  jitter.y = jitter,
  arrow.angle = 6,
  arrow.length = 0.5,
  arrow.ends = "last",
  arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, unit(arrow.length, "lines"), ends = arrow.ends, type
    = arrow.type),
  lineend = "butt",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = TRUE
)

stat_streamline(
  mapping = NULL,
  data = NULL,
  geom = "streamline",
  position = "identity",
  ...,
  L = 5,
  min.L = 0,
  res = 1,
  S = NULL,
  dt = NULL,
  xwrap = NULL,
  ywrap = NULL,
  skip = 1,
  skip.x = skip,
  skip.y = skip,
  n = NULL,
  nx = n,
  ny = n,
  jitter = 1,
  jitter.x = jitter,
  jitter.y = jitter,
  arrow.angle = 6,
  arrow.length = 0.5,
  arrow.ends = "last",
  arrow.type = "closed",
  arrow = grid::arrow(arrow.angle, unit(arrow.length, "lines"), ends = arrow.ends, type
    = arrow.type),
  lineend = "butt",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = TRUE
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{aes()}} or
\code{\link[ggplot2:aes_]{aes_()}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[ggplot2:ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[ggplot2:fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame}, and
will be used as the layer data. A \code{function} can be created
from a \code{formula} (e.g. \code{~ head(.x, 10)}).}

\item{stat}{The statistical transformation to use on the data for this
layer, as a string.}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{colour = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{L, }{typical length of a streamline in x and y units}

\item{min.L}{minimum length of segments to show}

\item{res, }{resolution parameter (higher numbers increases the resolution)}

\item{S}{optional numeric number of timesteps for integration}

\item{dt}{optional numeric size "timestep" for integration}

\item{xwrap, ywrap}{vector of length two used to wrap the circular dimension.}

\item{skip}{numeric specifying number of gridpoints not to draw
in the x and y direction}

\item{skip.x}{numeric specifying number of gridpoints not to draw
in the x and y direction}

\item{skip.y}{numeric specifying number of gridpoints not to draw
in the x and y direction}

\item{n, nx, ny}{optional numeric indicating the number of points to draw in the
x and y direction (replaces \code{skip} if not \code{NULL})}

\item{jitter, jitter.x, jitter.y}{amount of jitter of the starting points}

\item{arrow.angle}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow.length}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow.ends}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow.type}{parameters passed to
\link[grid:arrow]{grid::arrow}}

\item{arrow}{specification for arrow heads, as created by arrow().}

\item{lineend}{Line end style (round, butt, square).}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[ggplot2:borders]{borders()}}.}

\item{geom}{The geometric object to use display the data}
}
\description{
Streamlines are paths that are always tangential to a vector field. In the
case of a steady field, it's identical to the path of a massless particle that
moves with the "flow".
}
\details{
Streamlines are computed by simple integration with a forward Euler method.
By default, \code{stat_streamline()} computes \code{dt} and \code{S} from \code{L}, \code{res},
the resolution of the grid and the mean magnitude of the field. \code{S} is
then defined as the number of steps necessary to make a streamline of length
\code{L} under an uniform mean field and \code{dt} is chosen so that each step is no
larger than the resolution of the data (divided by the \code{res} parameter). Be
aware that this rule of thumb might fail in field with very skewed distribution
of magnitudes.

Alternatively, \code{L} and/or \code{res} are ignored if \code{S} and/or \code{dt} are specified
explicitly. This not only makes it possible to fine-tune the result but also
divorces the integration parameters from the properties of the data and makes
it possible to compare streamlines between different fields.

The starting grid is a semi regular grid defined, either by the resolution of the
field and the \code{skip.x} and \code{skip.y} parameters o the \code{nx} and \code{ny} parameters,
jittered by an amount proportional to the resolution of the data and the
\code{jitter.x} and \code{jitter.y} parameters.

It might be important that the units of the vector field are compatible to the units
of the x and y dimensions. For example, passing \code{dx} and \code{dy} in m/s on a
longitude-latitude grid will might misleading results (see \link{spherical}).

Missing values are not permitted and the field must be defined on a
regular grid, for now.
}
\section{Aesthetics}{

\code{stat_streamline} understands the following aesthetics (required aesthetics are in bold)

\itemize{
\item \strong{x}
\item \strong{y}
\item \strong{dx}
\item \strong{dy}
\item \code{alpha}
\item \code{colour}
\item \code{linetype}
\item \code{size}
}
}

\section{Computed variables}{

\describe{
\item{step}{step in the simulation}
\item{dx}{dx at each location of the streamline}
\item{dy}{dy at each location of the streamline}
}
}

\examples{
\dontrun{
library(data.table)
library(ggplot2)
data(geopotential)

geopotential <- copy(geopotential)[date == date[1]]
geopotential[, gh.z := Anomaly(gh), by = .(lat)]
geopotential[, c("u", "v") := GeostrophicWind(gh.z, lon, lat)]

(g <- ggplot(geopotential, aes(lon, lat)) +
    geom_contour2(aes(z = gh.z), xwrap = c(0, 360)) +
    geom_streamline(aes(dx = dlon(u, lat), dy = dlat(v)), L = 60,
                    xwrap = c(0, 360)))

# The circular parameter is particularly important for polar coordinates
g + coord_polar()

# If u and v are not converted into degrees/second, the resulting
# streamlines have problems, specially near the pole.
ggplot(geopotential, aes(lon, lat)) +
    geom_contour(aes(z = gh.z)) +
    geom_streamline(aes(dx = u, dy = v), L = 50)

# The step variable can be mapped to size or alpha to
# get cute "drops". It's important to note that ..dx.. (the calculated variable)
# is NOT the same as dx (from the data).
ggplot(geopotential, aes(lon, lat)) +
    geom_streamline(aes(dx = dlon(u, lat), dy = dlat(v), alpha = ..step..,
                        color = sqrt(..dx..^2 + ..dy..^2), size = ..step..),
                        L = 40, xwrap = c(0, 360), res = 2, arrow = NULL,
                        lineend = "round") +
    scale_size(range = c(0, 0.6))

# Using topographic information to simulate "rivers" from slope
topo <- GetTopography(295, -55+360, -30, -42, res = 1/20)  # needs internet!
topo[, c("dx", "dy") := Derivate(h ~ lon + lat)]
topo[h <= 0, c("dx", "dy") := 0]

# See how in this example the integration step is too coarse in the
# western montanous region where the slope is much higher than in the
# flatlands of La Pampa at in the east.
ggplot(topo, aes(lon, lat)) +
    geom_relief(aes(z = h), interpolate = TRUE, data = topo[h >= 0]) +
    geom_contour(aes(z = h), breaks = 0, color = "black") +
    geom_streamline(aes(dx = -dx, dy = -dy), L = 10, skip = 3, arrow = NULL,
                    color = "#4658BD") +
    coord_quickmap()
 }

}
\seealso{
Other ggplot2 helpers: 
\code{\link{DivideTimeseries}()},
\code{\link{MakeBreaks}()},
\code{\link{WrapCircular}()},
\code{\link{geom_arrow}()},
\code{\link{geom_contour2}()},
\code{\link{geom_contour_fill}()},
\code{\link{geom_label_contour}()},
\code{\link{geom_relief}()},
\code{\link{guide_colourstrip}()},
\code{\link{map_labels}},
\code{\link{reverselog_trans}()},
\code{\link{scale_divergent}},
\code{\link{scale_longitude}},
\code{\link{stat_na}()},
\code{\link{stat_subset}()}
}
\concept{ggplot2 helpers}
\keyword{datasets}
