% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/miclust.R
\name{miclust}
\alias{miclust}
\title{Cluster analysis in multiple imputed data sets with optional variable
  selection.}
\usage{
miclust(
  data,
  method = "kmeans",
  search = c("none", "backward", "forward"),
  ks = 2:3,
  maxvars = NULL,
  usedimp = NULL,
  distance = c("manhattan", "euclidean"),
  centpos = c("means", "medians"),
  initcl = c("hc", "rand"),
  verbose = TRUE,
  seed = NULL
)
}
\arguments{
\item{data}{object of class \code{midata} obtained with the function \code{\link{getdata}}.}

\item{method}{clustering method. Currently, only \code{"kmeans"} is accepted.}

\item{search}{search algorithm for the selection variable procedure: \code{"backward"},
\code{"forward"} or \code{"none"}. If \code{"none"} (default), no variable selection
is performed.}

\item{ks}{the values of the explored number of clusters. Default is exploring 2 and 3
clusters.}

\item{maxvars}{if \code{method = "forward"}, the maximum number of variables to be
selected.}

\item{usedimp}{numeric. Which imputed data sets must be included in the cluster
analysis. If \code{NULL} (default), all available imputed data sets are included.
If \code{usedimp} is numeric (or a numeric vector), its values indicate which imputed
data sets are included.}

\item{distance}{two metrics are allowed to compute distances: \code{"manhattan"}
(default) and \code{"euclidean"}.}

\item{centpos}{position computation of the cluster centroid. If \code{"means"}
(default) the position of the centroid is computed by the mean. If \code{"medians"},
by the median.}

\item{initcl}{starting values for the clustering algorithm. If \code{"rand"}, they are
randomly selected; if \code{"hc"}, they are computed via hierarchical clustering. See
Details below.}

\item{verbose}{a logical value indicating output status messages. Default is \code{TRUE}.}

\item{seed}{a number. Seed for reproducibility of results. Default is \code{NULL} (no
seed).}
}
\value{
A list with class "miclust" including the following items:
 \describe{
   \item{clustering}{a list of lists containing the results of the clustering algorithm
    for each analyzed data set and for each analyzed number of clusters. Includes
    information about selected variables and the cluster vector.}
   \item{completecasesperc}{if \code{data} contains a single data frame, percentage
    of complete cases in \code{data}.}
   \item{data}{input \code{data}.}
   \item{ks}{the values of the explored number of clusters.}
   \item{usedimp}{indicator of the imputed data sets used.}
   \item{kfin}{optimal number of clusters.}
   \item{critcf}{if \code{data} contains a single data frame, \code{critcf} contains
    the optimal (maximum) value of CritCF (see Details) and the number of selected
    variables in the reduction procedure for each explored number of clusters. If
    \code{data} is a list, \code{critcf} contains the optimal value of CritCF for
    each imputed data set and for each explored value of the number of clusters.}
   \item{numberofselectedvars}{number of selected variables.}
   \item{selectedkdistribution}{if \code{data} is a list, frequency of selection of
    each analyzed number of clusters.}
   \item{method}{input \code{method}.}
   \item{search}{input \code{search}.}
   \item{maxvars}{input \code{maxvars}.}
   \item{distance}{input \code{distance}.}
   \item{centpos}{input \code{centpos}.}
   \item{selmetriccent}{an object of class \code{kccaFamily} needed by the specific
    \code{summary} method.}
   \item{initcl}{input \code{initcl}.}
  }
}
\description{
Performs cluster analysis in multiple imputed data sets with optional variable
  selection. Results can be summarized and visualized with the \code{summary}
  and \code{plot} methods.
}
\details{
The optimal number of clusters and the final set of variables are selected
 according to CritCF. CritCF is defined as
 \deqn{CritCF = \left(\frac{2m}{2m + 1} \cdot \frac{1}{1 + W / B}\right)^{\frac{1 + \log_2(k + 1)}{1 + \log_2(m + 1)}},}
 where \eqn{m} is the number of variables, \eqn{k} is the number of clusters,
 and \eqn{W} and \eqn{B} are the within- and between-cluster inertias. Higher
 values of CritCF are preferred (Breaban, 2011). See References below for further
 details about the clustering algorithm.

 For computational reasons, option \code{"rand"} is suggested instead of \code{"hc"}
 for high dimensional \code{data}.
}
\examples{
### data preparation:
minhanes1 <- getdata(data = minhanes)

##################
###
### Example 1:
###
### Multiple imputation clustering process with backward variable selection
###
##################

### using only the imputations 1 to 10 for the clustering process and exploring
### 2 vs. 3 clusters:
minhanes1clust <- miclust(data = minhanes1, search = "backward", ks = 2:3,
                          usedimp = 1:10, seed = 4321)
minhanes1clust
minhanes1clust$kfin  ### optimal number of clusters

### graphical summary:
plot(minhanes1clust)

### selection frequency of the variables for the optimal number of clusters:
y <- getvariablesfrequency(minhanes1clust)
y
plot(y$percfreq, type = "h", main = "", xlab = "Variable",
     ylab = "Percentage of times selected", xlim = 0.5 + c(0, length(y$varnames)),
     lwd = 15, col = "blue", xaxt = "n")
axis(1, at = 1:length(y$varnames), labels = y$varnames)

### default summary for the optimal number of clusters:
summary(minhanes1clust)

## summary forcing 3 clusters:
summary(minhanes1clust, k = 3)

##################
###
### Example 2:
###
### Same analysis but without variable selection
###
##################
\donttest{
minhanes2clust <- miclust(data = minhanes1, ks = 2:3, usedimp = 1:10, seed = 4321)
minhanes2clust
plot(minhanes2clust)
summary(minhanes2clust)
}

##################
###
### Example 3:
###
### Complete case clustering process with backward variable selection
###
##################

nhanes0 <- getdata(data = minhanes[[1]])
nhanes2clust <- miclust(data = nhanes0, search = "backward", ks = 2:3, seed = 4321)
nhanes2clust

summary(nhanes2clust)

### nothing to plot for a single data set analysis
# plot(nhanes2clust)

##################
###
### Example 4:
###
### Complete case clustering process without variable selection
###
##################
\donttest{
nhanes3clust <- miclust(data = nhanes0, ks = 2:3, seed = 4321)
nhanes3clust
summary(nhanes3clust)
}
}
\references{
\itemize{
  \item Basagana X, Barrera-Gomez J, Benet M, Anto JM, Garcia-Aymerich J. A
   framework for multiple imputation in cluster analysis. American Journal of
   Epidemiology. 2013;177(7):718-25.
  \item Breaban M, Luchian H. A unifying criterion for unsupervised clustering
   and feature selection. Pattern Recognition 2001;44(4):854-65.
  }
}
\seealso{
\code{\link{getdata}} for data preparation before using \code{miclust}.
}
