% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/personfit.R
\name{personfit}
\alias{personfit}
\title{Person fit statistics}
\usage{
personfit(x, method = "EAP", Theta = NULL, stats.only = TRUE, ...)
}
\arguments{
\item{x}{a computed model object of class \code{SingleGroupClass} or \code{MultipleGroupClass}}

\item{method}{type of factor score estimation method. See \code{\link{fscores}} for more detail}

\item{Theta}{a matrix of factor scores used for statistics that require empirical estimates. If
supplied, arguments typically passed to \code{fscores()} will be ignored and these values will
be used instead}

\item{stats.only}{logical; return only the person fit statistics without their associated
response pattern?}

\item{...}{additional arguments to be passed to \code{fscores()}}
}
\description{
\code{personfit} calculates the Zh values from Drasgow, Levine and Williams (1985) for
unidimensional and multidimensional models, as well as the infit and outfit statistics.
The returned object is a \code{data.frame}
consisting either of the tabulated data or full data with the statistics appended to the
rightmost columns.
}
\examples{

\dontrun{

#make some data
set.seed(1)
a <- matrix(rlnorm(20),ncol=1)
d <- matrix(rnorm(20),ncol=1)
items <- rep('2PL', 20)
data <- simdata(a,d, 2000, items)

# first observation responds 1 for most difficult, 0 for easiest
data[1,] <- ifelse(d > 0, 0, 1)

# second observations answers first half as 1 second half as 0
data[2,] <- rep(1:0, each = 10)

x <- mirt(data, 1)
fit <- personfit(x)
head(fit)

#using precomputed Theta
Theta <- fscores(x, method = 'MAP', full.scores = TRUE)
head(personfit(x, Theta=Theta))

# multiple group Rasch model example
set.seed(12345)
a <- matrix(rep(1, 16), ncol=1)
d <- matrix(rnorm(16,0,.7),ncol=1)
itemtype <- rep('dich', nrow(a))
N <- 1000
dataset1 <- simdata(a, d, N, itemtype)
dataset2 <- simdata(a, d, N, itemtype, sigma = matrix(1.5))
dat <- rbind(dataset1, dataset2)

# first observation responds 1 for most difficult, 0 for easiest
dat[1,] <- ifelse(d > 0, 0, 1)

group <- c(rep('D1', N), rep('D2', N))
models <- 'F1 = 1-16'
mod_Rasch <- multipleGroup(dat, models, itemtype = 'Rasch', group = group)
coef(mod_Rasch, simplify=TRUE)
pf <- personfit(mod_Rasch, method='MAP')
head(pf)

  }

}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory
Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.
\doi{10.18637/jss.v048.i06}

Drasgow, F., Levine, M. V., & Williams, E. A. (1985). Appropriateness measurement with
polychotomous item response models and standardized indices.
\emph{British Journal of Mathematical and Statistical Psychology, 38}, 67-86.

Reise, S. P. (1990). A comparison of item- and person-fit methods of assessing model-data fit
in IRT. \emph{Applied Psychological Measurement, 14}, 127-137.

Wright B. D. & Masters, G. N. (1982). \emph{Rating scale analysis}. MESA Press.
}
\seealso{
\code{\link{itemfit}}
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
\keyword{fit}
\keyword{person}
