% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/missRanger.R
\name{missRanger}
\alias{missRanger}
\title{Fast Imputation of Missing Values by Chained Random Forests}
\usage{
missRanger(
  data,
  formula = . ~ .,
  pmm.k = 0L,
  maxiter = 10L,
  seed = NULL,
  verbose = 1,
  returnOOB = FALSE,
  case.weights = NULL,
  data_only = TRUE,
  keep_forests = FALSE,
  ...
)
}
\arguments{
\item{data}{A \code{data.frame} with missing values to impute.}

\item{formula}{A two-sided formula specifying variables to be imputed
(left hand side) and variables used to impute (right hand side).
Defaults to \code{. ~ .}, i.e., use all variables to impute all variables.
For instance, if all variables (with missings) should be imputed by all variables
except variable "ID", use \code{. ~ . - ID}. Note that a "." is evaluated
separately for each side of the formula. Further note that variables with missings
must appear in the left hand side if they should be used on the right hand side.}

\item{pmm.k}{Number of candidate non-missing values to sample from in the
predictive mean matching steps. 0 to avoid this step.}

\item{maxiter}{Maximum number of chaining iterations.}

\item{seed}{Integer seed to initialize the random generator.}

\item{verbose}{Controls how much info is printed to screen.
0 to print nothing. 1 (default) to print a progress bar per iteration,
2 to print the OOB prediction error per iteration and variable
(1 minus R-squared for regression).
Furthermore, if \code{verbose} is positive, the variables used for imputation are
listed as well as the variables to be imputed (in the imputation order).
This will be useful to detect if some variables are unexpectedly skipped.}

\item{returnOOB}{Logical flag. If TRUE, the final average out-of-bag prediction
errors per variable is added to the resulting data as attribute "oob".
Only relevant when \code{data_only = TRUE} (and when forests are grown).}

\item{case.weights}{Vector with non-negative case weights.}

\item{data_only}{If \code{TRUE} (default), only the imputed data is returned.
Otherwise, a "missRanger" object with additional information is returned.}

\item{keep_forests}{Should the random forests of the final imputations
be returned? The default is \code{FALSE}. Setting this option will use a lot of memory.
Only relevant when \code{data_only = TRUE} (and when forests are grown).}

\item{...}{Arguments passed to \code{\link[ranger:ranger]{ranger::ranger()}}. If the data set is large,
better use less trees (e.g. \code{num.trees = 20}) and/or a low value of
\code{sample.fraction}. The following arguments are incompatible, amongst others:
\code{write.forest}, \code{probability}, \code{split.select.weights},
\code{dependent.variable.name}, and \code{classification}.}
}
\value{
If \code{data_only} an imputed \code{data.frame}. Otherwise, a "missRanger" object with
the following elements that can be extracted via \code{$}:
\itemize{
\item \code{data}: The imputed data.
\item \code{forests}: When \code{keep_forests = TRUE}, a list of "ranger" models used to
generate the imputed data. \code{NULL} otherwise.
\item \code{visit_seq}: Variables to be imputed (in this order).
\item \code{impute_by}: Variables used for imputation.
\item \code{best_iter}: Best iteration.
\item \code{pred_errors}: Per-iteration OOB prediction errors (1 - R^2 for regression,
classification error otherwise).
\item \code{mean_pred_errors}: Per-iteration averages of OOB prediction errors.
}
}
\description{
Uses the "ranger" package (Wright & Ziegler) to do fast missing value imputation by
chained random forests, see Stekhoven & Buehlmann and Van Buuren & Groothuis-Oudshoorn.
Between the iterative model fitting, it offers the option of predictive mean matching.
This firstly avoids imputation with values not present in the original data
(like a value 0.3334 in a 0-1 coded variable).
Secondly, predictive mean matching tries to raise the variance in the resulting
conditional distributions to a realistic level. This allows to do multiple imputation
when repeating the call to \code{\link[=missRanger]{missRanger()}}.
}
\details{
The iterative chaining stops as soon as \code{maxiter} is reached or if the average
out-of-bag (OOB) prediction errors stop reducing.
In the latter case, except for the first iteration, the second last (= best)
imputed data is returned.

OOB prediction errors are quantified as 1 - R^2 for numeric variables, and as
classification error otherwise. If a variable has been imputed only univariately,
the value is 1.

A note on \code{mtry}: Be careful when passing a non-default \code{mtry} to
\code{\link[ranger:ranger]{ranger::ranger()}} because the number of available covariates might be growing during
the first iteration, depending on the missing pattern.
Values \code{NULL} (default) and 1 are safe choices.
Additionally, recent versions of \code{\link[ranger:ranger]{ranger::ranger()}} allow \code{mtry} to be a
single-argument function of the number of available covariables,
e.g., \code{mtry = function(m) max(1, m \%/\% 3)}.
}
\examples{
irisWithNA <- generateNA(iris, seed = 34)
irisImputed <- missRanger(irisWithNA, pmm.k = 3, num.trees = 100)
head(irisImputed)
head(irisWithNA)

\dontrun{
# Extended output
imp <- missRanger(irisWithNA, pmm.k = 3, num.trees = 100, data_only = FALSE)
head(imp$data)
imp$pred_errors

# If you even want to keep the random forests of the best iteration
imp <- missRanger(
  irisWithNA, pmm.k = 3, num.trees = 100, data_only = FALSE, keep_forests = TRUE
)
imp$forests$Sepal.Width
imp$pred_errors[imp$best_iter, "Sepal.Width"]  # 1 - R-squared
}
}
\references{
\enumerate{
\item Wright, M. N. & Ziegler, A. (2016). ranger: A Fast Implementation of
Random Forests for High Dimensional Data in C++ and R. Journal of Statistical
Software, in press. <arxiv.org/abs/1508.04409>.
\item Stekhoven, D.J. and Buehlmann, P. (2012). 'MissForest - nonparametric missing
value imputation for mixed-type data', Bioinformatics, 28(1) 2012, 112-118.
https://doi.org/10.1093/bioinformatics/btr597.
\item Van Buuren, S., Groothuis-Oudshoorn, K. (2011). mice: Multivariate Imputation
by Chained Equations in R. Journal of Statistical Software, 45(3), 1-67.
http://www.jstatsoft.org/v45/i03/
}
}
