% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AutoFSelector.R
\name{AutoFSelector}
\alias{AutoFSelector}
\title{Class for Automatic Feature Selection}
\description{
The \link{AutoFSelector} wraps a \link[mlr3:Learner]{mlr3::Learner} and augments it with an automatic feature selection.
The \code{\link[=auto_fselector]{auto_fselector()}} function creates an \link{AutoFSelector} object.
}
\details{
The \link{AutoFSelector} is a \link[mlr3:Learner]{mlr3::Learner} which wraps another \link[mlr3:Learner]{mlr3::Learner} and performs the following steps during \verb{$train()}:
\enumerate{
\item The wrapped (inner) learner is trained on the feature subsets via resampling.
The feature selection can be specified by providing a \link{FSelector}, a \link[bbotk:Terminator]{bbotk::Terminator}, a \link[mlr3:Resampling]{mlr3::Resampling} and a \link[mlr3:Measure]{mlr3::Measure}.
\item A final model is fit on the complete training data with the best-found feature subset.
}

During \verb{$predict()} the \link{AutoFSelector} just calls the predict method of the wrapped (inner) learner.
}
\section{Resources}{

There are several sections about feature selection in the \href{https://mlr3book.mlr-org.com}{mlr3book}.
\itemize{
\item Estimate Model Performance with \href{https://mlr3book.mlr-org.com/chapters/chapter6/feature_selection.html#sec-autofselect}{nested resampling}.
}

The \href{https://mlr-org.com/gallery.html}{gallery} features a collection of case studies and demos about optimization.
}

\section{Nested Resampling}{

Nested resampling can be performed by passing an \link{AutoFSelector} object to \code{\link[mlr3:resample]{mlr3::resample()}} or \code{\link[mlr3:benchmark]{mlr3::benchmark()}}.
To access the inner resampling results, set \code{store_fselect_instance = TRUE} and execute \code{\link[mlr3:resample]{mlr3::resample()}} or \code{\link[mlr3:benchmark]{mlr3::benchmark()}} with \code{store_models = TRUE} (see examples).
The \link[mlr3:Resampling]{mlr3::Resampling} passed to the \link{AutoFSelector} is meant to be the inner resampling, operating on the training set of an arbitrary outer resampling.
For this reason it is not feasible to pass an instantiated \link[mlr3:Resampling]{mlr3::Resampling} here.
}

\examples{
# Automatic Feature Selection
\donttest{

# split to train and external set
task = tsk("penguins")
split = partition(task, ratio = 0.8)

# create auto fselector
afs = auto_fselector(
  fselector = fs("random_search"),
  learner = lrn("classif.rpart"),
  resampling = rsmp ("holdout"),
  measure = msr("classif.ce"),
  term_evals = 4)

# optimize feature subset and fit final model
afs$train(task, row_ids = split$train)

# predict with final model
afs$predict(task, row_ids = split$test)

# show result
afs$fselect_result

# model slot contains trained learner and fselect instance
afs$model

# shortcut trained learner
afs$learner

# shortcut fselect instance
afs$fselect_instance


# Nested Resampling

afs = auto_fselector(
  fselector = fs("random_search"),
  learner = lrn("classif.rpart"),
  resampling = rsmp ("holdout"),
  measure = msr("classif.ce"),
  term_evals = 4)

resampling_outer = rsmp("cv", folds = 3)
rr = resample(task, afs, resampling_outer, store_models = TRUE)

# retrieve inner feature selection results.
extract_inner_fselect_results(rr)

# performance scores estimated on the outer resampling
rr$score()

# unbiased performance of the final model trained on the full data set
rr$aggregate()
}
}
\section{Super class}{
\code{\link[mlr3:Learner]{mlr3::Learner}} -> \code{AutoFSelector}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{instance_args}}{(\code{list()})\cr
All arguments from construction to create the \link{FSelectInstanceSingleCrit}.}

\item{\code{fselector}}{(\link{FSelector})\cr
Optimization algorithm.}
}
\if{html}{\out{</div>}}
}
\section{Active bindings}{
\if{html}{\out{<div class="r6-active-bindings">}}
\describe{
\item{\code{archive}}{([ArchiveFSelect)\cr
Returns \link{FSelectInstanceSingleCrit} archive.}

\item{\code{learner}}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Trained learner.}

\item{\code{fselect_instance}}{(\link{FSelectInstanceSingleCrit})\cr
Internally created feature selection instance with all intermediate results.}

\item{\code{fselect_result}}{(\link[data.table:data.table]{data.table::data.table})\cr
Short-cut to \verb{$result} from \link{FSelectInstanceSingleCrit}.}

\item{\code{predict_type}}{(\code{character(1)})\cr
Stores the currently active predict type, e.g. \code{"response"}.
Must be an element of \verb{$predict_types}.}

\item{\code{hash}}{(\code{character(1)})\cr
Hash (unique identifier) for this object.}

\item{\code{phash}}{(\code{character(1)})\cr
Hash (unique identifier) for this partial object, excluding some components which are varied systematically during tuning (parameter values) or feature selection (feature names).}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-AutoFSelector-new}{\code{AutoFSelector$new()}}
\item \href{#method-AutoFSelector-base_learner}{\code{AutoFSelector$base_learner()}}
\item \href{#method-AutoFSelector-importance}{\code{AutoFSelector$importance()}}
\item \href{#method-AutoFSelector-selected_features}{\code{AutoFSelector$selected_features()}}
\item \href{#method-AutoFSelector-oob_error}{\code{AutoFSelector$oob_error()}}
\item \href{#method-AutoFSelector-loglik}{\code{AutoFSelector$loglik()}}
\item \href{#method-AutoFSelector-print}{\code{AutoFSelector$print()}}
\item \href{#method-AutoFSelector-clone}{\code{AutoFSelector$clone()}}
}
}
\if{html}{\out{
<details><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="format"><a href='../../mlr3/html/Learner.html#method-Learner-format'><code>mlr3::Learner$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="help"><a href='../../mlr3/html/Learner.html#method-Learner-help'><code>mlr3::Learner$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="predict"><a href='../../mlr3/html/Learner.html#method-Learner-predict'><code>mlr3::Learner$predict()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="predict_newdata"><a href='../../mlr3/html/Learner.html#method-Learner-predict_newdata'><code>mlr3::Learner$predict_newdata()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="reset"><a href='../../mlr3/html/Learner.html#method-Learner-reset'><code>mlr3::Learner$reset()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3" data-topic="Learner" data-id="train"><a href='../../mlr3/html/Learner.html#method-Learner-train'><code>mlr3::Learner$train()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-new"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$new(
  fselector,
  learner,
  resampling,
  measure = NULL,
  terminator,
  store_fselect_instance = TRUE,
  store_benchmark_result = TRUE,
  store_models = FALSE,
  check_values = FALSE,
  callbacks = list(),
  ties_method = "least_features"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{fselector}}{(\link{FSelector})\cr
Optimization algorithm.}

\item{\code{learner}}{(\link[mlr3:Learner]{mlr3::Learner})\cr
Learner to optimize the feature subset for.}

\item{\code{resampling}}{(\link[mlr3:Resampling]{mlr3::Resampling})\cr
Resampling that is used to evaluated the performance of the feature subsets.
Uninstantiated resamplings are instantiated during construction so that all feature subsets are evaluated on the same data splits.
Already instantiated resamplings are kept unchanged.}

\item{\code{measure}}{(\link[mlr3:Measure]{mlr3::Measure})\cr
Measure to optimize. If \code{NULL}, default measure is used.}

\item{\code{terminator}}{(\link{Terminator})\cr
Stop criterion of the feature selection.}

\item{\code{store_fselect_instance}}{(\code{logical(1)})\cr
If \code{TRUE} (default), stores the internally created \link{FSelectInstanceSingleCrit} with all intermediate results in slot \verb{$fselect_instance}.
Is set to \code{TRUE}, if \code{store_models = TRUE}}

\item{\code{store_benchmark_result}}{(\code{logical(1)})\cr
Store benchmark result in archive?}

\item{\code{store_models}}{(\code{logical(1)}).
Store models in benchmark result?}

\item{\code{check_values}}{(\code{logical(1)})\cr
Check the parameters before the evaluation and the results for
validity?}

\item{\code{callbacks}}{(list of \link{CallbackFSelect})\cr
List of callbacks.}

\item{\code{ties_method}}{(\code{character(1)})\cr
The method to break ties when selecting sets while optimizing and when selecting the best set.
Can be \code{"least_features"} or \code{"random"}.
The option \code{"least_features"} (default) selects the feature set with the least features.
If there are multiple best feature sets with the same number of features, one is selected randomly.
The \code{random} method returns a random feature set from the best feature sets.
Ignored if multiple measures are used.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-base_learner"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-base_learner}{}}}
\subsection{Method \code{base_learner()}}{
Extracts the base learner from nested learner objects like \code{GraphLearner} in \CRANpkg{mlr3pipelines}.
If \code{recursive = 0}, the (tuned) learner is returned.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$base_learner(recursive = Inf)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{recursive}}{(\code{integer(1)})\cr
Depth of recursion for multiple nested objects.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
\link{Learner}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-importance"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-importance}{}}}
\subsection{Method \code{importance()}}{
The importance scores of the final model.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$importance()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
Named \code{numeric()}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-selected_features"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-selected_features}{}}}
\subsection{Method \code{selected_features()}}{
The selected features of the final model.
These features are selected internally by the learner.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$selected_features()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{character()}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-oob_error"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-oob_error}{}}}
\subsection{Method \code{oob_error()}}{
The out-of-bag error of the final model.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$oob_error()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{numeric(1)}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-loglik"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-loglik}{}}}
\subsection{Method \code{loglik()}}{
The log-likelihood of the final model.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$loglik()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{logLik}.
Printer.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-print"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-print}{}}}
\subsection{Method \code{print()}}{
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$print()}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{(ignored).}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-AutoFSelector-clone"></a>}}
\if{latex}{\out{\hypertarget{method-AutoFSelector-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{AutoFSelector$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
