% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/TuningInstance.R
\docType{data}
\name{TuningInstance}
\alias{TuningInstance}
\title{TuningInstance Class}
\format{\link[R6:R6Class]{R6::R6Class} object.}
\description{
Specifies a general tuning scenario, including performance evaluator and archive for Tuners to
act upon. This class encodes the black box objective function,
that a \link{Tuner} has to optimize. It allows the basic operations of querying the objective
at design points (\code{$eval_batch()}), storing the evaluations in an internal archive
and querying the archive (\code{$archive()}).

Evaluations of hyperparameter configurations are performed in batches by calling \code{\link[mlr3:benchmark]{mlr3::benchmark()}} internally.
Before and after a batch is evaluated, the \link{Terminator} is queried for the remaining budget.
If the available budget is exhausted, an exception is raised, and no further evaluations can be performed from this point on.

A list of measures can be passed to the instance, and they will always be all evaluated.
However, single-criteria tuners optimize only the first measure.

The tuner is also supposed to store its final result, consisting of a selected hyperparameter configuration
and associated estimated performance values, by calling the method \code{instance$assign_result}.
}
\section{Construction}{
\preformatted{inst = TuningInstance$new(task, learner, resampling, measures,
  param_set, terminator, bm_args = list())
}

This defines the resampled performance of a learner on a task, a feasibility region
for the parameters the tuner is supposed to optimize, and a termination criterion.
\itemize{
\item \code{task} :: \link[mlr3:Task]{mlr3::Task}.
\item \code{learner} :: \link[mlr3:Learner]{mlr3::Learner}.
\item \code{resampling} :: \link[mlr3:Resampling]{mlr3::Resampling}\cr
Note that the resampling is instantiated at the beginning so that all configurations
are evaluated on the same data splits.
\item \code{measures} :: list of \link[mlr3:Measure]{mlr3::Measure}.
\item \code{param_set} :: \link[paradox:ParamSet]{paradox::ParamSet}.
\item \code{terminator} :: \link{Terminator}.
\item \code{bm_args} :: named \code{list()}\cr
Further arguments for \code{\link[mlr3:benchmark]{mlr3::benchmark()}}.
}
}

\section{Fields}{

\itemize{
\item \code{task} :: \link[mlr3:Task]{mlr3::Task}; from construction.
\item \code{learner} :: \link[mlr3:Learner]{mlr3::Learner}; from construction.
\item \code{resampling} :: \link[mlr3:Resampling]{mlr3::Resampling}; from construction.
\item \code{measures} :: list of \link[mlr3:Measure]{mlr3::Measure}; from construction.
\item \code{param_set} :: \link[paradox:ParamSet]{paradox::ParamSet}; from construction.
\item \code{terminator} :: \link{Terminator}; from construction.
\item \code{bmr} :: \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult}\cr
A benchmark result, container object for all performed \link[mlr3:ResampleResult]{mlr3::ResampleResult}s when evaluating hyperparameter configurations.
\item \code{n_evals} :: \code{integer(1)}\cr
Number of configuration evaluations stored in the container.
\item \code{start_time} :: \code{POSIXct(1)}\cr
Time the tuning was started.
This is set in the beginning of \code{$tune()} of \link{Tuner}.
\item \code{result} :: named \code{list()}\cr
Result of the tuning, i.e., the optimal configuration and its estimated performance:
\itemize{
\item \code{"perf"}: Named vector of estimated performance values of the best configuration found.
\item \code{"tune_x"}: Named list of optimal hyperparameter settings, without potential \code{trafo} function applied.
\item \code{"params"}: Named list of optimal hyperparameter settings, similar to \code{tune_x}, but with potential \code{trafo} function applied.
Also, if the learner had some extra parameters statically set before tuning, these are included here.
}
}
}

\section{Methods}{

\itemize{
\item \code{eval_batch(dt)}\cr
\code{\link[data.table:data.table]{data.table::data.table()}} -> named \code{list()}\cr
Evaluates all hyperparameter configurations in \code{dt} through resampling, where each configuration is a row, and columns are scalar parameters.
Updates the internal \link{BenchmarkResult} \code{$bmr} by reference, and returns a named list with the following elements:
\itemize{
\item \code{"batch_nr"}: Number of the new batch.
This number is calculated in an auto-increment fashion and also stored inside the \link{BenchmarkResult} as column \code{batch_nr}
\item \code{"uhashes"}: unique hashes of the added \link{ResampleResult}s.
\item \code{"perf"}: A \code{\link[data.table:data.table]{data.table::data.table()}} of evaluated performances for each row of the \code{dt}.
Has the same number of rows as \code{dt}, and the same number of columns as length of \code{measures}.
Columns are named with measure-IDs. A cell entry is the (aggregated) performance of that configuration for that measure.
}

Before and after each batch-evaluation, the \link{Terminator} is checked, and if it is positive, an exception of class \code{terminated_error} is raised.
This function should be internally called by the tuner.
\item \code{tuner_objective(x)}\cr
\code{numeric()} -> \code{numeric(1)}\cr
Evaluates a hyperparameter configuration (untransformed) of only numeric values, and returns a scalar objective value,
where the return value is negated if the measure is maximized.
Internally, \code{$eval_batch()} is called with a single row.
This function serves as a objective function for tuners of numeric spaces - which should always be minimized.
\item \code{best(measure = NULL)}\cr
(\link[mlr3:Measure]{mlr3::Measure}, \code{character(1)}) -> \link[mlr3:ResampleResult]{mlr3::ResampleResult}\cr
Queries the \link[mlr3:BenchmarkResult]{mlr3::BenchmarkResult} for the best \link[mlr3:ResampleResult]{mlr3::ResampleResult} according to \code{measure} (default is the first measure in \code{$measures}).
In case of ties, one of the tied values is selected randomly.
\item \code{archive(unnest = "no")}\cr
\code{character(1)} -> \code{\link[data.table:data.table]{data.table::data.table()}}\cr
Returns a table of contained resample results, similar to the one returned by \code{\link[mlr3:benchmark]{mlr3::benchmark()}}'s \code{$aggregate()} method.
Some interesting columns of this table are:
\itemize{
\item All evaluated measures are included as numeric columns, named with their measure ID.
\item \code{tune_x}: A list column that contains the parameter settings the tuner evaluated, without potential \code{trafo} applied.
\item \code{params}: A list column that contains the parameter settings that were actually used in the learner.
Similar to \code{tune_x}, but with potential \code{trafo} applied.
Also, if the learner had some extra parameters statically set before tuning, these are included here.
\code{unnest} can have the values \code{"no"}, \code{"tune_x"} or \code{"params"}. If it is not set to \code{"no"}, settings of the respective list-column
are stored in separate columns instead of the list-column, and dependent, inactive parameters are encoded with \code{NA}.
}
\item \code{assign_result(tune_x, perf)}\cr
(\code{list}, \code{numeric}) -> \code{NULL}\cr
The tuner writes the best found list of settings and estimated performance values here. For internal use.
\itemize{
\item \code{tune_x}: Must be a named list of settings only of parameters from \code{param_set} and be feasible, untransformed.
\item \code{perf} : Must be a named numeric vector of performance measures, named with performance IDs, regarding all elements in \code{measures}.
}
}
}

\examples{
library(data.table)
library(paradox)
library(mlr3)

# Objects required to define the performance evaluator:
task = tsk("iris")
learner = lrn("classif.rpart")
resampling = rsmp("holdout")
measures = msr("classif.ce")
param_set = ParamSet$new(list(
  ParamDbl$new("cp", lower = 0.001, upper = 0.1),
  ParamInt$new("minsplit", lower = 1, upper = 10))
)

terminator = term("evals", n_evals = 5)
inst = TuningInstance$new(
  task = task,
  learner = learner,
  resampling = resampling,
  measures = measures,
  param_set = param_set,
  terminator = terminator
)

# first 4 points as cross product
design = CJ(cp = c(0.05, 0.01), minsplit = c(5, 3))
inst$eval_batch(design)
inst$archive()

# try more points, catch the raised terminated message
tryCatch(
  inst$eval_batch(data.table(cp = 0.01, minsplit = 7)),
  terminated_error = function(e) message(as.character(e))
)

# try another point although the budget is now exhausted
# -> no extra evaluations
tryCatch(
  inst$eval_batch(data.table(cp = 0.01, minsplit = 9)),
  terminated_error = function(e) message(as.character(e))
)

inst$archive()

### Error handling
# get a learner which breaks with 50\% probability
# set encapsulation + fallback
learner = lrn("classif.debug", error_train = 0.5)
learner$encapsulate = c(train = "evaluate", predict = "evaluate")
learner$fallback = lrn("classif.featureless")

param_set = ParamSet$new(list(
  ParamDbl$new("x", lower = 0, upper = 1)
))

inst = TuningInstance$new(
  task = tsk("wine"),
  learner = learner,
  resampling = rsmp("cv", folds = 3),
  measures = msr("classif.ce"),
  param_set = param_set,
  terminator = term("evals", n_evals = 5)
)

tryCatch(
  inst$eval_batch(data.table(x = 1:5 / 5)),
  terminated_error = function(e) message(as.character(e))
)

archive = inst$archive()

# column errors: multiple errors recorded
print(archive)
}
\concept{TuningInstance}
\keyword{datasets}
