% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_mme.R, R/calculate_mme_local.R
\name{calculate_mme}
\alias{calculate_mme}
\alias{calculate_mme_local}
\title{Calculate morphine milligram equivalents (MME)}
\usage{
calculate_mme(therapy_days, observation_window_days, medications)

calculate_mme_local(therapy_days, observation_window_days, medications)
}
\arguments{
\item{therapy_days}{Either a single positive number or a vector of two
positive numbers indicating the sum of prescription duration (days) for
each medication, but \emph{with each calendar day counted only ONCE}. When a
single number is provided, it is used for the both the "with
buprenorphine" and "without buprenorphine" MME calculations; when a
vector of 2 numbers is provided (e.g., \code{c(10, 18)}) then the first and
second numbers in the vector are used for the "with buprenorphine" and
"without buprenorphine" MME calculations, respectively. This is the
denominator for MME/Day definition 2.
\itemize{
\item If there is only one prescription \emph{or} if there is no calendar overlap
(i.e., no days on which more than one prescription is active) this
will be the same as the total days supply returned by the calculator
API (\code{total_days}).
\item If there are overlapping prescriptions, this is the number of \emph{unique}
calendar days.
}}

\item{observation_window_days}{Either a single positive number or a vector of
two positive numbers indicating a study-defined fixed observation window
of time. Typical choices are 7 day, 14 day, 30 day, 90 day. When a single
number is provided, it is used for the both the "with buprenorphine" and
"without buprenorphine" MME calculations; when a vector of 2 numbers is
provided (e.g., \code{c(7, 30)}) then the first and second numbers in the
vector are used for the "with buprenorphine" and "without buprenorphine"
MME calculations, respectively. This is the denominator for MME/Day
definition 3.}

\item{medications}{A list of medication definitions. Each element must be a
list containing each of the following fields:
\itemize{
\item \code{medication_name}: a string matching an API-accepted medication name and
its dosage units. To see a full list of API-accepted values, run
\code{\link[=get_med_list]{get_med_list()}}.
\item \code{dose}: a positive number indicating the dose of the associated opioid
medication listed in the \code{medication_name} field. Units of \code{dose} should
match the units listed in \code{medication_name}.
\item \code{doses_per_24_hours}: a positive number indicating the number of doses in
24 hours.
\item \code{days_of_medication}: a positive number indicating the duration of the
opioid medication prescription listed in the associated
\code{medication_name} in days.
}}
}
\value{
A list of MME calculations from the API. Will error if any medications are invalid
or if any numeric parameters are not positive numbers.
}
\description{
Calculates the single-day MME and total MME for each individual prescription
opioid medication submitted for calculation. Also calculates total MME,
total days of supply, and four distinct Total MME/Day calculations from
the NIH HEAL Online MME Calculator across all prescription medications
for two different medication groupings: 1) opioids without buprenorphine
and 2) opioids with buprenorphine.
}
\details{
\code{calculate_mme()} and \code{calculate_mme_local()} produce the same calculation
results with and without using the API, respectively. This helps overcome the
online calculator API rate limit of 50 (patient-level) requests per 15
minutes. In addition to returning user-specified arguments, \code{calculate_mme()}
also returns several  other variables mentioned in the \strong{Description}
section, which are described in more detail below. Output variable
description details are below; see
\href{https://www.doi.org/10.1097/j.pain.0000000000003529}{Adams, \emph{et al}. (2025)}
for a comprehensive overview.
}
\section{Prescription-Level}{
\strong{Conversion Factor for <\code{medication_name}>} (\code{factor}): the conversion
factor used for calculating total MME/day.

\strong{MME for <\code{medication_name}>} (\code{mme}): Morphine milligram equivalent
for the whole prescription specified in \code{medication_name}, calculated as
\code{(dose) * (doses_per_24_hours) * (factor) * (days_of_medication)}.

\strong{24h MME for <\code{medication_name}>} (\code{single_day_mme}): Morphine milligram
equivalent for the prescription specified in \code{medication_name} for a
single day, calculated as \code{(dose) * (doses_per_24_hours) * (factor)}.

\strong{One day}: Typically, the day with highest opioid exposure is entered, and
the sum of 24-hour MME across the drugs that apply to this day is
calculated. Highest MME in one day is definition 4.
}

\section{Summary-Level:}{
\strong{On-therapy Days} (\code{therapy_days}): The sum of prescription duration
(\code{days_of_medication}) for each medication, but \emph{with each calendar day
counted only ONCE}. User-supplied; this is the denominator for MME/Day
definition 2.
\itemize{
\item If there is only one prescription, or if there is no calendar overlap
(no days on which more than one prescription is active), this will be
the same as the total days supply.
\item If there are overlapping prescriptions, this is the number of unique
calendar days.
}

\strong{Total MME} (\code{total_mme}): The MME for each medication, summed across all prescriptions.
This is the numerator for MME/Day definitions 1, 2, and 3.

\strong{Total Days Supply} (\code{total_days}): The sum of the entered prescription
duration (\code{days_of_medication}) for each of the \code{medications} (Med 1
duration + med 2 duration...). Automatically calculated. This is the
denominator for MME/Day definition 1.
\subsection{MME/Day}{

MME/Day is an aggregate measure, calculating the total MME divided by a
specified time window (a number of days). The MME/Day definitions specify
the number of days:

\strong{MME/Day Definition 1 (\code{mme1}): Total Days Supply}

MME Definition 1 = Total MME / Total Days Supply time window (sum of entered prescription durations).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme1 = total_mme / total_days
}\if{html}{\out{</div>}}
\itemize{
\item Note that the same calendar day may contribute multiple times, if
overlapping prescriptions.
\item Reason to select this definition: This is the least complicated
calculation; appears best suited when immediate-release opioids are
prescribed for short discrete times.
\item Identified challenge with this definition: It consistently underestimated
MME per day when overlapping prescriptions were present or when
immediate-release and extended release opioids were prescribed
concurrently.
}

\strong{MME/Day Definition 2 (\code{mme2}): On-therapy Days}

MME Definition 2 = Total MME / On-therapy Days time window (sum of entered
prescription durations except each calendar day is counted only ONCE).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme2 = total_mme / therapy_days
}\if{html}{\out{</div>}}
\itemize{
\item Note - On-therapy Days unique calendar days.
\item Reason to select this definition: Provides a smoothed measure useful in
studies of dose-dependent adverse effects, including opioid-induced
constipation or overdose in patients with opioid tolerance or who have
been stable on opioids.
\item Identified challenge with this definition: The metric is time-varying and
affords the greatest flexibility to define medication gap periods and
leftover/unused medications to improve pharmacoepidemiologic studies.
}

\strong{MME/Day Definition 3 (\code{mme3}): Fixed Observation Window}

Uses the Total MME study-specified fixed observation window. MME Definition
3 = Total MME / Number of days in observation window:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme3 = total_mme / observation_window_days
}\if{html}{\out{</div>}}
\itemize{
\item If this definition is selected, it is important to report on the duration
of the fixed window.
\item Reason to select this definition: Most suitable for studies with a known
or suspected duration of risk during which adverse events are expected
to occur, such as incidence of opioid use disorder. This definition may
be useful when prescriptions are filled at irregular time intervals on a
as needed basis (\emph{pro re nata}, PRN).
\item Identified challenge with this definition: The definition consistently had
the lowest milligrams per day for immediate-release opioids. It is the
most robust to misspecification, amenable to transformations, and has
the least noise when constructing continuous functions. However, since
it assumes uniform exposure/risk within a window, there is less scope
for time-varying adjustment.
\item This is the definition recommended by the Department of Health and Human
Services Office of the Inspector General.
}

\strong{MME/Day Definition 4 (\code{mme4}): Maximum Daily Dose}

Uses the sum of 24-hour MME for the day with highest opioid exposure.

MME Definition 4 = Drug 1 (dose (mg) x # of doses per day) x conversion
factor + Drug 2 (dose (mg) x # of doses per day) x conversion factor + ...

\if{html}{\out{<div class="sourceCode r">}}\preformatted{mme4 = sum(dose * doses_per_24_hours * factor)
}\if{html}{\out{</div>}}
\itemize{
\item Report the highest single-day exposure.
\item Reason to select this definition: A toxicological perspective may be
appropriate for patients with no opioid tolerance and in the presence
of comorbidities for respiratory depression. It appears to be best
suited for immediate dose-dependent toxic effects, such as respiratory
depression.
\item Identified challenged with this definition: This definition may have
limited use if it includes opioids where fatal toxicity does not involve
respiratory depression (e.g., tramadol) or have atypical \emph{mu}-opioid
receptor agonism (e.g., tapentadol, buprenorphine).
\item The definition assumes uniform risk of adverse outcomes regardless of time
on-therapy. More so than the others, this definition is prone to
influence from early refills, unused medication, and how the 90 MME
threshold is operationalized.
\item This definition underlies the algorithm embedded in the CDC Opioid
Guideline mobile app. There may be difficulty reconciling findings with
studies using the other definitions because it returns a MME per day
that is significantly higher.
\item This calculator sums the 24-hour MME for every prescription, without
considering calendar dates.
}
}
}

\examples{
meds_list <- list(
  list(
    medication_name = "Buprenorphine buccal film (mcg) buccal",
    dose = 50,
    doses_per_24_hours = 2,
    days_of_medication = 5
    ),
  list(
    medication_name = "Hydrocodone (mg)",
    dose = 75,
    doses_per_24_hours = 3,
    days_of_medication = 10
    )
)

calculate_mme(10, 5, meds_list)

# Clean up meds_list
rm(meds_list)
}
\seealso{
\code{\link[=calculate_mme_df]{calculate_mme_df()}} which this function wraps.
}
