% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelplot.R
\name{modelplot}
\alias{modelplot}
\title{Model Summary Plots with Estimates and Confidence Intervals}
\usage{
modelplot(
  models,
  conf_level = 0.95,
  coef_map = NULL,
  coef_omit = NULL,
  coef_rename = NULL,
  vcov = NULL,
  exponentiate = FALSE,
  add_rows = NULL,
  facet = FALSE,
  draw = TRUE,
  background = NULL,
  ...
)
}
\arguments{
\item{models}{a model, (named) list of models, or nested list of models.
\itemize{
\item Single model: \code{modelsummary(model)}
\item Unnamed list of models: \code{modelsummary(list(model1, model2))}
\itemize{
\item Models are labelled automatically. The default label style can be altered by setting a global option. See below.
}
\item Named list of models: \code{modelsummary(list("A"=model1, "B"=model2))}
\itemize{
\item Models are labelled using the list names.
}
\item Nested list of models:
\itemize{
\item When using the \code{shape} argument with "rbind", "rcollapse", or "cbind" values, \code{models} can be a nested list of models to display "panels" or "stacks" of regression models. See the \code{shape} argument documentation and examples below.
}
}}

\item{conf_level}{numeric value between 0 and 1. confidence level to use for
confidence intervals. Setting this argument to \code{NULL} does not extract
confidence intervals, which can be faster for some models.}

\item{coef_map}{character vector. Subset, rename, and reorder coefficients.
Coefficients omitted from this vector are omitted from the table. The order
of the vector determines the order of the table. \code{coef_map} can be a named
or an unnamed character vector. If \code{coef_map} is a named vector, its values
define the labels that must appear in the table, and its names identify the
original term names stored in the model object: \code{c("hp:mpg"="HPxM/G")}. See
Examples section below.}

\item{coef_omit}{integer vector or regular expression to identify which coefficients to omit (or keep) from the table. Positive integers determine which coefficients to omit. Negative integers determine which coefficients to keep. A regular expression can be used to omit coefficients, and perl-compatible "negative lookaheads" can be used to specify which coefficients to \emph{keep} in the table. Examples:
\itemize{
\item c(2, 3, 5): omits the second, third, and fifth coefficients.
\item c(-2, -3, -5): negative values keep the second, third, and fifth coefficients.
\item \code{"ei"}: omit coefficients matching the "ei" substring.
\item \code{"^Volume$"}: omit the "Volume" coefficient.
\item \code{"ei|rc"}: omit coefficients matching either the "ei" or the "rc" substrings.
\item \code{"^(?!Vol)"}: keep coefficients starting with "Vol" (inverse match using a negative lookahead).
\item \code{"^(?!.*ei)"}: keep coefficients matching the "ei" substring.
\item \code{"^(?!.*ei|.*pt)"}: keep coefficients matching either the "ei" or the "pt" substrings.
\item See the Examples section below for complete code.
}}

\item{coef_rename}{logical, named or unnamed character vector, or function
\itemize{
\item Logical: TRUE renames variables based on the "label" attribute of each column. See the Example section below.
\item Unnamed character vector of length equal to the number of coefficients in the final table, after \code{coef_omit} is applied.
\item Named character vector: Values refer to the variable names that will appear in the table. Names refer to the original term names stored in the model object. Ex: c("hp:mpg"="hp X mpg")
\item Function: Accepts a character vector of the model's term names and returns a named vector like the one described above. The \code{modelsummary} package supplies a \code{coef_rename()} function which can do common cleaning tasks: \code{modelsummary(model, coef_rename = coef_rename)}
}}

\item{vcov}{robust standard errors and other manual statistics. The \code{vcov}
argument accepts six types of input (see the 'Details' and 'Examples'
sections below):
\itemize{
\item NULL returns the default uncertainty estimates of the model object
\item string, vector, or (named) list of strings. "iid", "classical", and "constant" are aliases for \code{NULL}, which returns the model's default uncertainty estimates. The strings "HC", "HC0", "HC1" (alias: "stata"), "HC2", "HC3" (alias: "robust"), "HC4", "HC4m", "HC5", "HAC", "NeweyWest", "Andrews", "panel-corrected", "outer-product", and "weave" use variance-covariance matrices computed using functions from the \code{sandwich} package, or equivalent method. The behavior of those functions can (and sometimes \emph{must}) be altered by passing arguments to \code{sandwich} directly from \code{modelsummary} through the ellipsis (\code{...}), but it is safer to define your own custom functions as described in the next bullet.
\item function or (named) list of functions which return variance-covariance matrices with row and column names equal to the names of your coefficient estimates (e.g., \code{stats::vcov}, \code{sandwich::vcovHC}, \code{function(x) vcovPC(x, cluster="country")}).
\item formula or (named) list of formulas with the cluster variable(s) on the right-hand side (e.g., ~clusterid).
\item named list of \code{length(models)} variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
\item a named list of length(models) vectors with names equal to the names of your coefficient estimates. See 'Examples' section below. Warning: since this list of vectors can include arbitrary strings or numbers, \code{modelsummary} cannot automatically calculate p values. The \code{stars} argument may thus use incorrect significance thresholds when \code{vcov} is a list of vectors.
}}

\item{exponentiate}{TRUE, FALSE, or logical vector of length equal to the
number of models. If TRUE, the \code{estimate}, \code{conf.low}, and \code{conf.high}
statistics are exponentiated, and the \code{std.error} is transformed to
\code{exp(estimate)*std.error}.}

\item{add_rows}{a data.frame (or tibble) with the same number of columns as
your main table. By default, rows are appended to the bottom of the table.
You can define a "position" attribute of integers to set the row positions.
See Examples section below.}

\item{facet}{TRUE or FALSE. When the 'models' argument includes several
model objects, TRUE draws terms in separate facets, and FALSE draws terms
side-by-side (dodged).}

\item{draw}{TRUE returns a 'ggplot2' object, FALSE returns the data.frame
used to draw the plot.}

\item{background}{A list of 'ggplot2' geoms to add to the background of the
plot.  This is especially useful to display annotations "behind" the
'geom_pointrange' that 'modelplot' draws.}

\item{...}{all other arguments are passed through to three functions. See the documentation of these functions for lists of available arguments.
\itemize{
\item \link[parameters:model_parameters]{parameters::model_parameters} extracts parameter estimates. Available arguments depend on model type, but include:
\itemize{
\item \code{standardize}, \code{include_reference}, \code{centrality}, \code{dispersion}, \code{test}, \code{ci_method}, \code{prior}, \code{diagnostic}, \code{rope_range}, \code{power}, \code{cluster}, etc.
}
\item \link[performance:model_performance]{performance::model_performance} extracts goodness-of-fit statistics. Available arguments depend on model type, but include:
\itemize{
\item \code{metrics}, \code{estimator}, etc.
}
\item \link[tinytable:tt]{tinytable::tt}, \link[kableExtra:kbl]{kableExtra::kbl} or \link[gt:gt]{gt::gt} draw tables, depending on the value of the \code{output} argument.
}}
}
\description{
Dot-Whisker plot of coefficient estimates with confidence intervals. For
more information, see the Details and Examples sections below, and the
vignettes on the \code{modelsummary} website:
https://modelsummary.com/
\itemize{
\item \href{https://modelsummary.com/articles/modelplot.html}{modelplot Vignette.}
}
}
\section{Examples}{


\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(modelsummary)

# single model
mod <- lm(hp ~ vs + drat, mtcars)
modelplot(mod)

# omit terms with string matches or regexes
modelplot(mod, coef_omit = 'Interc')

# rename, reorder and subset with 'coef_map'
cm <- c('vs' = 'V-shape engine',
  'drat' = 'Rear axle ratio')
modelplot(mod, coef_map = cm)

# several models
models <- list()
models[['Small model']] <- lm(hp ~ vs, mtcars)
models[['Medium model']] <- lm(hp ~ vs + factor(cyl), mtcars)
models[['Large model']] <- lm(hp ~ vs + drat + factor(cyl), mtcars)
modelplot(models)

# add_rows: add an empty reference category

mod <- lm(hp ~ factor(cyl), mtcars)

add_rows = data.frame(
  term = "factory(cyl)4",
  model = "(1)",
  estimate = NA)
attr(add_rows, "position") = 3
modelplot(mod, add_rows = add_rows)


# customize your plots with 'ggplot2' functions
library(ggplot2)

modelplot(models) +
  scale_color_brewer(type = 'qual') +
  theme_classic()

# pass arguments to 'geom_pointrange' through the ... ellipsis
modelplot(mod, color = 'red', size = 1, fatten = .5)

# add geoms to the background, behind geom_pointrange
b <- list(geom_vline(xintercept = 0, color = 'orange'),
  annotate("rect", alpha = .1,
    xmin = -.5, xmax = .5,
    ymin = -Inf, ymax = Inf),
  geom_point(aes(y = term, x = estimate), alpha = .3,
    size = 10, color = 'red', shape = 'square'))
modelplot(mod, background = b)

# logistic regression example
df <- as.data.frame(Titanic)
mod_titanic <- glm(
  Survived ~ Class + Sex,
  family = binomial,
  weight = Freq,
  data = df
)

# displaying odds ratio using a log scale
modelplot(mod_titanic, exponentiate = TRUE) +
  scale_x_log10() +
  xlab("Odds Ratios and 95\% confidence intervals")
}\if{html}{\out{</div>}}
}

\references{
Arel-Bundock V (2022). “modelsummary: Data and Model Summaries in R.” \emph{Journal of Statistical Software}, \emph{103}(1), 1-23. \doi{10.18637/jss.v103.i01}.'
}
