% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modmedmlm.R
\name{boot.modmed.mlm}
\alias{boot.modmed.mlm}
\title{Boot function for (moderated) mediation with 2-level multilevel models}
\usage{
boot.modmed.mlm(
  data,
  indices,
  L2ID,
  ...,
  type = "all",
  modval1 = NULL,
  modval2 = NULL,
  boot.lvl = c("both", "1", "2")
)
}
\arguments{
\item{data}{Data frame in long format.}

\item{indices}{\code{boot} requires the function signature to accept a vector of
index numbers and so this argument is required. If the index numbers are all in order starting at 1,
then the relevant model will be fit to the data without any resampling. If some other vector is supplied,
then resampling is done as described in details.}

\item{L2ID}{Name of column that contains grouping variable in 'data' (e.g., "SubjectID")}

\item{...}{Arguments passed to \code{\link{modmed.mlm}} to define the mediation analysis model.}

\item{type}{Character that defines what information to extract from the model. Default and options are in \code{\link{extract.modmed.mlm}}.
As examples, "indirect" will compute the indirect effect, "all" will save all random and fixed effects for possible additional
computations, "indirect.diff" will compute the difference in the indirect effect at two values of a possible moderating variable.}

\item{modval1}{(Optional) Numeric. If the model has a moderator, this value will be passed to \code{\link{extract.modmed.mlm}}
to compute the indirect effect or other effects at that value. See \code{\link{extract.modmed.mlm}} for details.}

\item{modval2}{(Optional). If the model has a moderator, it is possible to compute the difference in the indirect
at two values of the moderator. If given and an appropriate option for such a difference is chosen for \code{type},
this value and that of \code{modval1} will be passed to \code{\link{extract.modmed.mlm}} to compute and save the difference.
This is useful for obtaining a CI for the difference in the indirect effect at two different levels of the moderator.}

\item{boot.lvl}{Character that defines at what level resampling should occur. Options are "both", "1", or "2". "both" will sample L2 units
and then L1 units w/in each cluster. This has been noted to result in unequal sample sizes if the original clusters did not have equal sample sizes.
"2" resamples only L2 units and leaves all L1 units intact. "1" will assume that whatever indices are fed from the boot function will
be used. This probably only makes sense if \code{strata} is specified.}
}
\value{
A vector of parameter estimates, depending on the value of \code{type} specified as input. Once the model is estimated,
  \code{\link{extract.modmed.mlm} is used to obtain the parameter estimates.}
}
\description{
Boot function for (moderated) mediation with 2-level multilevel models
}
\details{
Implements function to do bootstrapping with the 1-1-1 multilevel mediation analysis models as used in Falk, Vogel,
  Hammami & Miočević (in press). For use with boot package. This function aides in implementing case resampling methods
  with support for resampling at level 2, level 1, or both (e.g., see Hox and van de Schoot, 2013; van der Leeden, Meijer, & Busing, 2008).
  These functions also support moderated mediation. See also \code{\link{modmed.mlm}}. Note that \code{\link{nlm}} was used as the optimizer
  for some of the examples below as it was found to be faster for the models/simulations studied by Falk et al (in press).
}
\examples{

\donttest{

# Note that for all examples below, R should be increased to something
#  MUCH larger (e.g., 1000). Small values here are used only so that the code
#  runs relatively quickly when tested.

## Mediation for 1-1-1 model
data(BPG06dat)

#Setup parallel processing
# snow appears to work on Windows; something else may be better on Unix/Mac/Linux
library(parallel)
library(boot)
ncpu<-2
RNGkind("L'Ecuyer-CMRG") # set type of random number generation that works in parallel
cl<-makeCluster(ncpu)
clusterSetRNGStream(cl, 9912) # set random number seeds for cluster

# bootstrap all fixed and random effects (type="all")
boot.result<-boot(BPG06dat, statistic=boot.modmed.mlm, R=10,
  L2ID = "id", X = "x", Y = "y", M = "m",
  random.a=TRUE, random.b=TRUE, random.cprime=TRUE,
  type="all",
  control=list(opt="nlm"),
  parallel="snow",ncpus=ncpu,cl=cl)

# Point estimate and 95\% CI for indirect effect
extract.boot.modmed.mlm(boot.result, type="indirect", ci.conf=.95)

stopCluster(cl) # shut down cluster

## Moderated mediation

data(simdat)

# Note: use of nlm apparently fails in this moderated mediation model
# default optimizer for lme instead is used

## Bootstrap w/ moderation of a and b paths
set.seed(1234)

boot.result2<-boot(simdat, statistic=boot.modmed.mlm, R=5,
 L2ID = "L2id", X = "X", Y = "Y", M = "M",
   random.a=TRUE, random.b=TRUE, random.cprime=TRUE,
   moderator = "mod", mod.a=TRUE, mod.b=TRUE,
   type="all")

# indirect effect point estimate and 95\% CI when moderator = 0
extract.boot.modmed.mlm(boot.result2, type="indirect")
extract.boot.modmed.mlm(boot.result2, type="indirect", modval1=0)

# indirect effect point estimate and 95\% CI when moderator = 1
extract.boot.modmed.mlm(boot.result2, type="indirect", modval1=1)

# indirect effect difference point estimate and 95\% CI
extract.boot.modmed.mlm(boot.result2, type="indirect.diff",
   modval1=0, modval2=1)

# Example to not fail when using missing values
# See documentation for lme function from the nlme package for other
# options for na.action
dat.miss <- BPG06dat
dat.miss$m[c(1,2,3,4)]<-NA
dat.miss$y[c(5,6,7,8)]<-NA
boot.result<-boot(dat.miss, statistic=boot.modmed.mlm, R=5,
  L2ID = "id", X = "x", Y = "y", M = "m",
  random.a=TRUE, random.b=TRUE, random.cprime=TRUE,
  type="all",
  control=list(opt="nlm"),
  na.action = na.omit)

}
}
\references{
Bauer, D. J., Preacher, K. J., & Gil, K. M. (2006). Conceptualizing and testing random indirect effects and moderated mediation in multilevel models: New procedures and recommendations. Psychological Methods, 11(2), 142–163. \doi{10.1037/1082-989X.11.2.142}

Falk, C. F., Vogel, T., Hammami, S., & Miočević, M. (in press). Multilevel mediation analysis in R: A comparison of bootstrap and Bayesian approaches. Behavior Research Methods. \doi{10.3758/s13428-023-02079-4}  Preprint: \doi{10.31234/osf.io/ync34}

Hox, J., & van de Schoot, R. (2013). Robust methods for multilevel analysis. In M. A. Scott, J. S. Simonoff & B. D. Marx (Eds.), The SAGE Handbook of Multilevel Modeling (pp. 387-402). SAGE Publications Ltd. \doi{10.4135/9781446247600.n22}

van der Leeden, R., Meijer, E., & Busing, F. M. T. A. (2008). Resampling multilevel models. In J. de Leeuw & E. Meijer (Eds.), Handbook of Multilevel Analysis (pp. 401-433). Springer.
}
