context("neuron fileformats")

test_that("We can query fileformats",{
  expect_equal(fileformats(ext='swc',rval='names'),'swc')
  expect_equal(fileformats(ext='am', class='neuron', rval='names'),
               c('hxlineset','hxskel'))
  expect_is(fileformats(class='neuron',rval='info'),'matrix')
  
  expect_is(fw<-getformatwriter(file='test.rds'),'list')
  expect_equal(fw$ext,'.rds')
  expect_equal(fw$read,readRDS)
  expect_equal(fw$write,saveRDS)
  
  expect_equal(getformatwriter(file='test.am', format='rds')$file,'test.am')
  expect_equal(getformatwriter(file='test.am', format='rds', ext=NA)$file,'test.rds')
  expect_equal(getformatwriter(file='test.am', ext='.rds')$ext,'.rds')
  
  expect_error(getformatwriter(file='test.rds', ext='.rhubarb'))
  
  expect_equal(fileformats(format='hxl', ext='_skel.am', class='neuron'),
               'hxlineset')
})

context("read neurons")

test_that("We can read single neurons in rda or rds format", {
  rda=tempfile(fileext='.rda')
  rds=tempfile(fileext='.rds')
  on.exit(unlink(c(rda,rds)))
  
  n=Cell07PNs[[1]]
  save(n,file=rda)
  saveRDS(n,file=rds)
  expect_equivalent(n,read.neuron(rda))
  expect_equivalent(n,read.neuron(rds))
  
  # check that we can read neurons in rda or rds format
  # even if they do not have an appropriate file extension
  file.copy(rds,tfrds<-tempfile())
  file.copy(rda,tfrda<-tempfile())
  on.exit(unlink(c(tfrda,tfrds)),add=TRUE)
  
  expect_equivalent(n,read.neuron(tfrds,format='rds'))
  expect_equivalent(n,read.neuron(tfrda,format='rda'))
  
  # check that a length 1 neuronlist works ok
  expect_equivalent(n,read.neurons(tfrds,format='rds')[[1]])
})

test_that("We can read single dotprops objects in rda or rds format", {
  rda=tempfile(fileext='.rda')
  rds=tempfile(fileext='.rds')
  on.exit(unlink(c(rda,rds)))
  
  n=kcs20[[1]]
  save(n,file=rda)
  saveRDS(n,file=rds)
  expect_equivalent(n,read.neuron(rda))
  expect_equivalent(n,read.neuron(rds))
  
  # check that we can read dotprops objects in rda or rds format
  # even if they do not have an appropriate file extension
  file.copy(rds,tfrds<-tempfile())
  file.copy(rda,tfrda<-tempfile())
  on.exit(unlink(c(tfrda,tfrds)),add=TRUE)
  expect_equivalent(n,read.neuron(tfrds,format='rds'))
  expect_equivalent(n,read.neuron(tfrda,format='rda'))
  
  expect_equivalent(n,read.neurons(tfrds,format='rds')[[1]])
})

test_that("We can read neurons in swc format", {
  swc='testdata/neuron/EBT7R.CNG.swc'
  expect_is(n<-read.neuron(swc),'neuron')
  expect_equal(n$NeuronName,'EBT7R.CNG')
})

test_that("We can set the NeuronName field", {
  swc='testdata/neuron/EBT7R.CNG.swc'
  n<-read.neuron(swc, NeuronName="rhubarb")
  expect_equal(n$NeuronName,'rhubarb')
  # check that we can use a user defined function to define the NeuronName
  nfun=function(x) sub("\\..*","",basename(x))
  n<-read.neuron(swc, NeuronName=nfun)
  expect_equal(n$NeuronName,'EBT7R')
})

test_that("We can read in neurons as a neuronlist",{
  expect_is(nl<-read.neurons(paths='testdata/neuron/',pattern='\\.CNG\\.swc$',
               neuronnames=function(x) sub("\\..*","",basename(x))),'neuronlist')
  expect_equal(length(nl),2)
  
  # check that InputFileName field is not mangled
  expect_true('InputFileName'%in%names(nl[[1]]))
  
  fieldsToIgnore=c("CreatedAt",'InputFileStat')
  n.read.neurons=nl[[1]]
  n.read.neurons[fieldsToIgnore]=NULL
  n.read.neuron=read.neuron(n.read.neurons$InputFileName)
  n.read.neuron[fieldsToIgnore]=NULL
  expect_equal(unclass(n.read.neuron),unclass(n.read.neurons), 
               info = 'check equality of neuron read by read.neuron & read.neurons')
  
  # check that problem files are named on error/warning
  expect_message(read.neurons('testdata/neuron/Neurites.am'),
                 regexp = 'While reading file.*Neurites\\.am')
})

test_that("We can read hxlineset format neurons",{
  
  ebt7=structure(list(NeuronName = "EBT7R", InputFileName = "testdata/neuron/EBT7R.am", 
      CreatedAt = structure(1391870899.00482, class = c("POSIXct", 
      "POSIXt")), NodeName = structure("mac1041-14.lmb.internal", .Names = "nodename"), 
      InputFileStat = structure(list(size = 9870, isdir = FALSE, 
          mode = structure(493L, class = "octmode"), mtime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), ctime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), atime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), uid = 501L, gid = 80L, uname = "jefferis", 
          grname = "admin"), .Names = c("size", "isdir", "mode", 
      "mtime", "ctime", "atime", "uid", "gid", "uname", "grname"
      ), row.names = "testdata/neuron/EBT7R.am", class = "data.frame"), 
      InputFileMD5 = structure("9e5016e8722314537bb4344ab2877f03", .Names = "testdata/neuron/EBT7R.am"), 
      NumPoints = 343L, StartPoint = 1L, BranchPoints = c(26L, 
      38L, 56L, 65L, 77L, 90L, 92L, 117L, 121L, 127L, 135L, 141L, 
      153L, 154L, 158L, 173L, 191L, 195L, 206L, 211L, 214L, 234L, 
      239L, 243L, 255L, 264L, 279L, 280L, 284L, 285L, 313L, 318L, 
      327L, 336L), EndPoints = c(1L, 37L, 49L, 64L, 76L, 86L, 104L, 
      120L, 133L, 138L, 148L, 162L, 166L, 169L, 175L, 188L, 197L, 
      202L, 216L, 223L, 232L, 238L, 254L, 257L, 268L, 276L, 282L, 
      295L, 300L, 307L, 316L, 320L, 322L, 334L, 337L, 343L), nTrees = 1, 
      NumSegs = 69L, SegList = structure(list(1:26, 26:37, c(26L, 
      38L), 38:49, c(38L, 50L, 51L, 52L, 53L, 54L, 55L, 56L), 56:64, 
          c(56L, 65L), 65:76, c(65L, 77L), 77:86, c(77L, 87L, 88L, 
          89L, 90L), 90:92, 92:104, c(92L, 105L, 106L, 107L, 108L, 
          109L, 110L, 111L, 112L, 113L, 114L, 115L, 116L, 117L), 
          117:120, c(117L, 121L), 121:127, 127:133, c(127L, 134L, 
          135L), 135:138, c(135L, 139L, 140L, 141L), 141:148, c(141L, 
          149L, 150L, 151L, 152L, 153L), 153:154, 154:158, 158:162, 
          c(158L, 163L, 164L, 165L, 166L), c(154L, 167L, 168L, 
          169L), c(153L, 170L, 171L, 172L, 173L), 173:175, c(173L, 
          176L, 177L, 178L, 179L, 180L, 181L, 182L, 183L, 184L, 
          185L, 186L, 187L, 188L), c(121L, 189L, 190L, 191L), 191:195, 
          195:197, c(195L, 198L, 199L, 200L, 201L, 202L), c(191L, 
          203L, 204L, 205L, 206L), 206:211, 211:214, 214:216, c(214L, 
          217L, 218L, 219L, 220L, 221L, 222L, 223L), c(211L, 224L, 
          225L, 226L, 227L, 228L, 229L, 230L, 231L, 232L), c(206L, 
          233L, 234L), 234:238, c(234L, 239L), 239:243, 243:254, 
          c(243L, 255L), 255:257, c(255L, 258L, 259L, 260L, 261L, 
          262L, 263L, 264L), 264:268, c(264L, 269L, 270L, 271L, 
          272L, 273L, 274L, 275L, 276L), c(239L, 277L, 278L, 279L
          ), 279:280, 280:282, c(280L, 283L, 284L), 284:285, 285:295, 
          c(285L, 296L, 297L, 298L, 299L, 300L), c(284L, 301L, 
          302L, 303L, 304L, 305L, 306L, 307L), c(279L, 308L, 309L, 
          310L, 311L, 312L, 313L), 313:316, c(313L, 317L, 318L), 
          318:320, c(318L, 321L, 322L), c(90L, 323L, 324L, 325L, 
          326L, 327L), 327:334, c(327L, 335L, 336L), 336:337, c(336L, 
          338L, 339L, 340L, 341L, 342L, 343L)), class = c("seglist", 
      "list")), d = structure(list(PointNo = 1:343, Label = c(2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
      2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2), 
          X = c(12.75, 14.43, 15.84, 16.54, 17.53, 18.09, 18.65, 
          19.36, 19.5, 19.5, 19.22, 19.92, 20.76, 22.59, 24.28, 
          25.97, 28.36, 29.91, 32.72, 34.83, 37.36, 39.89, 42.28, 
          44.96, 46.93, 49.18, 51.99, 53.82, 55.23, 56.77, 58.04, 
          58.32, 58.32, 59.3, 61.84, 64.93, 66.2, 50.3, 49.46, 
          48.33, 47.49, 46.08, 45.66, 45.24, 45.94, 47.77, 48.61, 
          49.88, 50.02, 52.69, 54.38, 56.91, 59.45, 61.27, 62.54, 
          64.93, 65.49, 65.49, 65.78, 66.62, 65.92, 64.79, 63.67, 
          63.1, 66.06, 67.04, 69.85, 72.67, 76.89, 79.7, 82.37, 
          84.48, 86.59, 88.14, 89.41, 90.81, 69.43, 70.14, 71.12, 
          70.7, 69.43, 68.17, 67.32, 66.9, 67.18, 68.03, 70.98, 
          73.51, 75.62, 76.61, 79.28, 80.12, 79.42, 79.28, 79.98, 
          80.54, 81.25, 83.22, 84.62, 86.17, 90.11, 92.64, 94.05, 
          96.02, 81.67, 83.78, 86.03, 87.44, 89.69, 93.48, 97.56, 
          99.67, 102.21, 104.6, 106.28, 107.97, 108.96, 107.69, 
          106.28, 105.86, 110.93, 112.76, 114.02, 115.85, 117.82, 
          119.37, 120.63, 119.93, 118.94, 118.94, 118.1, 116.27, 
          115.01, 122.46, 123.87, 123.16, 122.46, 121.9, 125.84, 
          127.67, 128.93, 128.23, 127.95, 127.67, 126.96, 126.82, 
          126.12, 125.56, 130.76, 132.87, 134.56, 137.23, 138.64, 
          139.62, 141.31, 143.98, 145.39, 147.08, 148.34, 149.05, 
          149.47, 149.19, 148.34, 149.61, 150.73, 151.16, 140.47, 
          141.31, 141.87, 139.34, 142.15, 144.12, 145.25, 145.67, 
          145.67, 146.51, 147.92, 149.19, 150.73, 152, 153.12, 
          152.56, 151.58, 150.17, 149.47, 148.34, 147.5, 146.23, 
          112.33, 114.16, 115.01, 114.44, 112.05, 110.79, 110.08, 
          110.36, 110.5, 110.22, 109.52, 108.54, 107.69, 106.57, 
          115.85, 116.98, 118.24, 118.52, 117.26, 116.41, 115.15, 
          113.88, 113.18, 112.47, 111.07, 110.22, 111.35, 111.35, 
          109.66, 108.82, 107.55, 106.85, 105.44, 103.89, 103.33, 
          115.57, 116.27, 117.96, 119.93, 121.9, 124.01, 125.41, 
          127.95, 128.37, 119.65, 120.35, 119.79, 120.63, 121.62, 
          121.76, 121.48, 123.59, 125.27, 126.4, 127.1, 125.98, 
          125.84, 124.57, 124.15, 124.43, 125.13, 126.26, 127.24, 
          128.09, 130.76, 132.17, 128.09, 127.95, 127.38, 130.48, 
          132.31, 133.29, 135.96, 137.09, 138.92, 138.92, 137.51, 
          136.39, 136.1, 136.1, 140.04, 140.47, 141.59, 142.15, 
          142.72, 143.42, 144.4, 144.54, 122.32, 123.87, 124.57, 
          124.99, 124.57, 123.73, 126.12, 127.38, 127.95, 129.21, 
          130.48, 132.45, 133.43, 134.84, 136.81, 138.21, 139.06, 
          139.9, 140.47, 129.21, 130.9, 132.31, 133.99, 135.12, 
          126.82, 126.4, 124.99, 123.73, 122.74, 123.3, 123.59, 
          126.4, 129.35, 131.46, 132.73, 134.7, 137.51, 138.78, 
          139.48, 139.2, 139.48, 140.04, 140.47, 140.61, 141.45, 
          142.01, 77.45, 79.14, 80.83, 82.51, 83.22, 82.79, 82.37, 
          82.23, 84.34, 86.59, 88.7, 89.55, 84.62, 85.75, 85.75, 
          87.86, 90.25, 92.5, 94.47, 97.28, 98.41), Y = c(-121.51, 
          -119.4, -117.57, -115.89, -114.62, -113.21, -111.24, 
          -109.27, -107.31, -105.76, -103.93, -101.96, -99.57, 
          -97.46, -96.19, -94.79, -93.8, -92.68, -91.27, -90, -89.44, 
          -89.02, -88.74, -88.74, -88.88, -89.02, -90.57, -92.54, 
          -93.52, -95.07, -96.76, -99.01, -100.41, -102.38, -104.49, 
          -106.74, -107.31, -88.46, -86.63, -85.08, -83.67, -82.27, 
          -81, -79.17, -77.06, -76.92, -77.35, -78.89, -79.45, 
          -87.75, -86.91, -86.91, -88.32, -88.04, -87.33, -86.35, 
          -84.8, -83.82, -83.25, -82.55, -81.99, -81.56, -81.56, 
          -81.71, -86.21, -87.89, -91.13, -93.52, -95.21, -96.76, 
          -97.74, -100.41, -102.38, -103.79, -104.21, -104.63, 
          -85.08, -87.19, -87.61, -88.6, -89.72, -91.69, -94.08, 
          -97.32, -100.84, -102.1, -84.66, -83.67, -83.39, -83.11, 
          -82.13, -81.56, -80.16, -79.03, -78.05, -76.64, -76.08, 
          -75.24, -74.67, -75.09, -76.5, -77.35, -78.19, -80.72, 
          -80.72, -79.74, -79.6, -79.45, -79.45, -79.17, -79.03, 
          -78.89, -78.75, -78.75, -78.33, -78.05, -77.63, -76.92, 
          -76.22, -75.38, -77.63, -78.61, -79.45, -80.86, -81.56, 
          -82.41, -83.39, -82.41, -81.71, -81.14, -80.58, -79.88, 
          -79.31, -84.1, -84.8, -86.07, -86.49, -86.91, -85.36, 
          -86.21, -87.47, -87.33, -86.21, -84.8, -82.69, -80.58, 
          -78.33, -76.08, -86.91, -87.19, -88.46, -90, -90.71, 
          -90.15, -88.74, -87.61, -86.63, -85.78, -87.05, -89.02, 
          -91.13, -92.25, -84.24, -83.11, -82.69, -82.41, -89.72, 
          -89.58, -89.86, -92.25, -93.66, -93.94, -93.38, -94.08, 
          -95.07, -92.11, -91.41, -89.44, -87.75, -85.93, -84.24, 
          -82.27, -80.02, -77.06, -74.81, -72.98, -71.72, -69.89, 
          -76.08, -74.81, -74.11, -74.25, -74.95, -74.67, -74.81, 
          -73.41, -72.56, -75.8, -76.36, -76.5, -75.52, -74.95, 
          -72.84, -71.16, -68.34, -67.92, -67.78, -67.78, -67.5, 
          -67.64, -67.64, -67.5, -67.08, -66.51, -66.09, -65.25, 
          -67.22, -67.5, -67.22, -67.22, -67.64, -67.78, -67.78, 
          -67.64, -67.5, -67.64, -68.62, -69.19, -69.33, -68.76, 
          -68.76, -68.76, -67.22, -66.37, -65.81, -65.11, -64.69, 
          -64.69, -65.53, -66.09, -66.09, -66.65, -66.65, -69.05, 
          -70.17, -71.86, -73.13, -76.22, -77.35, -79.74, -81.56, 
          -82.97, -84.8, -85.36, -66.37, -65.25, -64.26, -66.37, 
          -66.23, -66.94, -67.78, -69.61, -71.44, -73.13, -73.97, 
          -75.24, -76.22, -76.64, -74.11, -75.8, -78.33, -80.58, 
          -82.41, -83.96, -85.22, -86.07, -63.84, -62.72, -62.58, 
          -61.59, -60.33, -59.76, -60.47, -59.34, -59.76, -61.17, 
          -60.89, -60.33, -59.9, -59.76, -60.18, -60.33, -61.03, 
          -62.01, -62.01, -58.78, -58.22, -58.64, -58.92, -59.48, 
          -58.64, -58.64, -58.78, -59.48, -60.04, -61.17, -61.31, 
          -62.72, -62.01, -61.31, -60.75, -60.47, -61.73, -62.86, 
          -63.84, -63.7, -62.15, -62.29, -62.29, -62.15, -62.86, 
          -63.28, -83.67, -84.1, -84.66, -85.64, -86.21, -87.47, 
          -88.32, -91.41, -93.52, -95.07, -95.07, -95.35, -86.35, 
          -86.49, -87.19, -86.49, -86.49, -87.75, -89.58, -90.29, 
          -90.85), Z = c(0, -2, -4, -6, -9, -12, -15, -18, -20, 
          -23, -27, -30, -33, -36, -37, -38, -39, -42, -44, -45, 
          -47, -48, -50, -51, -52, -55, -56, -57, -60, -59, -61, 
          -61, -61, -61, -61, -60, -60, -56, -57, -58, -59, -60, 
          -62, -63, -63, -66, -67, -67, -67, -56, -57, -57, -57, 
          -57, -56, -58, -59, -61, -63, -68, -70, -71, -69, -71, 
          -58, -58, -58, -60, -61, -61, -62, -62, -62, -64, -67, 
          -66, -59, -59, -62, -66, -68, -69, -69, -69, -71, -71, 
          -60, -60, -59, -59, -59, -59, -59, -60, -61, -62, -64, 
          -65, -66, -67, -67, -68, -70, -69, -59, -58, -57, -57, 
          -56, -54, -52, -51, -50, -49, -48, -47, -47, -47, -46, 
          -44, -47, -45, -44, -43, -42, -41, -40, -35, -34, -32, 
          -31, -30, -28, -38, -36, -33, -32, -31, -34, -32, -29, 
          -27, -26, -25, -23, -22, -21, -20, -29, -29, -29, -28, 
          -26, -26, -26, -26, -26, -27, -28, -31, -32, -35, -27, 
          -27, -27, -26, -23, -21, -18, -27, -26, -26, -25, -24, 
          -22, -25, -26, -27, -27, -27, -27, -26, -25, -24, -23, 
          -22, -21, -22, -46, -45, -41, -37, -36, -34, -34, -34, 
          -34, -33, -32, -29, -27, -26, -43, -41, -41, -41, -39, 
          -38, -36, -35, -35, -34, -33, -33, -32, -30, -32, -31, 
          -30, -30, -30, -29, -29, -32, -31, -29, -28, -26, -25, 
          -24, -25, -25, -42, -43, -41, -39, -38, -37, -43, -45, 
          -46, -47, -47, -48, -48, -50, -50, -51, -51, -51, -52, 
          -53, -54, -53, -49, -48, -47, -48, -48, -48, -49, -50, 
          -51, -51, -52, -53, -55, -54, -53, -52, -52, -51, -51, 
          -50, -49, -49, -44, -43, -42, -40, -40, -44, -38, -35, 
          -32, -31, -28, -25, -24, -24, -24, -25, -24, -24, -24, 
          -32, -31, -29, -28, -27, -34, -34, -33, -32, -31, -29, 
          -27, -40, -40, -40, -40, -40, -40, -41, -43, -47, -39, 
          -38, -35, -32, -39, -39, -62, -63, -65, -66, -67, -67, 
          -67, -68, -68, -70, -71, -70, -67, -68, -71, -70, -71, 
          -71, -70, -70, -69), W = c(1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 1.27, 
          1.27, 1.27, 1.27, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.7, 1.13, 0.42, 0.42, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.84, 
          0.84, 0.84, 0.84, 0.84, 0.84, 0.84, 0.42, 0.42, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.98, 0.14, 0.14, 0.14, 0.14, 
          0.14, 0.14, 0.14, 0.14, 0.14, 0.98, 0.98, 0.98, 0.42, 
          0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 0.84, 0.98, 
          0.98, 0.98, 0.98, 0.98, 0.98, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.42, 0.42, 0.42, 0.42, 0.42, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.7, 0.7, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.7, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 
          0.56, 0.56, 0.56, 0.56, 0.56, 0.7, 0.56, 0.56, 0.56, 
          0.56, 0.42, 0.42, 0.42, 0.42, 0.42, 0.42, 1.41, 0.7, 
          0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7, 0.7), Parent = c(-1L, 
          1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 
          14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 
          25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
          36L, 26L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 
          47L, 48L, 38L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 
          58L, 59L, 60L, 61L, 62L, 63L, 56L, 65L, 66L, 67L, 68L, 
          69L, 70L, 71L, 72L, 73L, 74L, 75L, 65L, 77L, 78L, 79L, 
          80L, 81L, 82L, 83L, 84L, 85L, 77L, 87L, 88L, 89L, 90L, 
          91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 101L, 
          102L, 103L, 92L, 105L, 106L, 107L, 108L, 109L, 110L, 
          111L, 112L, 113L, 114L, 115L, 116L, 117L, 118L, 119L, 
          117L, 121L, 122L, 123L, 124L, 125L, 126L, 127L, 128L, 
          129L, 130L, 131L, 132L, 127L, 134L, 135L, 136L, 137L, 
          135L, 139L, 140L, 141L, 142L, 143L, 144L, 145L, 146L, 
          147L, 141L, 149L, 150L, 151L, 152L, 153L, 154L, 155L, 
          156L, 157L, 158L, 159L, 160L, 161L, 158L, 163L, 164L, 
          165L, 154L, 167L, 168L, 153L, 170L, 171L, 172L, 173L, 
          174L, 173L, 176L, 177L, 178L, 179L, 180L, 181L, 182L, 
          183L, 184L, 185L, 186L, 187L, 121L, 189L, 190L, 191L, 
          192L, 193L, 194L, 195L, 196L, 195L, 198L, 199L, 200L, 
          201L, 191L, 203L, 204L, 205L, 206L, 207L, 208L, 209L, 
          210L, 211L, 212L, 213L, 214L, 215L, 214L, 217L, 218L, 
          219L, 220L, 221L, 222L, 211L, 224L, 225L, 226L, 227L, 
          228L, 229L, 230L, 231L, 206L, 233L, 234L, 235L, 236L, 
          237L, 234L, 239L, 240L, 241L, 242L, 243L, 244L, 245L, 
          246L, 247L, 248L, 249L, 250L, 251L, 252L, 253L, 243L, 
          255L, 256L, 255L, 258L, 259L, 260L, 261L, 262L, 263L, 
          264L, 265L, 266L, 267L, 264L, 269L, 270L, 271L, 272L, 
          273L, 274L, 275L, 239L, 277L, 278L, 279L, 280L, 281L, 
          280L, 283L, 284L, 285L, 286L, 287L, 288L, 289L, 290L, 
          291L, 292L, 293L, 294L, 285L, 296L, 297L, 298L, 299L, 
          284L, 301L, 302L, 303L, 304L, 305L, 306L, 279L, 308L, 
          309L, 310L, 311L, 312L, 313L, 314L, 315L, 313L, 317L, 
          318L, 319L, 318L, 321L, 90L, 323L, 324L, 325L, 326L, 
          327L, 328L, 329L, 330L, 331L, 332L, 333L, 327L, 335L, 
          336L, 336L, 338L, 339L, 340L, 341L, 342L)), .Names = c("PointNo", 
      "Label", "X", "Y", "Z", "W", "Parent"), class = "data.frame", row.names = c(NA, 
      343L))), .Names = c("NeuronName", "InputFileName", "CreatedAt", 
  "NodeName", "InputFileStat", "InputFileMD5", "NumPoints", "StartPoint", 
  "BranchPoints", "EndPoints", "nTrees", "NumSegs", "SegList", 
  "d"), class = c("neuron", "list"))
  
  expect_equal(read.neuron('testdata/neuron/EBT7R.am'), ebt7)
})

test_that("we can read hxskel format neurons",{
  
  Neurites=structure(list(NeuronName = "Neurites", InputFileName = "testdata/neuron/Neurites.am", 
      CreatedAt = structure(1391883985.19409, class = c("POSIXct", 
      "POSIXt")), NodeName = structure("mac1041-14.lmb.internal", .Names = "nodename"), 
      InputFileStat = structure(list(size = 16617, isdir = FALSE, 
          mode = structure(420L, class = "octmode"), mtime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), ctime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), atime = structure(1391851874, class = c("POSIXct", 
          "POSIXt")), uid = 501L, gid = 80L, uname = "jefferis", 
          grname = "admin"), .Names = c("size", "isdir", "mode", 
      "mtime", "ctime", "atime", "uid", "gid", "uname", "grname"
      ), row.names = "testdata/neuron/Neurites.am", class = "data.frame"), 
      InputFileMD5 = structure("11c837b13f2f0814fffaf94a58912e16", .Names = "testdata/neuron/Neurites.am"), 
      NumPoints = 291L, StartPoint = 1L, BranchPoints = c(98L, 
      256L, 272L), EndPoints = c(1L, 54L, 202L, 257L, 274L), nTrees = 1, 
      NumSegs = 7L, SegList = structure(list(c(1L, 3L, 4L, 5L, 
      6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 
      19L, 20L, 21L, 22L, 2L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 
      31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 
      43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 23L, 
      55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 
      67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 
      79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 
      91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L), c(98L, 54L), c(98L, 
      100L, 101L, 102L, 103L, 104L, 105L, 106L, 107L, 108L, 109L, 
      110L, 111L, 112L, 113L, 114L, 115L, 116L, 117L, 118L, 119L, 
      120L, 121L, 122L, 123L, 124L, 125L, 126L, 127L, 128L, 129L, 
      130L, 131L, 132L, 133L, 134L, 135L, 136L, 137L, 138L, 139L, 
      140L, 141L, 142L, 143L, 144L, 145L, 146L, 147L, 99L, 149L, 
      150L, 151L, 152L, 153L, 154L, 155L, 156L, 157L, 158L, 159L, 
      160L, 161L, 162L, 163L, 164L, 165L, 166L, 167L, 168L, 169L, 
      170L, 171L, 172L, 173L, 174L, 175L, 176L, 177L, 178L, 179L, 
      180L, 181L, 182L, 183L, 184L, 185L, 186L, 187L, 188L, 189L, 
      190L, 191L, 192L, 193L, 194L, 195L, 196L, 197L, 198L, 199L, 
      200L, 201L, 148L, 203L, 204L, 205L, 206L, 207L, 208L, 209L, 
      210L, 211L, 212L, 213L, 214L, 215L, 216L, 217L, 218L, 219L, 
      220L, 221L, 222L, 223L, 224L, 225L, 226L, 227L, 228L, 229L, 
      230L, 231L, 232L, 233L, 234L, 235L, 236L, 237L, 238L, 239L, 
      240L, 241L, 242L, 243L, 244L, 245L, 246L, 247L, 248L, 249L, 
      250L, 251L, 252L, 253L, 254L, 255L, 256L), c(256L, 202L), 
          c(256L, 258L, 259L, 260L, 261L, 262L, 263L, 264L, 265L, 
          266L, 267L, 268L, 269L, 270L, 271L, 272L), c(272L, 273L, 
          257L), c(272L, 275L, 276L, 277L, 278L, 279L, 280L, 281L, 
          282L, 283L, 284L, 285L, 286L, 287L, 288L, 289L, 290L, 
          291L, 274L)), class = c("seglist", "list")), d = structure(list(
          PointNo = 1:291, Label = c(2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
          2L, 2L, 2L, 2L), X = c(374.888, 367.069, 374.368, 373.722, 
          373.54, 373.06, 372.566, 372.127, 371.78, 371.355, 371.035, 
          370.838, 370.688, 370.279, 370.065, 369.657, 369.463, 
          369.148, 368.773, 368.404, 367.915, 367.46, 353.928, 
          366.653, 366.212, 365.852, 365.353, 364.904, 364.445, 
          363.922, 363.397, 362.923, 362.431, 361.949, 361.538, 
          361.045, 360.716, 360.307, 359.857, 359.463, 359.048, 
          358.636, 358.247, 357.909, 357.534, 357.09, 356.569, 
          356.077, 355.622, 355.213, 354.876, 354.556, 354.252, 
          333.369, 353.573, 353.202, 352.765, 352.335, 351.875, 
          351.421, 350.954, 350.503, 350.076, 349.699, 349.23, 
          348.823, 348.355, 347.894, 347.434, 346.964, 346.519, 
          346.088, 345.621, 345.191, 344.733, 344.313, 343.89, 
          343.439, 342.975, 342.531, 342.051, 341.585, 341.088, 
          340.637, 340.216, 339.763, 339.265, 338.77, 338.271, 
          337.779, 337.29, 336.787, 336.276, 335.77, 335.291, 334.826, 
          334.337, 333.855, 314.925, 333.431, 332.992, 332.523, 
          332.064, 331.566, 331.112, 330.672, 330.289, 329.891, 
          329.427, 328.943, 328.547, 328.139, 327.73, 327.263, 
          326.831, 326.391, 325.979, 325.62, 325.355, 324.995, 
          324.667, 324.359, 324.053, 323.735, 323.446, 323.087, 
          322.742, 322.442, 322.189, 321.906, 321.57, 321.237, 
          320.855, 320.465, 320.063, 319.7, 319.292, 318.879, 318.48, 
          318.082, 317.704, 317.319, 316.905, 316.475, 316.045, 
          315.621, 315.262, 293.393, 314.521, 314.097, 313.667, 
          313.274, 312.86, 312.426, 311.999, 311.585, 311.14, 310.716, 
          310.291, 309.868, 309.473, 309.108, 308.732, 308.352, 
          307.986, 307.601, 307.212, 306.839, 306.433, 306.074, 
          305.695, 305.344, 305.029, 304.767, 304.485, 304.322, 
          303.999, 303.742, 303.387, 303.024, 302.64, 302.239, 
          301.846, 301.347, 300.981, 300.541, 300.122, 299.689, 
          299.236, 298.752, 298.364, 297.906, 297.534, 297.064, 
          296.574, 296.167, 295.678, 295.294, 294.802, 294.327, 
          293.862, 275.121, 292.979, 292.551, 292.082, 291.67, 
          291.281, 290.922, 290.559, 290.221, 289.845, 289.478, 
          289.128, 288.804, 288.504, 288.205, 287.898, 287.608, 
          287.347, 287.088, 286.884, 286.679, 286.425, 286.247, 
          286.058, 285.855, 285.57, 285.272, 284.972, 284.721, 
          284.576, 284.46, 284.221, 283.92, 283.517, 283.187, 282.757, 
          282.293, 281.837, 281.371, 280.97, 280.587, 280.223, 
          279.848, 279.444, 279.015, 278.557, 278.093, 277.641, 
          277.164, 276.729, 276.318, 275.959, 275.645, 275.467, 
          275.273, 268.316, 274.985, 274.684, 274.324, 273.95, 
          273.466, 273.011, 272.56, 272.122, 271.703, 271.273, 
          270.948, 270.647, 270.339, 269.921, 269.4, 268.847, 269.844, 
          269.25, 269.297, 269.457, 269.693, 269.842, 269.914, 
          269.975, 269.998, 270.004, 270.004, 270.016, 270.076, 
          270.148, 270.164, 270.134, 270.053, 269.952), Y = c(127.498, 
          134.529, 127.91, 128.142, 128.732, 129.072, 129.291, 
          129.574, 129.973, 130.268, 130.671, 131.121, 131.576, 
          131.884, 132.313, 132.596, 133.006, 133.344, 133.633, 
          133.918, 134.038, 134.25, 141.004, 134.77, 134.832, 135.048, 
          135.129, 135.284, 135.427, 135.327, 135.326, 135.565, 
          135.806, 136.055, 136.402, 136.576, 136.97, 137.27, 137.475, 
          137.761, 137.995, 138.223, 138.474, 138.76, 139.007, 
          139.161, 139.258, 139.454, 139.667, 139.9, 140.188, 140.466, 
          140.748, 144.76, 141.07, 141.079, 141.028, 141.074, 141.077, 
          141.207, 141.391, 141.627, 141.889, 142.213, 142.355, 
          142.632, 142.791, 142.98, 143.181, 143.362, 143.584, 
          143.825, 143.991, 144.222, 144.314, 144.336, 144.445, 
          144.598, 144.775, 144.973, 145.076, 145.205, 145.151, 
          145.128, 145.099, 145.166, 145.118, 145.114, 145.121, 
          145.163, 145.167, 145.122, 145.106, 145.117, 144.984, 
          144.832, 144.796, 144.78, 132.196, 144.586, 144.405, 
          144.287, 144.112, 144.067, 143.929, 143.814, 143.593, 
          143.394, 143.345, 143.285, 143.041, 142.826, 142.584, 
          142.413, 142.206, 142.019, 141.737, 141.398, 140.983, 
          140.651, 140.291, 139.924, 139.559, 139.195, 138.793, 
          138.459, 138.112, 137.718, 137.289, 136.889, 136.541, 
          136.192, 135.892, 135.592, 135.299, 134.955, 134.669, 
          134.383, 134.109, 133.858, 133.599, 133.35, 133.113, 
          132.926, 132.761, 132.623, 132.417, 118.666, 132.05, 
          131.875, 131.696, 131.497, 131.345, 131.181, 131.079, 
          130.955, 130.861, 130.727, 130.591, 130.372, 130.103, 
          129.799, 129.519, 129.29, 129.051, 128.823, 128.543, 
          128.244, 128.004, 127.744, 127.441, 127.107, 126.75, 
          126.371, 126.005, 125.563, 125.197, 124.767, 124.393, 
          124.028, 123.677, 123.327, 122.961, 122.78, 122.412, 
          122.141, 121.865, 121.648, 121.473, 121.329, 121.03, 
          120.839, 120.505, 120.305, 120.158, 119.851, 119.705, 
          119.382, 119.272, 119.094, 118.883, 104.161, 118.403, 
          118.16, 118.009, 117.756, 117.471, 117.145, 116.817, 
          116.469, 116.167, 115.855, 115.531, 115.195, 114.832, 
          114.473, 114.196, 113.976, 113.711, 113.451, 113.188, 
          112.888, 112.588, 112.217, 111.876, 111.55, 111.339, 
          111.122, 110.939, 110.842, 110.675, 110.421, 110.144, 
          109.787, 109.507, 109.183, 109.025, 108.881, 108.722, 
          108.559, 108.269, 107.953, 107.623, 107.325, 107.117, 
          107.037, 106.931, 106.762, 106.549, 106.4, 106.154, 105.875, 
          105.555, 105.238, 104.867, 104.524, 102.149, 104.142, 
          103.75, 103.42, 103.083, 102.868, 102.637, 102.515, 102.415, 
          102.263, 102.174, 102.22, 102.307, 102.279, 102.152, 
          102.093, 102.156, 107.951, 102.476, 102.893, 103.305, 
          103.706, 104.127, 104.53, 104.874, 105.156, 105.423, 
          105.705, 105.992, 106.278, 106.555, 106.817, 107.076, 
          107.351, 107.644), Z = c(61.9981, 64.3031, 62.0689, 62.0696, 
          62.2841, 62.2736, 62.1473, 62.0312, 62.0649, 62.0972, 
          62.1912, 62.3661, 62.5525, 62.6129, 62.823, 62.9742, 
          63.2231, 63.461, 63.6682, 63.8911, 64.0093, 64.1344, 
          66.7161, 64.4829, 64.7311, 65.0288, 65.1708, 65.3869, 
          65.6093, 65.6313, 65.5564, 65.5039, 65.5628, 65.5888, 
          65.6379, 65.5624, 65.5866, 65.5547, 65.4929, 65.5247, 
          65.5698, 65.6666, 65.8168, 66.0407, 66.2888, 66.5425, 
          66.6698, 66.7291, 66.7337, 66.7142, 66.7145, 66.7104, 
          66.7198, 65.0324, 67.0547, 67.3831, 67.6107, 67.8564, 
          68.06, 68.2439, 68.3291, 68.3076, 68.2468, 68.2, 68.1154, 
          68.0588, 68.0102, 67.9724, 68.0149, 67.9788, 67.8952, 
          67.7923, 67.724, 67.6134, 67.4358, 67.1662, 66.9274, 
          66.7869, 66.7589, 66.6408, 66.5193, 66.3527, 66.2505, 
          66.0244, 65.7299, 65.4971, 65.3895, 65.2693, 65.1819, 
          65.0667, 64.923, 64.8429, 64.8306, 64.8293, 64.898, 64.9695, 
          65.0019, 65.0381, 58.3124, 64.8468, 64.6799, 64.5215, 
          64.3933, 64.3027, 64.1217, 63.9013, 63.6562, 63.4155, 
          63.2212, 63.095, 62.9015, 62.704, 62.5429, 62.467, 62.3134, 
          62.1545, 62.0728, 61.9662, 61.8454, 61.7214, 61.589, 
          61.4366, 61.2806, 61.1532, 61.087, 60.9767, 60.8783, 
          60.8144, 60.8091, 60.7345, 60.6313, 60.5315, 60.4507, 
          60.4005, 60.3706, 60.3386, 60.2805, 60.2323, 60.0883, 
          59.9093, 59.7103, 59.524, 59.4076, 59.2722, 59.0982, 
          58.881, 58.6044, 53.0065, 58.0661, 57.8822, 57.7212, 
          57.5049, 57.2925, 57.1396, 56.9202, 56.6813, 56.4743, 
          56.2354, 55.991, 55.8031, 55.6306, 55.4725, 55.3218, 
          55.1169, 54.8904, 54.6747, 54.5429, 54.4207, 54.2754, 
          54.0578, 53.9671, 53.8641, 53.7235, 53.5268, 53.3172, 
          53.1101, 52.9368, 52.7875, 52.6906, 52.5853, 52.4909, 
          52.4995, 52.5359, 52.5537, 52.5622, 52.6043, 52.728, 
          52.881, 53.0148, 53.0332, 52.8953, 52.7735, 52.6787, 
          52.6789, 52.644, 52.6706, 52.6549, 52.7715, 52.8828, 
          52.9455, 52.9852, 44.6427, 52.9415, 52.8796, 52.816, 
          52.706, 52.5988, 52.5053, 52.4356, 52.3407, 52.2378, 
          52.1263, 51.9953, 51.8267, 51.6621, 51.4764, 51.1894, 
          50.8444, 50.5126, 50.18, 49.8177, 49.4918, 49.2072, 48.95, 
          48.6568, 48.3524, 48.0093, 47.6759, 47.3171, 46.8885, 
          46.4296, 45.996, 45.6238, 45.3828, 45.1923, 44.9475, 
          44.7043, 44.549, 44.4061, 44.3211, 44.2644, 44.2683, 
          44.2176, 44.0996, 43.9066, 43.6726, 43.5092, 43.4422, 
          43.4181, 43.3891, 43.3677, 43.4146, 43.5467, 43.7659, 
          44.042, 44.342, 40.2316, 44.1425, 43.9771, 43.7827, 43.6176, 
          43.5734, 43.4302, 43.1812, 42.8989, 42.608, 42.3042, 
          41.8753, 41.4239, 40.9543, 40.5827, 40.3432, 40.3135, 
          33.7535, 40.0501, 39.7702, 39.528, 39.3275, 39.0805, 
          38.7714, 38.3951, 37.9694, 37.5357, 37.1134, 36.6941, 
          36.2787, 35.8591, 35.4238, 34.985, 34.5605, 34.1519), 
          W = c(1.3882, 2.7819, 1.5615, 1.5824, 1.8039, 1.9936, 
          2.2251, 2.5169, 2.6649, 2.6124, 2.4982, 2.3369, 2.1493, 
          2.1147, 2.0964, 2.1652, 2.1859, 2.2724, 2.4049, 2.4828, 
          2.5644, 2.6883, 4.0784, 2.7723, 2.7734, 2.7565, 2.6667, 
          2.4426, 2.3305, 2.1823, 1.9618, 1.9482, 1.7556, 1.6962, 
          1.7556, 1.6805, 1.7182, 1.6916, 1.6257, 1.6843, 1.7985, 
          1.9919, 2.2562, 2.6641, 3.1971, 3.6209, 3.9178, 4.2101, 
          4.428, 4.5577, 4.5461, 4.4357, 4.2392, 4.5268, 3.8395, 
          3.8834, 4.072, 4.1348, 4.1366, 4.0006, 3.8899, 3.8102, 
          3.7388, 3.7013, 3.6252, 3.5688, 3.4873, 3.4683, 3.5046, 
          3.6658, 3.9193, 4.13, 4.2278, 4.1145, 4.0719, 4.1144, 
          4.2006, 4.2801, 4.2713, 4.1837, 4.0027, 3.8537, 3.7887, 
          3.7537, 3.8005, 3.8964, 4.3029, 4.5979, 4.7156, 4.5342, 
          4.4095, 4.3308, 4.3359, 4.3579, 4.3955, 4.4551, 4.5105, 
          4.52, 5.5757, 4.3702, 4.1605, 4.0272, 3.8673, 3.8175, 
          3.759, 3.6974, 3.6828, 3.695, 3.6597, 3.6224, 3.6337, 
          3.7271, 3.8679, 4.1209, 4.4256, 4.6062, 4.7647, 4.6636, 
          4.5291, 4.2886, 4.23, 4.2487, 4.33, 4.4368, 4.8105, 4.984, 
          5.1266, 5.2741, 5.5003, 5.6065, 5.6154, 5.6269, 5.5736, 
          5.557, 5.5537, 5.6078, 5.6999, 5.8844, 5.9727, 5.9657, 
          5.9697, 5.9663, 5.9181, 5.8303, 5.7631, 5.596, 5.5707, 
          3.6889, 5.5117, 5.423, 5.337, 5.4996, 5.584, 5.7197, 
          5.6598, 5.8183, 5.9847, 6.2232, 6.43, 6.6574, 6.6474, 
          6.3972, 6.0178, 5.6352, 5.3175, 5.102, 5.0389, 4.9197, 
          4.7698, 4.5178, 4.4641, 4.4854, 4.5123, 4.5296, 4.5329, 
          4.4523, 4.3745, 4.3218, 4.1692, 4.0811, 3.9654, 3.9957, 
          4.0861, 4.1231, 4.1128, 4.0458, 3.9995, 3.8927, 3.7398, 
          3.6189, 3.3257, 3.1791, 3.1781, 3.2329, 3.1418, 3.1643, 
          3.14, 3.2668, 3.4014, 3.5382, 3.6131, 2.862, 3.7155, 
          3.7428, 3.7716, 3.8027, 3.8367, 3.8737, 3.9134, 3.9556, 
          4.0001, 4.0473, 4.098, 4.152, 4.2081, 4.2633, 4.3128, 
          4.3503, 4.3692, 4.3635, 4.3298, 4.2673, 4.1786, 4.069, 
          3.9459, 3.8181, 3.6953, 3.5871, 3.5015, 3.4437, 3.414, 
          3.4079, 3.4162, 3.4276, 3.4307, 3.4172, 3.3828, 3.3278, 
          3.2565, 3.1755, 3.0926, 3.0151, 2.9486, 2.8965, 2.8602, 
          2.8387, 2.8296, 2.8291, 2.8334, 2.839, 2.8441, 2.848, 
          2.8513, 2.8546, 2.8581, 2.8609, 4.7105, 3.0057, 3.1443, 
          3.2729, 3.3917, 3.5048, 3.6179, 3.7365, 3.8635, 3.9988, 
          4.1389, 4.2774, 4.4061, 4.5169, 4.6037, 4.6643, 4.6992, 
          3.9769, 4.6679, 4.662, 4.6388, 4.5942, 4.5276, 4.4423, 
          4.3443, 4.2416, 4.143, 4.0569, 3.9905, 3.9477, 3.9285, 
          3.9288, 3.9415, 3.9583, 3.9718), Parent = c(-1L, 22L, 
          1L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
          15L, 16L, 17L, 18L, 19L, 20L, 21L, 53L, 2L, 24L, 25L, 
          26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 
          37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
          48L, 49L, 50L, 51L, 52L, 98L, 23L, 55L, 56L, 57L, 58L, 
          59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 
          70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
          81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 
          92L, 93L, 94L, 95L, 96L, 97L, 147L, 98L, 100L, 101L, 
          102L, 103L, 104L, 105L, 106L, 107L, 108L, 109L, 110L, 
          111L, 112L, 113L, 114L, 115L, 116L, 117L, 118L, 119L, 
          120L, 121L, 122L, 123L, 124L, 125L, 126L, 127L, 128L, 
          129L, 130L, 131L, 132L, 133L, 134L, 135L, 136L, 137L, 
          138L, 139L, 140L, 141L, 142L, 143L, 144L, 145L, 146L, 
          201L, 99L, 149L, 150L, 151L, 152L, 153L, 154L, 155L, 
          156L, 157L, 158L, 159L, 160L, 161L, 162L, 163L, 164L, 
          165L, 166L, 167L, 168L, 169L, 170L, 171L, 172L, 173L, 
          174L, 175L, 176L, 177L, 178L, 179L, 180L, 181L, 182L, 
          183L, 184L, 185L, 186L, 187L, 188L, 189L, 190L, 191L, 
          192L, 193L, 194L, 195L, 196L, 197L, 198L, 199L, 200L, 
          256L, 148L, 203L, 204L, 205L, 206L, 207L, 208L, 209L, 
          210L, 211L, 212L, 213L, 214L, 215L, 216L, 217L, 218L, 
          219L, 220L, 221L, 222L, 223L, 224L, 225L, 226L, 227L, 
          228L, 229L, 230L, 231L, 232L, 233L, 234L, 235L, 236L, 
          237L, 238L, 239L, 240L, 241L, 242L, 243L, 244L, 245L, 
          246L, 247L, 248L, 249L, 250L, 251L, 252L, 253L, 254L, 
          255L, 273L, 256L, 258L, 259L, 260L, 261L, 262L, 263L, 
          264L, 265L, 266L, 267L, 268L, 269L, 270L, 271L, 272L, 
          291L, 272L, 275L, 276L, 277L, 278L, 279L, 280L, 281L, 
          282L, 283L, 284L, 285L, 286L, 287L, 288L, 289L, 290L)), .Names = c("PointNo", 
      "Label", "X", "Y", "Z", "W", "Parent"), class = "data.frame", row.names = c(NA, 
      291L))), .Names = c("NeuronName", "InputFileName", "CreatedAt", 
  "NodeName", "InputFileStat", "InputFileMD5", "NumPoints", "StartPoint", 
  "BranchPoints", "EndPoints", "nTrees", "NumSegs", "SegList", 
  "d"), class = c("neuron", "list"))
  # NB this neuron does not have an origin set
  expect_warning(n<-read.neuron('testdata/neuron/Neurites.am'), 
                 regexp = 'No valid origin found')
  expect_is(n,'neuron')
  g1<-as.ngraph(Neurites)
  g2<-as.ngraph(n)
  expect_true(igraph::graph.isomorphic(g1,g2))
  expect_equal(n, Neurites)
  tmpfile=tempfile(fileext='.wurgle')
  on.exit(unlink(tmpfile))
  file.copy('testdata/neuron/Neurites.am',tmpfile)
  
  expect_equal(suppressWarnings(read.neuron(tmpfile)), n,
               fieldsToExclude='NeuronName')
})

test_that("we can identify amira hxskel neurons",{
  # hxlineset neuron
  expect_false(is.hxskel('testdata/neuron/EBT7R.am'))
  # swc neuron
  expect_false(is.hxskel('testdata/neuron/EBT7R.CNG.swc'))
  # hxskel neuron
  expect_true(is.hxskel('testdata/neuron/Neurites.am'))
  # hxskel
  p='testdata/neuron/Neurites.am'
  expect_true(is.hxskel(p,bytes=readBin(p,what=raw(),n=80)))
})

test_that("reading identical neuron in 2 amira formats and 3 encodings works",{
  expect_warning(l<-read.neuron("testdata/neuron/testneuron_lineset.am"),
                 regexp = 'Data section 3 .* specifies radius')
  expect_is(l,'neuron')
  expect_equal(l,read.neuron("testdata/neuron/testneuron_am3d.am"),
               fieldsToExclude='NeuronName')
  expect_equal(l,read.neuron("testdata/neuron/testneuron_am3d_ascii.am.gz"),
               fieldsToExclude='NeuronName')
  # FIXME see https://github.com/jefferis/nat/issues/14
  expect_error(suppressWarnings(read.neuron("testdata/neuron/testneuron_am3d.am.gz")))
})

test_that("we can identify amira hxlineset neurons",{
  # hxlineset neuron
  expect_true(is.hxlineset('testdata/neuron/EBT7R.am'))
  # swc neuron
  expect_false(is.hxlineset('testdata/neuron/EBT7R.CNG.swc'))
  # hxskel neuron
  expect_false(is.hxlineset('testdata/neuron/Neurites.am'))
  # hxlineset via byte array
  p='testdata/neuron/EBT7R.am'
  expect_true(is.hxlineset(p,bytes=readBin(p,what=raw(),n=80)))
  q='testdata/neuron/Neurites.am'
  expect_false(is.hxlineset(q,bytes=readBin(p,what=raw(),n=80)))
  # gzipped direct
  expect_true(is.hxlineset('testdata/neuron/testneuron_fclineset.am.gz'))
  # gzipped via byte array
  # nb this relies on decompression which is looked after by getformatfuns
  r='testdata/neuron/testneuron_fclineset.am.gz'
  gzf=gzfile(r, open='rb')
  on.exit(close(gzf))
  expect_true(is.hxlineset(r,bytes=readBin(gzf,what=raw(),n=80)))
})

test_that("read a flycircuit lineset neuron w/o radius info",{
  f="testdata/neuron/testneuron_fclineset.am.gz"
  expect_warning(n<-read.neuron(f), regexp = 'No width data')
  expect_is(n, 'neuron')
})

context("write neurons")

test_that("write neuron/dotprops to rds file",{
  x=kcs20[[1]]
  td=tempfile()
  dir.create(td)
  on.exit(unlink(td,recursive=TRUE))
  
  expect_equal(f<-write.neuron(x, dir=td), 
               file.path(td,'FruMARCM-M001205_seg002_03.rds'))
  # can't overwrite get a warning and an NA back
  expect_warning(f2<-write.neuron(x, f))
  expect_true(is.na(f2))
  # can overwrite with force
  expect_equal(write.neuron(x, f, Force=TRUE), f)
  unlink(f)
  
  expect_equal(write.neuron(x, dir=td, ext='.RDS'),
               file.path(td,'FruMARCM-M001205_seg002_03.RDS'))
  
  y=Cell07PNs[[1]]
  expect_error(write.neuron(y, dir=td),'Ambiguous file format')
  expect_equal(write.neuron(y, dir=td, format='rds', ext='.RDS'),
               file.path(td,'EBH11R.RDS'))
  expect_equal(write.neuron(y, dir=td, format='rds', ext='_skel.rds'),
               file.path(td,'EBH11R_skel.rds'))
  
})

test_that("write neuron to swc file",{
  y=Cell07PNs[[1]]
  td=tempfile()
  dir.create(td)
  on.exit(unlink(td,recursive=TRUE))
  
  expect_equal(write.neuron(y, dir=td, ext='.swc'),
               file.path(td,'EBH11R.swc'))
  expect_equal(f<-write.neuron(y, dir=td, format='swc', ext='_skel.swc'),
               file.path(td,'EBH11R_skel.swc'))
  expect_equal(read.neuron(f),y,fieldsToExclude='NeuronName')
})

test_that("write neuron to amira hxskel file",{
  y=Cell07PNs[[1]]
  td=tempfile()
  dir.create(td)
  on.exit(unlink(td,recursive=TRUE))
  
  expect_equal(f<-write.neuron(y, dir=td, format='hxskel'),
               file.path(td,'EBH11R.am'))
  expect_equal(read.neuron(f),y,fieldsToExclude='NeuronName')
})

test_that("write neuron to amira hxlineset file",{
  y=Cell07PNs[[1]]
  td=tempfile()
  dir.create(td)
  on.exit(unlink(td,recursive=TRUE))
  
  expect_equal(f<-write.neuron(y, dir=td, format='hxlineset'),
               file.path(td,'EBH11R.am'))
  expect_equal(read.neuron(f),y,fieldsToExclude='NeuronName')
})

test_that("write neuron to unknown format",{
  expect_error(write.neuron(Cell07PNs[[1]], dir=td, format='rhubarb'))
})

test_that("write.neurons",{
  td=tempfile()
  dir.create(td)
  on.exit(unlink(td,recursive=TRUE))
  neurons_to_write=subset(Cell07PNs,Scored.By%in%c("ACH","CJP"),rval='names')
  expect_is(written_files<-write.neurons(Cell07PNs, dir=td,
                INDICES=neurons_to_write,
                subdir=file.path(Glomerulus),format='swc'),'character')
  files_found=dir(td,recursive=T,pattern='swc$')
  expect_true(all(basename(written_files)%in%basename(files_found)))
})
