% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nowcast.R
\name{nowcast}
\alias{nowcast}
\title{Nowcasting of a quarterly time series using a dynamic factor model.}
\usage{
nowcast(formula, data, r = NULL, q = NULL, p = NULL, method = "EM",
  blocks = NULL, frequency = NULL)
}
\arguments{
\item{formula}{An object of class "formula": a symbolic description of the model to be fitted.}

\item{data}{A monthly time series matrix (\code{mts}) of stationary variables.}

\item{r}{number of commom factors.}

\item{q}{Dynamic rank. Number of error terms.}

\item{p}{AR order of factor model.}

\item{method}{There are three options: \code{"2s"} (two stages without factors aggregation as in Giannone et al. 2008); \code{"2s_agg"} (two stages with factors aggregation); \code{"EM"} (Expected Maximization as in Bańbura et al. 2011).}

\item{blocks}{a matrix that defines the variables loaded into the factors.}

\item{frequency}{A vector of integers indicating the frequency of the variables: 4 for quarterly, 12 for monthly.}
}
\value{
A \code{list} containing two elements:
\item{yfcst}{the original \code{y} series and its in-sample and out-of-sample estimations.}
\item{reg}{regression model between \code{y} and the estimated factors. Not available for EM method.}
\item{factors}{the estimated factors and DFM model coefficients.}
\item{xfcst}{the original regressors and their out-of-sample estimations.}
}
\description{
Estimate nowcasting and forecasting models for quarterly or monthly time series. For more details read the Vignettes.
}
\examples{
\dontrun{
### Method 2s (Using the Mariano and Murasawa aggregation method on the variables)
data(USGDP)
gdp_position <- which(colnames(USGDP$base) == "RGDPGR")
base <- Bpanel(base = USGDP$base[,-gdp_position],
               trans = USGDP$legend$Transformation[-gdp_position],
               aggregate = TRUE)
data <- cbind(USGDP$base[,"RGDPGR"], base)
colnames(data) <- c("RGDPGR", colnames(base))
frequency <- c(4, rep(12, ncol(data) -1))
now2s <- nowcast(formula = RGDPGR ~ ., data = data, r = 2, p = 2, q = 2,
                 method = '2s', frequency = frequency)


### Method 2s_agg (Using the Mariano and Murasawa aggregation method on the factors)
data <- Bpanel(base = USGDP$base,
               trans = USGDP$legend$Transformation,
               aggregate = FALSE)
frequency <- c(rep(12, ncol(data) -1), 4)
now2s_agg <- nowcast(formula = RGDPGR ~ ., data = data, r = 2, p = 2, q = 2, 
                     method = '2s_agg', frequency = frequency)


### Method EM
# Replication of the NY FED nowcast
data(NYFED)
base <- NYFED$base
blocks <- NYFED$blocks$blocks
trans <- NYFED$legend$Transformation
frequency <- NYFED$legend$Frequency
data <- Bpanel(base = base, trans = trans, NA.replace = F, na.prop = 1)
nowEM <- nowcast(formula = GDPC1 ~ ., data = data, r = 1, p = 1, 
                 method = "EM", blocks = blocks, frequency = frequency)

}
}
\references{
Giannone, D., Reichlin, L., & Small, D. (2008). Nowcasting: The real-time informational content of macroeconomic data. Journal of Monetary Economics, 55(4), 665-676.<doi:10.1016/j.jmoneco.2008.05.010>

Bańbura, M., & Rünstler, G. (2011). A look into the factor model black box: publication lags and the role of hard and soft data in forecasting GDP. International Journal of Forecasting, 27(2), 333-346. <doi:10.1016/j.ijforecast.2010.01.011>

Bańbura M., Giannone, D. & Reichlin, L. (2011). Nowcasting, in Michael P. Clements and David F. Hendry, editors, Oxford Handbook on Economic Forecasting, pages 193-224, January 2011. <doi:10.1093/oxfordhb/9780195398649.001.0001>
}
\seealso{
\code{\link[nowcasting]{base_extraction}}
}
