% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_response_rates.R
\name{calculate_response_rates}
\alias{calculate_response_rates}
\title{Calculate Response Rates}
\usage{
calculate_response_rates(
  data,
  status,
  status_codes = c("ER", "EN", "IE", "UE"),
  weights,
  rr_formula = "RR3",
  elig_method = "CASRO-subgroup",
  e = NULL
)
}
\arguments{
\item{data}{A data frame containing the selected sample, one row per case.}

\item{status}{A character string giving the name of the variable representing response/eligibility status.
The \code{status} variable should have at most four categories,
representing eligible respondents (ER), eligible nonrespondents (EN),
known ineligible cases (IE), and cases whose eligibility is unknown (UE).}

\item{status_codes}{A named vector, with four entries named 'ER', 'EN', 'IE', and 'UE'.
\code{status_codes} indicates how the values of the \code{status} variable are to be interpreted.}

\item{weights}{(Optional) A character string giving the name of a variable representing weights in the data
to use for calculating weighted response rates}

\item{rr_formula}{A character vector including any of the following: 'RR1', 'RR3', and 'RR5'. \cr
These are the names of formulas defined by AAPOR. See the \emph{Formulas} section below for formulas.}

\item{elig_method}{If \code{rr_formula='RR3'}, this specifies how to estimate
an eligibility rate for cases with unknown eligibility. Must be one of the following: \cr
\cr
\code{'CASRO-overall'} \cr
Estimates an eligibility rate using the overall sample.
If response rates are calculated for subgroups, the single overall sample estimate
will be used as the estimated eligibility rate for subgroups as well. \cr
\cr
\code{'CASRO-subgroup'} \cr
Estimates eligibility rates separately for each subgroup. \cr
\cr
\code{'specified'} \cr
With this option, a numeric value is supplied by the user to the parameter \code{e}. \cr
\cr
For \code{elig_method='CASRO-overall'} or \code{elig_method='CASRO-subgroup'},
the eligibility rate is estimated as \eqn{(ER)/(ER + NR + IE)}.}

\item{e}{(Required if \code{elig_method='specified'}). A numeric value between 0 and 1 specifying the estimated eligibility rate for cases with unknown eligibility.
A character string giving the name of a numeric variable may also be supplied; in that case, the eligibility rate must be constant for all cases in a subgroup.}
}
\value{
Output consists of a data frame giving weighted and unweighted response rates. The following columns may be included, depending on the arguments supplied:
\itemize{
\item{\code{RR1_Unweighted}}
\item{\code{RR1_Weighted}}
\item{\code{RR3_Unweighted}}
\item{\code{RR3_Weighted}}
\item{\code{RR5_Unweighted}}
\item{\code{RR5_Weighted}}
\item{\code{n}: Total sample size}
\item{\code{Nhat}: Sum of weights for the total sample}
\item{\code{n_ER}: Number of eligible respondents}
\item{\code{Nhat_ER}: Sum of weights for eligible respondents}
\item{\code{n_EN}: Number of eligible nonrespondents}
\item{\code{Nhat_EN}: Sum of weights for eligible nonrespondents}
\item{\code{n_IE}: Number of ineligible cases}
\item{\code{Nhat_IE}: Sum of weights for ineligible cases}
\item{\code{n_UE}: Number of cases whose eligibility is unknown}
\item{\code{Nhat_UE}: Sum of weights for cases whose eligibility is unknown}
\item{\code{e_unwtd}: If \emph{RR3} is calculated, the eligibility rate estimate \emph{e} used for the unweighted response rate.}
\item{\code{e_wtd}: If \emph{RR3} is calculated, the eligibility rate estimate \emph{e} used for the weighted response rate.}
}

If the data frame is grouped (i.e. by using \code{df \%>\% group_by(Region)}),
then the output contains one row per subgroup.
}
\description{
Calculates response rates using one of the response rate formulas
defined by AAPOR (American Association of Public Opinion Research).
}
\section{Formulas}{

Denote the sample totals as follows:
\itemize{
\item{\strong{ER}: Total number of eligible respondents}
\item{\strong{EN}: Total number of eligible non-respondents}
\item{\strong{IE}: Total number of ineligible cases}
\item{\strong{UE}: Total number of cases whose eligibility is unknown}
}

For weighted response rates, these totals are calculated using weights.

The response rate formulas are then as follows:

\deqn{RR1 = ER / ( ER + EN + UE )}

RR1 essentially assumes that all cases with unknown eligibility are in fact eligible.

\deqn{RR3 = ER / ( ER + EN + (e * UE) )}

RR3 uses an estimate, \emph{e}, of the eligibility rate among cases with unknown eligibility.

\deqn{RR5 = ER / ( ER + EN )}

RR5 essentially assumes that all cases with unknown eligibility are in fact ineligible. \cr

For \emph{RR3}, an estimate, \code{e}, of the eligibility rate among cases with unknown eligibility must be used.
AAPOR strongly recommends that the basis for the estimate should be explicitly stated and detailed.

The CASRO methods, which might be appropriate for the design, use the formula \eqn{e = 1 - ( IE / (ER + EN + IE) )}.
\itemize{
\item For \code{elig_method='CASRO-overall'}, an estimate is calculated for the overall sample
and this single estimate is used when calculating response rates for subgroups.
\item For \code{elig_method='CASRO-subgroup'}, estimates are calculated separately for each subgroup.
}

Please consult AAPOR's current \emph{Standard Definitions} for in-depth explanations.
}

\examples{
# Load example data
data(involvement_survey_srs, package = "nrba")

involvement_survey_srs[["RESPONSE_STATUS"]] <- sample(1:4, size = 5000, replace = TRUE)

# Calculate overall response rates

involvement_survey_srs \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    weights = "BASE_WEIGHT",
    rr_formula = "RR3",
    elig_method = "CASRO-overall"
  )

# Calculate response rates by subgroup

library(dplyr)

involvement_survey_srs \%>\%
  group_by(STUDENT_RACE, STUDENT_SEX) \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    weights = "BASE_WEIGHT",
    rr_formula = "RR3",
    elig_method = "CASRO-overall"
  )

# Compare alternative approaches for handling of cases with unknown eligiblity

involvement_survey_srs \%>\%
  group_by(STUDENT_RACE) \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    rr_formula = "RR3",
    elig_method = "CASRO-overall"
  )

involvement_survey_srs \%>\%
  group_by(STUDENT_RACE) \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    rr_formula = "RR3",
    elig_method = "CASRO-subgroup"
  )

involvement_survey_srs \%>\%
  group_by(STUDENT_RACE) \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    rr_formula = "RR3",
    elig_method = "specified",
    e = 0.5
  )

involvement_survey_srs \%>\%
  transform(e_by_email = ifelse(PARENT_HAS_EMAIL == "Has Email", 0.75, 0.25)) \%>\%
  group_by(PARENT_HAS_EMAIL) \%>\%
  calculate_response_rates(
    status = "RESPONSE_STATUS",
    status_codes = c("ER" = 1, "EN" = 2, "IE" = 3, "UE" = 4),
    rr_formula = "RR3",
    elig_method = "specified",
    e = "e_by_email"
  )

}
\references{
The American Association for Public Opinion Research. 2016. \emph{Standard Definitions:
Final Dispositions of Case Codes and Outcome Rates for Surveys. 9th edition.} AAPOR.
}
