#' Get or set OpenMPT module controls
#' 
#' Each individual module has its own set of control parameters. Use these functions
#' to obtain or set the state of these parameters.
#' @param mod A tracker module object of class `openmpt`.
#' @param key A `character` string representing a specific control you whish to get or set.
#' Use `control_keys()` to list all available keys.
#' @param ... Ignored
#' @param value A replacement value for the specified control `key`. Check the
#' libopenmpt [documentation](https://buildbot.openmpt.org/builds/latest-unpacked/libopenmpt-docs/docs/classopenmpt_1_1module.html#a4870472969da4d659c5cc394bb1ed245)
#' for the appropriate replacement types and values for each of the `key` values.
#' @returns `control_keys()` returns a `vector` of strings containing all available
#' control keys for `mod`. `control()` returns the value for the specified `key'.
#' In case of an assign operator (`<-`) an updated version of `mod` is
#' returned, where the control key has been set if successful.
#' @examples
#' mod <- demo_mod()
#' control_keys(mod)
#' 
#' ## get a specific control value
#' control(mod, "play.at_end")
#' 
#' ## set a number of control values
#' control(mod, "play.at_end") <- "stop"
#' control(mod, "play.pitch_factor") <- 2
#' control(mod, "load.skip_plugins") <- TRUE
#' control(mod, "dither") <- 2L
#' 
#' ## Show all control settings
#' all_keys <- control_keys(mod)
#' structure(
#'   lapply(all_keys, control, mod = mod),
#'   names = all_keys
#' )
#' @rdname controls
#' @export
control_keys <- function(mod, ...) {
  get_ctls_(mod)
}

#' @rdname controls
#' @export
control <- function(mod, key, ...) {
  ctl_get_(mod, key)
}

#' @rdname controls
#' @export
`control<-` <- function(mod, key, ..., value) {
  ctl_set_(mod, key, value)
}

#' Get or set render parameters for a specific module
#' 
#' Each individual module has its own set of render parameters. Use these functions
#' to obtain or set the state of these parameters.
#' @param mod A tracker module object of class `openmpt`.
#' @param key One of `"MASTERGAIN"`, `"STEREOSEPARATION"`, `"INTERPOLATION"`, or `"VOLUMERAMPING"`.
#' details copied from libopenmpt
#' [documentation](https://lib.openmpt.org/doc/classopenmpt_1_1module.html#ab4ae2823cb180657142f5f1a93cd64aa)
#' \describe{
#'   \item{Master Gain}{
#'     The related value represents a relative gain in milliBel.
#'     The default value is 0. The supported value range is unlimited.
#'   }
#'   \item{Stereo Separation}{
#'     The related value represents the stereo separation generated by the libopenmpt mixer in percent.
#'     The default value is 100. The supported value range is from 0 up to 200.
#'   }
#'   \item{Interpolation Filter}{
#'     The related value represents the interpolation filter length used by the libopenmpt mixer.
#'     The default value is 0, which indicates a recommended default value.
#'     The supported value range is from 0 up to infinity. Values greater than the implementation
#'     limit are clamped to the maximum supported value.
#'     Currently supported values:
#'     \itemize{
#'       \item 0: internal default
#'       \item 1: no interpolation (zero order hold)
#'       \item 2: linear interpolation
#'       \item 4: cubic interpolation
#'       \item 8: windowed sinc with 8 taps
#'     }
#'   }
#'   \item{Volume Ramping Strength}{
#'     The related value represents the amount of volume ramping done by the libopenmpt mixer.
#'     The default value is -1, which indicates a recommended default value.
#'     The meaningful value range is from -1 up to 10.
#'     A value of 0 completely disables volume ramping. This might cause clicks in sound output.
#'     Higher values imply slower/softer volume ramps.
#'   }
#' }
#' @param ... Ignored.
#' @param value An `integer` replacement value for the render parameter selected with `key`
#' @returns Returns the current integer render parameter for the specified
#' `key` and `mod`. In case of an assign operator (`<-`) `mod` with an updated
#' set of render parameters is returned.
#' @examples
#' mod <- demo_mod()
#' 
#' render_param(mod, "STEREOSEPARATION") <- 50
#' render_param(mod, "STEREOSEPARATION")
#' render_param(mod, "MASTERGAIN")
#' render_param(mod, "INTERPOLATION")
#' render_param(mod, "VOLUMERAMPING")
#' @rdname render_param
#' @export
render_param <- function(mod, key, ...) {
  get_render_param_(mod, key)
}

#' @rdname render_param
#' @export
`render_param<-` <- function(mod, key, ..., value) {
  set_render_param_(mod, key, as.integer(value))
}
