% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Pagoda2.R
\name{Pagoda2}
\alias{Pagoda2}
\title{Pagoda2 R6 class}
\description{
The class encompasses gene count matrices, providing methods for normalization, calculating embeddings, and differential expression.
}
\examples{

## ------------------------------------------------
## Method `Pagoda2$new`
## ------------------------------------------------

\donttest{ 
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=500)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object 
p2_object <- Pagoda2$new(counts, log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
}


## ------------------------------------------------
## Method `Pagoda2$adjustVariance`
## ------------------------------------------------

\donttest{
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=500)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object 
p2_object <- Pagoda2$new(counts,log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
## Normalize gene expression variance
p2_object$adjustVariance(plot=TRUE, gam.k=10)
}


## ------------------------------------------------
## Method `Pagoda2$makeKnnGraph`
## ------------------------------------------------

\donttest{
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=300)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object   
p2_object <- Pagoda2$new(counts,log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
## Normalize gene expression variance
p2_object$adjustVariance(plot=TRUE, gam.k=10)
## Generate a kNN graph of cells that will allow us to identify clusters of cells
p2_object$makeKnnGraph(k=20, center=FALSE, distance='L2')
}

}
\author{
Simon Steiger
}
\keyword{internal}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{counts}}{Gene count matrix, normalized on total counts (default=NULL)}

\item{\code{modelType}}{string Model used to normalize count matrices. Only supported values are 'raw', 'plain', and 'linearObs'.
-- 'plain': Normalize by regressing out on the non-zero observations of each gene (default).
-- 'raw': Use the raw count matrices, without normalization. The expression matrix taken "as is" without normalization, although log.scale still applies. 
-- 'linearObs': Fit a linear model of pooled counts across all genes against depth. This approach isn't recommened, as the depth dependency is not completely normalized out.}

\item{\code{clusters}}{Results of clustering (default=list())}

\item{\code{graphs}}{Graph representations of the dataset (default=list())}

\item{\code{reductions}}{Results of reductions, e.g. PCA (default=list())}

\item{\code{embeddings}}{Results of visualization algorithms, t-SNE or largeVis (default=list())}

\item{\code{diffgenes}}{Lists of differentially expressed genes (default=list())}

\item{\code{n.cores}}{number of cores (default=1)}

\item{\code{misc}}{list with additional info (default=list())}

\item{\code{batch}}{Batch factor for the dataset (default=NULL)}

\item{\code{genegraphs}}{Slot to store graphical representations in gene space (i.e. gene kNN graphs) (default=list())}

\item{\code{depth}}{Number of molecules measured per cell (default=NULL)}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-new}{\code{Pagoda2$new()}}
\item \href{#method-setCountMatrix}{\code{Pagoda2$setCountMatrix()}}
\item \href{#method-adjustVariance}{\code{Pagoda2$adjustVariance()}}
\item \href{#method-makeKnnGraph}{\code{Pagoda2$makeKnnGraph()}}
\item \href{#method-getKnnClusters}{\code{Pagoda2$getKnnClusters()}}
\item \href{#method-geneKnnbyPCA}{\code{Pagoda2$geneKnnbyPCA()}}
\item \href{#method-getHierarchicalDiffExpressionAspects}{\code{Pagoda2$getHierarchicalDiffExpressionAspects()}}
\item \href{#method-makeGeneKnnGraph}{\code{Pagoda2$makeGeneKnnGraph()}}
\item \href{#method-getDensityClusters}{\code{Pagoda2$getDensityClusters()}}
\item \href{#method-getDifferentialGenes}{\code{Pagoda2$getDifferentialGenes()}}
\item \href{#method-plotDiffGeneHeatmap}{\code{Pagoda2$plotDiffGeneHeatmap()}}
\item \href{#method-getRefinedLibSizes}{\code{Pagoda2$getRefinedLibSizes()}}
\item \href{#method-plotGeneHeatmap}{\code{Pagoda2$plotGeneHeatmap()}}
\item \href{#method-plotEmbedding}{\code{Pagoda2$plotEmbedding()}}
\item \href{#method-getOdGenes}{\code{Pagoda2$getOdGenes()}}
\item \href{#method-getNormalizedExpressionMatrix}{\code{Pagoda2$getNormalizedExpressionMatrix()}}
\item \href{#method-calculatePcaReduction}{\code{Pagoda2$calculatePcaReduction()}}
\item \href{#method-expandOdGenes}{\code{Pagoda2$expandOdGenes()}}
\item \href{#method-localPcaKnn}{\code{Pagoda2$localPcaKnn()}}
\item \href{#method-testPathwayOverdispersion}{\code{Pagoda2$testPathwayOverdispersion()}}
\item \href{#method-getEmbedding}{\code{Pagoda2$getEmbedding()}}
\item \href{#method-clone}{\code{Pagoda2$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-new"></a>}}
\if{latex}{\out{\hypertarget{method-new}{}}}
\subsection{Method \code{new()}}{
Initialize Pagoda2 class
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$new(
  x,
  modelType = "plain",
  n.cores = parallel::detectCores(logical = FALSE),
  verbose = TRUE,
  min.cells.per.gene = 0,
  trim = round(min.cells.per.gene/2),
  min.transcripts.per.cell = 10,
  batch = NULL,
  lib.sizes = NULL,
  log.scale = TRUE,
  keep.genes = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{x}}{input count matrix}

\item{\code{modelType}}{Model used to normalize count matrices (default='plain'). Only supported values are 'raw', 'plain', and 'linearObs'.}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{min.cells.per.gene}}{integer Minimum number of cells per gene, used to subset counts for coverage (default=0)}

\item{\code{trim}}{numeric Parameter used for winsorizing count data (default=round(min.cells.per.gene/2)). If value>0, will winsorize counts in normalized space in the hopes of getting a more stable depth estimates. If value<=0, ignored.}

\item{\code{min.transcripts.per.cell}}{integer Minimum number of transcripts per cells, used to subset counts for coverage (default=10)}

\item{\code{batch}}{fctor Batch factor for the dataset (default=NULL)}

\item{\code{lib.sizes}}{character vector of library sizes (default=NULL)}

\item{\code{log.scale}}{boolean If TRUE, scale counts by log() (default=TRUE)}

\item{\code{keep.genes}}{list of genes to keep in count matrix after filtering out by coverage but before normalization (default=NULL)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
new Pagoda2 object
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\donttest{ 
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=500)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object 
p2_object <- Pagoda2$new(counts, log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-setCountMatrix"></a>}}
\if{latex}{\out{\hypertarget{method-setCountMatrix}{}}}
\subsection{Method \code{setCountMatrix()}}{
Provide the initial count matrix, and estimate deviance residual matrix (correcting for depth and batch)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$setCountMatrix(
  countMatrix,
  depthScale = 1000,
  min.cells.per.gene = 0,
  trim = round(min.cells.per.gene/2),
  min.transcripts.per.cell = 10,
  lib.sizes = NULL,
  log.scale = FALSE,
  keep.genes = NULL,
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{countMatrix}}{input count matrix}

\item{\code{depthScale}}{numeric Scaling factor for normalizing counts (defaul=1e3). If 'plain', counts are scaled by counts = counts/as.numeric(depth/depthScale).}

\item{\code{min.cells.per.gene}}{integer Minimum number of cells per gene, used to subset counts for coverage (default=0)}

\item{\code{trim}}{numeric Parameter used for winsorizing count data (default=round(min.cells.per.gene/2)). If value>0, will winsorize counts in normalized space in the hopes of getting a more stable depth estimates. If value<=0, ignored.}

\item{\code{min.transcripts.per.cell}}{integer Minimum number of transcripts per cells, used to subset counts for coverage (default=10)}

\item{\code{lib.sizes}}{character vector of library sizes (default=NULL)}

\item{\code{log.scale}}{boolean If TRUE, scale counts by log() (default=TRUE)}

\item{\code{keep.genes}}{list of genes to keep in count matrix after filtering out by coverage but before normalization (default=NULL)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
normalized count matrix (or if modelTye='raw', the unnormalized count matrix)
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-adjustVariance"></a>}}
\if{latex}{\out{\hypertarget{method-adjustVariance}{}}}
\subsection{Method \code{adjustVariance()}}{
Adjust variance of the residual matrix, determine overdispersed sites
This is done to normalize the extent to which genes with (very) different expression magnitudes will contribute to the downstream anlaysis.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$adjustVariance(
  gam.k = 5,
  alpha = 0.05,
  plot = FALSE,
  use.raw.variance = FALSE,
  use.unadjusted.pvals = FALSE,
  do.par = TRUE,
  max.adjusted.variance = 1000,
  min.adjusted.variance = 0.001,
  cells = NULL,
  verbose = TRUE,
  min.gene.cells = 0,
  persist = is.null(cells),
  n.cores = self$n.cores
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{gam.k}}{integer The k used for the generalized additive model 'v ~ s(m, k =gam.k)' (default=5). If gam.k<2, linear regression is used 'lm(v ~ m)'.}

\item{\code{alpha}}{numeric The Type I error probability or the significance level (default=5e-2). This is the criterion used to measure statistical significance, i.e. if the p-value < alpha, then it is statistically significant.}

\item{\code{plot}}{boolean Whether to output the plot (default=FALSE)}

\item{\code{use.raw.variance}}{(default=FALSE). If modelType='raw', then this conditional will be used as TRUE.}

\item{\code{use.unadjusted.pvals}}{boolean Whether to use Benjamini-Hochberg adjusted p-values (default=FALSE).}

\item{\code{do.par}}{boolean Whether to put multiple graphs into a signle plot with par() (default=TRUE)}

\item{\code{max.adjusted.variance}}{numeric Maximum adjusted variance (default=1e3). The gene scale factor is defined as sqrt(pmax(min.adjusted.variance,pmin(max.adjusted.variance,df$qv))/exp(df$v))}

\item{\code{min.adjusted.variance}}{numeric Minimum adjusted variance (default=1e-3). The gene scale factor is defined as sqrt(pmax(min.adjusted.variance,pmin(max.adjusted.variance,df$qv))/exp(df$v))}

\item{\code{cells}}{character vector Subset of cells upon which to perform variance normalization with adjustVariance() (default=NULL)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{min.gene.cells}}{integer Minimum number of genes per cells (default=0). This parameter is used to filter counts.}

\item{\code{persist}}{boolean Whether to save results (default=TRUE, i.e. is.null(cells)).}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
residual matrix with adjusted variance
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\donttest{
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=500)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object 
p2_object <- Pagoda2$new(counts,log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
## Normalize gene expression variance
p2_object$adjustVariance(plot=TRUE, gam.k=10)
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-makeKnnGraph"></a>}}
\if{latex}{\out{\hypertarget{method-makeKnnGraph}{}}}
\subsection{Method \code{makeKnnGraph()}}{
Create k-nearest neighbor graph
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$makeKnnGraph(
  k = 30,
  nrand = 1000,
  type = "counts",
  weight.type = "1m",
  odgenes = NULL,
  n.cores = self$n.cores,
  distance = "cosine",
  center = TRUE,
  x = NULL,
  p = NULL,
  var.scale = (type == "counts"),
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{k}}{integer Number of k clusters for k-NN (default=30)}

\item{\code{nrand}}{numeric Number of randomizations i.e. the gene sets (of the same size) to be evaluated in parallel with each gene set (default=1e3)}

\item{\code{type}}{string Data type of the reduction (default='counts'). If type='counts', this will access the raw counts. Otherwise, 'type' must be name of the reductions.}

\item{\code{weight.type}}{string 'cauchy', 'normal', 'constant', '1m' (default='1m')}

\item{\code{odgenes}}{character vector Overdispersed genes to retrieve (default=NULL)}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{distance}}{string Distance metric used: 'cosine', 'L2', 'L1', 'cauchy', 'euclidean' (default='cosine')}

\item{\code{center}}{boolean Whether to use centering when distance='cosine' (default=TRUE). The parameter is ignored otherwise.}

\item{\code{x}}{counts or reduction to use (default=NULL). If NULL, uses counts. Otherwise, checks for the reduction in self$reductions[[type]]}

\item{\code{p}}{(default=NULL)}

\item{\code{var.scale}}{boolean Apply scaling if using raw counts (default=TRUE). If type="counts", var.scale is TRUE by default.}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
kNN graph, stored in self$graphs
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{\donttest{
## Load pre-generated a dataset of 50 bone marrow cells as matrix
cm <- readRDS(system.file("extdata", "sample_BM1_50.rds", package="pagoda2"))
## Perform QC, i.e. filter any cells that
counts <- gene.vs.molecule.cell.filter(cm, min.cell.size=300)
rownames(counts) <- make.unique(rownames(counts))
## Generate Pagoda2 object   
p2_object <- Pagoda2$new(counts,log.scale=TRUE, min.cells.per.gene=10, n.cores=1) 
## Normalize gene expression variance
p2_object$adjustVariance(plot=TRUE, gam.k=10)
## Generate a kNN graph of cells that will allow us to identify clusters of cells
p2_object$makeKnnGraph(k=20, center=FALSE, distance='L2')
}

}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getKnnClusters"></a>}}
\if{latex}{\out{\hypertarget{method-getKnnClusters}{}}}
\subsection{Method \code{getKnnClusters()}}{
Calculate clusters based on the kNN graph
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getKnnClusters(
  type = "counts",
  method = igraph::multilevel.community,
  name = "community",
  n.cores = self$n.cores,
  g = NULL,
  min.cluster.size = 1,
  persist = TRUE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{method}}{Method to use (default=igraph::multilevel.community). Accepted methods are either 'igraph::infomap.community' or 'igraph::multilevel.community'. 
If NULL, if the number of vertices of the graph is greater than or equal to 2000, 'igraph::multilevel.community' will be used. Otherwise, 'igraph::infomap.community' will be used.}

\item{\code{name}}{string Name of the community structure calculated from 'method' (default='community')}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{g}}{Input graph (default=NULL). If NULL, access graph from self$graphs[[type]].}

\item{\code{min.cluster.size}}{Minimum size of clusters (default=1). This parameter is primarily used to remove very small clusters.}

\item{\code{persist}}{boolean Whether to save the clusters and community structure (default=TRUE)}

\item{\code{...}}{Additional parameters to pass to 'method'}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
the community structure calculated from 'method'
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-geneKnnbyPCA"></a>}}
\if{latex}{\out{\hypertarget{method-geneKnnbyPCA}{}}}
\subsection{Method \code{geneKnnbyPCA()}}{
Deprecated function. Use makeGeneKnnGraph() instead.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$geneKnnbyPCA()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getHierarchicalDiffExpressionAspects"></a>}}
\if{latex}{\out{\hypertarget{method-getHierarchicalDiffExpressionAspects}{}}}
\subsection{Method \code{getHierarchicalDiffExpressionAspects()}}{
Take a given clustering and generate a hierarchical clustering
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getHierarchicalDiffExpressionAspects(
  type = "counts",
  groups = NULL,
  clusterName = NULL,
  method = "ward.D",
  dist = "pearson",
  persist = TRUE,
  z.threshold = 2,
  n.cores = self$n.cores,
  min.set.size = 5,
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type of the reduction (default='counts'). If type='counts', this will access the raw counts. Otherwise, 'type' must be name of the reductions.}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{clusterName}}{string Cluster name to access (default=NULL)}

\item{\code{method}}{string The agglomeration method to be used in stats::hcust(method=method) (default='ward.D'). Accepted values are: "ward.D", "ward.D2", "single", "complete", "average" (= UPGMA), "mcquitty" (= WPGMA), "median" (= WPGMC) or "centroid" (= UPGMC). For more information, see stats::hclust().}

\item{\code{dist}}{string 'pearson', 'spearman', 'euclidean', 'L2', 'JS' (default='pearson')}

\item{\code{persist}}{boolean Whether to save the clusters and community structure (default=TRUE)}

\item{\code{z.threshold}}{numeric Threshold of z-scores to filter, >=z.threshold are kept (default=2)}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{min.set.size}}{integer Minimum threshold of sets to keep (default=5)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
hierarchical clustering
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-makeGeneKnnGraph"></a>}}
\if{latex}{\out{\hypertarget{method-makeGeneKnnGraph}{}}}
\subsection{Method \code{makeGeneKnnGraph()}}{
Calculates gene Knn network for gene similarity
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$makeGeneKnnGraph(
  nPcs = 100,
  center = TRUE,
  fastpath = TRUE,
  maxit = 1000,
  k = 30,
  n.cores = self$n.cores,
  verbose = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{nPcs}}{integer Number of principal components (default=100). This is the parameter 'nv' in irlba::irlba(), the number of right singular vectors to estimate.}

\item{\code{center}}{boolean Whether to center the PCA (default=TRUE)}

\item{\code{fastpath}}{boolean Whether to try a (fast) C algorithm implementation if possible (default=TRUE). This parameter is equivalent to 'fastpath' in irlba::irlba().}

\item{\code{maxit}}{integer Maximum number of iterations (default=1000). This parameter is equivalent to 'maxit' in irlba::irlba().}

\item{\code{k}}{integer Number of k clusters for calculating k-NN on the resulting principal components (default=30).}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
graph with gene similarity
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getDensityClusters"></a>}}
\if{latex}{\out{\hypertarget{method-getDensityClusters}{}}}
\subsection{Method \code{getDensityClusters()}}{
Calculate density-based clusters
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getDensityClusters(
  type = "counts",
  embeddingType = NULL,
  name = "density",
  eps = 0.5,
  v = 0.7,
  s = 1,
  verbose = TRUE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{embeddingType}}{The type of embedding used when calculating with `getEmbedding()` (default=NULL). Accepted values are: 'largeVis', 'tSNE', 'FR', 'UMAP', 'UMAP_graph'}

\item{\code{name}}{string Name fo the clustering (default='density').}

\item{\code{eps}}{numeric value of the eps parameter, fed into dbscan::dbscan(x=emb, eps=eps, ...)}

\item{\code{v}}{numeric The “value” to be used to complete the HSV color descriptions (default=0.7). Equivalent to the 'v' parameter in grDevices::rainbow().}

\item{\code{s}}{numeric The “saturation” to be used to complete the HSV color descriptions (default=1). Equivalent to the 's' parameter in grDevices::rainbow().}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{...}}{Additional parameters passed to dbscan::dbscan(emb, ...)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
density-based clusters
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getDifferentialGenes"></a>}}
\if{latex}{\out{\hypertarget{method-getDifferentialGenes}{}}}
\subsection{Method \code{getDifferentialGenes()}}{
Determine differentially expressed genes, comparing each group against all others using Wilcoxon rank sum test
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getDifferentialGenes(
  type = "counts",
  clusterType = NULL,
  groups = NULL,
  name = "customClustering",
  z.threshold = 3,
  upregulated.only = FALSE,
  verbose = FALSE,
  append.specificity.metrics = TRUE,
  append.auc = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{clusterType}}{Optional cluster type to use as a group-defining factor (default=NULL)}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{name}}{string Slot to store the results in (default='customClustering')}

\item{\code{z.threshold}}{numeric Minimal absolute Z score (adjusted) to report (default=3)}

\item{\code{upregulated.only}}{boolean Whether to report only genes that are expressed significantly higher in each group (default=FALSE)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=FALSE)}

\item{\code{append.specificity.metrics}}{boolean Whether to append specifity metrics (default=TRUE). Uses the function sccore::appendSpecificityMetricsToDE().}

\item{\code{append.auc}}{boolean If TRUE, append AUC values (default=FALSE). Parameter ignored if append.specificity.metrics is FALSE.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
List with each element of the list corresponding to a cell group in the provided/used factor (i.e. factor levels) 
    Each element of a list is a data frame listing the differentially epxressed genes (row names), with the following columns: 
    Z - adjusted Z score, with positive values indicating higher expression in a given group compare to the rest
    M - log2 fold change
    highest- a boolean flag indicating whether the expression of a given gene in a given vcell group was on average higher than in every other cell group
    fe - fraction of cells in a given group having non-zero expression level of a given gene
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plotDiffGeneHeatmap"></a>}}
\if{latex}{\out{\hypertarget{method-plotDiffGeneHeatmap}{}}}
\subsection{Method \code{plotDiffGeneHeatmap()}}{
Plot heatmap of DE results
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$plotDiffGeneHeatmap(
  type = "counts",
  clusterType = NULL,
  groups = NULL,
  n.genes = 100,
  z.score = 2,
  gradient.range.quantile = 0.95,
  inner.clustering = FALSE,
  gradientPalette = NULL,
  v = 0.8,
  s = 1,
  box = TRUE,
  drawGroupNames = FALSE,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{clusterType}}{Optional cluster type to use as a group-defining factor (default=NULL)}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{n.genes}}{integer Number of genes to plot (default=100)}

\item{\code{z.score}}{numeric Threshold of z-scores to filter (default=2). Only greater than or equal to this value are kept.}

\item{\code{gradient.range.quantile}}{numeric Trimming quantile (default=0.95)}

\item{\code{inner.clustering}}{boolean Whether to cluster cells within each cluster (default=FALSE)}

\item{\code{gradientPalette}}{palette of colors to use (default=NULL). If NULL, uses 'colorRampPalette(c('gray90','red'), space = "Lab")(1024)'}

\item{\code{v}}{numeric The “value” to be used to complete the HSV color descriptions (default=0.7). Equivalent to the 'v' parameter in grDevices::rainbow().}

\item{\code{s}}{numeric The “saturation” to be used to complete the HSV color descriptions (default=1). Equivalent to the 's' parameter in grDevices::rainbow().}

\item{\code{box}}{boolean Whether to draw a box around the current plot in the given color and linetype (default=TRUE)}

\item{\code{drawGroupNames}}{boolean Whether to draw group names (default=FALSE)}

\item{\code{...}}{Additional parameters passed to internal function used for heatmap plotting, my.heatmap2()}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
heatmap of DE results
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getRefinedLibSizes"></a>}}
\if{latex}{\out{\hypertarget{method-getRefinedLibSizes}{}}}
\subsection{Method \code{getRefinedLibSizes()}}{
Recalculate library sizes using robust regression within clusters
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getRefinedLibSizes(
  clusterType = NULL,
  groups = NULL,
  type = "counts",
  n.cores = self$n.cores
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{clusterType}}{Name of cluster to access (default=NULL). If NULL, takes the most recently generated clustering. Parameter ignored if groups is not NULL.}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells (default=NULL)}

\item{\code{type}}{string Either 'counts' or the name of a stored embedding, names(self$embeddings) (default=NULL)}

\item{\code{n.cores}}{numeric Number of cores to use (default=self$n.cores=1)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
recalculated library sizes
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plotGeneHeatmap"></a>}}
\if{latex}{\out{\hypertarget{method-plotGeneHeatmap}{}}}
\subsection{Method \code{plotGeneHeatmap()}}{
Plot heatmap for a given set of genes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$plotGeneHeatmap(
  genes,
  type = "counts",
  clusterType = NULL,
  groups = NULL,
  gradient.range.quantile = 0.95,
  cluster.genes = FALSE,
  inner.clustering = FALSE,
  gradientPalette = NULL,
  v = 0.8,
  s = 1,
  box = TRUE,
  drawGroupNames = FALSE,
  useRaster = TRUE,
  smooth.span = max(1, round(nrow(self$counts)/1024)),
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{genes}}{character vector Gene names}

\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{clusterType}}{Optional cluster type to use as a group-defining factor (default=NULL)}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{gradient.range.quantile}}{numeric Trimming quantile (default=0.95)}

\item{\code{cluster.genes}}{boolean Whether to cluster genes within each cluster using stats::hclust() (default=FALSE)}

\item{\code{inner.clustering}}{boolean Whether to cluster cells within each cluster (default=FALSE)}

\item{\code{gradientPalette}}{palette of colors to use (default=NULL). If NULL, uses 'colorRampPalette(c('gray90','red'), space = "Lab")(1024)'}

\item{\code{v}}{numeric The “value” to be used to complete the HSV color descriptions (default=0.7). Equivalent to the 'v' parameter in grDevices::rainbow().}

\item{\code{s}}{numeric The “saturation” to be used to complete the HSV color descriptions (default=1). Equivalent to the 's' parameter in grDevices::rainbow().}

\item{\code{box}}{boolean Whether to draw a box around the current plot in the given color and linetype (default=TRUE)}

\item{\code{drawGroupNames}}{boolean Whether to draw group names (default=FALSE)}

\item{\code{useRaster}}{boolean If TRUE a bitmap raster is used to plot the image instead of polygons (default=TRUE). The grid must be regular in that case, otherwise an error is raised. For more information, see graphics::image().}

\item{\code{smooth.span}}{(default=max(1,round(nrow(self$counts)/1024)))}

\item{\code{...}}{Additional parameters passed to internal function used for heatmap plotting, my.heatmap2()}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
plot of gene heatmap
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-plotEmbedding"></a>}}
\if{latex}{\out{\hypertarget{method-plotEmbedding}{}}}
\subsection{Method \code{plotEmbedding()}}{
Show embedding
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$plotEmbedding(
  type = NULL,
  embeddingType = NULL,
  clusterType = NULL,
  groups = NULL,
  colors = NULL,
  gene = NULL,
  plot.theme = ggplot2::theme_bw(),
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Either 'counts' or the name of a stored embedding, names(self$embeddings) (default=NULL)}

\item{\code{embeddingType}}{string Embedding type (default=NULL). If NULL, takes the most recently generated embedding.}

\item{\code{clusterType}}{Name of cluster to access (default=NULL). If NULL, takes the most recently generated clustering. Parameter ignored if groups is not NULL.}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells (default=NULL)}

\item{\code{colors}}{character vector List of gene names (default=NULL)}

\item{\code{gene}}{(default=NULL)}

\item{\code{plot.theme}}{(default=ggplot2::theme_bw())}

\item{\code{...}}{Additional parameters passed to sccore::embeddingPlot()}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
plot of the embedding
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getOdGenes"></a>}}
\if{latex}{\out{\hypertarget{method-getOdGenes}{}}}
\subsection{Method \code{getOdGenes()}}{
Get overdispersed genes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getOdGenes(
  n.odgenes = NULL,
  alpha = 0.05,
  use.unadjusted.pvals = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{n.odgenes}}{integer Number of overdispersed genes to retrieve (default=NULL). If NULL, will return all.}

\item{\code{alpha}}{numeric The Type I error probability or the significance level (default=5e-2). This is the criterion used to measure statistical significance, i.e. if the p-value < alpha, then it is statistically significant.}

\item{\code{use.unadjusted.pvals}}{boolean Whether to use Benjamini-Hochberg adjusted p-values (default=FALSE).}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
vector of overdispersed genes
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getNormalizedExpressionMatrix"></a>}}
\if{latex}{\out{\hypertarget{method-getNormalizedExpressionMatrix}{}}}
\subsection{Method \code{getNormalizedExpressionMatrix()}}{
Return variance-normalized matrix for specified genes or a number of OD genes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getNormalizedExpressionMatrix(genes = NULL, n.odgenes = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{genes}}{vector of gene names to explicitly return (default=NULL)}

\item{\code{n.odgenes}}{integer Number of overdispersed genes to retrieve (default=NULL). If NULL, will return all.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
cell by gene matrix
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-calculatePcaReduction"></a>}}
\if{latex}{\out{\hypertarget{method-calculatePcaReduction}{}}}
\subsection{Method \code{calculatePcaReduction()}}{
Calculate PCA reduction of the data
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$calculatePcaReduction(
  nPcs = 20,
  type = "counts",
  name = "PCA",
  use.odgenes = TRUE,
  n.odgenes = NULL,
  odgenes = NULL,
  center = TRUE,
  cells = NULL,
  fastpath = TRUE,
  maxit = 100,
  verbose = TRUE,
  var.scale = (type == "counts"),
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{nPcs}}{numeric Number of principal components (PCs) (default=20)}

\item{\code{type}}{string Dataset view to reduce (counts by default, but can specify a name of an existing reduction) (default='counts')}

\item{\code{name}}{string Name for the PCA reduction to be created (default='PCA')}

\item{\code{use.odgenes}}{boolean Whether pre-calculated set of overdispersed genes should be used (default=TRUE)}

\item{\code{n.odgenes}}{integer Number of overdispersed genes to retrieve (default=NULL). If NULL, will return all.}

\item{\code{odgenes}}{Explicitly specify a set of overdispersed genes to use for the reduction (default=NULL)}

\item{\code{center}}{boolean Whether data should be centered prior to PCA (default=TRUE)}

\item{\code{cells}}{optional subset of cells on which PCA should be run (default=NULL)}

\item{\code{fastpath}}{boolean Use C implementation for speedup (default=TRUE)}

\item{\code{maxit}}{numeric Maximum number of iterations (default=100). For more information, see 'maxit' parameter in irlba::irlba().}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{var.scale}}{boolean Apply scaling if using raw counts (default=TRUE). If type="counts", var.scale is TRUE by default.}

\item{\code{...}}{additional arguments forwarded to irlba::irlba}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Invisible PCA result (the reduction itself is saved in self$reductions[[name]])"
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-expandOdGenes"></a>}}
\if{latex}{\out{\hypertarget{method-expandOdGenes}{}}}
\subsection{Method \code{expandOdGenes()}}{
Reset overdispersed genes 'odgenes' to be a superset of the standard odgene selection (guided by n.odgenes or alpha), 
    and a set of recursively determined odgenes based on a given group (or a cluster info)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$expandOdGenes(
  type = "counts",
  clusterType = NULL,
  groups = NULL,
  min.group.size = 30,
  od.alpha = 0.1,
  use.odgenes = FALSE,
  n.odgenes = NULL,
  odgenes = NULL,
  n.odgene.multiplier = 1,
  gam.k = 10,
  verbose = FALSE,
  n.cores = self$n.cores,
  min.odgenes = 10,
  max.odgenes = Inf,
  recursive = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{clusterType}}{Optional cluster type to use as a group-defining factor (default=NULL)}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{min.group.size}}{integer Number of minimum cells for filtering out group size (default=30)}

\item{\code{od.alpha}}{numeric The Type I error probability or the significance level for calculating overdispersed genes (default=1e-1). This is the criterion used to measure statistical significance, i.e. if the p-value < alpha, then it is statistically significant.}

\item{\code{use.odgenes}}{boolean Whether pre-calculated set of overdispersed genes should be used (default=FALSE)}

\item{\code{n.odgenes}}{integer Number of overdispersed genes to retrieve (default=NULL). If NULL, will return all.}

\item{\code{odgenes}}{Explicitly specify a set of overdispersed genes to use for the reduction (default=NULL) #' @param odgenes (default=NULL)}

\item{\code{n.odgene.multiplier}}{numeric (default=1)}

\item{\code{gam.k}}{integer The k used for the generalized additive model 'v ~ s(m, k =gam.k)' (default=10). If gam.k<2, linear regression is used 'lm(v ~ m)'.}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{min.odgenes}}{integer Minimum number of overdispersed genes to use (default=10)}

\item{\code{max.odgenes}}{integer Maximum number of overdispersed genes to use (default=Inf)}

\item{\code{recursive}}{boolean Whether to determine groups for which variance normalization will be rerun (default=TRUE)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
List of overdispersed genes
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-localPcaKnn"></a>}}
\if{latex}{\out{\hypertarget{method-localPcaKnn}{}}}
\subsection{Method \code{localPcaKnn()}}{
local PCA implementation
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$localPcaKnn(
  nPcs = 5,
  type = "counts",
  clusterType = NULL,
  groups = NULL,
  k = 30,
  b = 1,
  a = 1,
  min.group.size = 30,
  name = "localPCA",
  od.alpha = 0.1,
  n.odgenes = NULL,
  gam.k = 10,
  verbose = FALSE,
  n.cores = self$n.cores,
  min.odgenes = 5,
  take.top.odgenes = FALSE,
  recursive = TRUE,
  euclidean = FALSE,
  perplexity = k,
  return.pca = FALSE,
  skip.pca = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{nPcs}}{integer Number of principal components (default=5)}

\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{clusterType}}{Optional cluster type to use as a group-defining factor (default=NULL)}

\item{\code{groups}}{factor named with cell names specifying the clusters of cells to be compared (one against all) (default=NULL). To compare two cell clusters against each other, simply pass a factor containing only two levels.}

\item{\code{k}}{integer Number of components for kNN graph (default=30)}

\item{\code{b}}{numeric Constant within exp(-b*(ncid/cldsd)^2), used for calculating cell relevance per cluster (default=1)}

\item{\code{a}}{numeric Constant within "(1-exp(-a*(dsq)/(p$pcs$trsd^2)))*(pk /outerproduct pk)" (default=1)}

\item{\code{min.group.size}}{integer Number of minimum cells for filtering out group size (default=30)}

\item{\code{name}}{string Title (default='localPCA')}

\item{\code{od.alpha}}{numeric Significance level for calculating overdispersed genes (default=1e-1). P-values will be filtered by <log(od.alpha).}

\item{\code{n.odgenes}}{integer Number of overdispersed genes to retrieve (default=NULL). If NULL, will return all.}

\item{\code{gam.k}}{integer The k used for the generalized additive model 'v ~ s(m, k =gam.k)' (default=10). If gam.k<2, linear regression is used 'lm(v ~ m)'.}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{min.odgenes}}{integer Minimum number of overdispersed genes to use (default=5)}

\item{\code{take.top.odgenes}}{boolean Take top overdispersed genes in decreasing order (default=FALSE)}

\item{\code{recursive}}{boolean Whether to recursively determine groups for which variance normalization will be rerun (default=FALSE)}

\item{\code{euclidean}}{boolean Whether to applied euclidean-based distance similarity during variance normalization (default=FALSE)}

\item{\code{perplexity}}{integer Perplexity parameter within Rtsne::Rtsne() (default=k). Please see Rtsne for more details.}

\item{\code{return.pca}}{boolean Whether to return the PCs (default=FALSE)}

\item{\code{skip.pca}}{boolean If TRUE and return.pca=TRUE, will return a list of scale factors, cells, and overdispersed genes, i.e. list(sf=sf, cells=cells, odgenes=odgenes) (default=FALSE). Otherwise, ignored.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
localPcaKnn return here
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-testPathwayOverdispersion"></a>}}
\if{latex}{\out{\hypertarget{method-testPathwayOverdispersion}{}}}
\subsection{Method \code{testPathwayOverdispersion()}}{
Test pathway overdispersion
Note: this is a compressed version of the PAGODA1 approach in SCDE <https://hms-dbmi.github.io/scde/>
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$testPathwayOverdispersion(
  setenv,
  type = "counts",
  max.pathway.size = 1000,
  min.pathway.size = 10,
  n.randomizations = 5,
  verbose = FALSE,
  n.cores = self$n.cores,
  score.alpha = 0.05,
  plot = FALSE,
  cells = NULL,
  adjusted.pvalues = TRUE,
  z.score = qnorm(0.05/2, lower.tail = FALSE),
  use.oe.scale = FALSE,
  return.table = FALSE,
  name = "pathwayPCA",
  correlation.distance.threshold = 0.2,
  loading.distance.threshold = 0.01,
  top.aspects = Inf,
  recalculate.pca = FALSE,
  save.pca = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{setenv}}{Specific environment for pathway analysis}

\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{max.pathway.size}}{integer Maximum number of observed genes in a valid gene set (default=1e3)}

\item{\code{min.pathway.size}}{integer Minimum number of observed genes that should be contained in a valid gene set (default=10)}

\item{\code{n.randomizations}}{numeric Number of random gene sets (of the same size) to be evaluated in parallel with each gene set (default=5). (This can be kept at 5 or 10, but should be increased to 50-100 if the significance of pathway overdispersion will be determined relative to random gene set models.)}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{score.alpha}}{numeric Significance level of the confidence interval for determining upper/lower bounds (default=0.05)}

\item{\code{plot}}{boolean Whether to output the plot (default=FALSE)}

\item{\code{cells}}{character vector Specific cells to investigate (default=NULL)}

\item{\code{adjusted.pvalues}}{boolean Whether to use adjusted p-values (default=TRUE)}

\item{\code{z.score}}{numeric Z-score to be used as a cutoff for statistically significant patterns (default=qnorm(0.05/2, lower.tail = FALSE))}

\item{\code{use.oe.scale}}{boolean Whether the variance of the returned aspect patterns should be normalized using observed/expected value instead of the default chi-squared derived variance corresponding to overdispersion Z-score (default=FALSE)}

\item{\code{return.table}}{boolean Whether to return a text table with results (default=FALSE)}

\item{\code{name}}{string Title (default='pathwayPCA')}

\item{\code{correlation.distance.threshold}}{numeric Similarity threshold for grouping interdependent aspects in pagoda.reduce.redundancy() (default=0.2)}

\item{\code{loading.distance.threshold}}{numeric Similarity threshold for grouping interdependent aspects in pagoda.reduce.loading.redundancy() (default=0.2)}

\item{\code{top.aspects}}{Restrict output to the top N aspects of heterogeneity (default=Inf)}

\item{\code{recalculate.pca}}{boolean Whether to recalculate PCA (default=FALSE)}

\item{\code{save.pca}}{boolean Whether to save the PCA results (default=TRUE). If TRUE, caches them in self$misc[['pwpca']].}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
pathway output
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-getEmbedding"></a>}}
\if{latex}{\out{\hypertarget{method-getEmbedding}{}}}
\subsection{Method \code{getEmbedding()}}{
Return embedding
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$getEmbedding(
  type = "counts",
  embeddingType = "largeVis",
  name = NULL,
  dims = 2,
  M = 1,
  gamma = 1/M,
  perplexity = 50,
  verbose = TRUE,
  sgd_batches = NULL,
  diffusion.steps = 0,
  diffusion.power = 0.5,
  distance = "pearson",
  n.cores = self$n.cores,
  n.sgd.cores = n.cores,
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{type}}{string Data type (default='counts'). Currently only 'counts' supported.}

\item{\code{embeddingType}}{string Type of embedding to construct (default='largeVis'). Possible values are: 'largeVis', 'tSNE', 'FR' (Fruchterman–Reingold), 'UMAP', 'UMAP_graph'}

\item{\code{name}}{string Name of the embedding (default=NULL). If NULL, the name = embeddingType.}

\item{\code{dims}}{integer Parameter 'dims' Matrix::sparseMatrix(); a non-negative, integer, dimensions vector of length 2 (default=2). See Matrix package documentation for more details.}

\item{\code{M}}{numeric (largeVis) The number of negative edges to sample for each positive edge (default=5). Parameter only used if embeddingType is 'largeVis'.}

\item{\code{gamma}}{numeric (largeVis) The strength of the force pushing non-neighbor nodes apart (default=7). Parameter only used if embeddingType is 'largeVis'.}

\item{\code{perplexity}}{numeric Parameter 'perplexity' within largeVis::buildWijMatrix() (default=50). Please see the largeVis documentation for more details.}

\item{\code{verbose}}{boolean Whether to give verbose output (default=TRUE)}

\item{\code{sgd_batches}}{numeric The number of edges to process during SGD (default=NULL). Passed to projectKNNs(). Defaults to a value set based on the size of the dataset. If the parameter given is
between \code{0} and \code{1}, the default value will be multiplied by the parameter.}

\item{\code{diffusion.steps}}{integer Iteration steps to use. If 0, no steps are run. (default=0)}

\item{\code{diffusion.power}}{numeric Factor to be used when calculating diffusion, (default=0.5)}

\item{\code{distance}}{string 'pearson', 'spearman', 'euclidean', 'L2', 'JS' (default='pearson')}

\item{\code{n.cores}}{numeric Number of cores to use (default=1)}

\item{\code{n.sgd.cores}}{numeric Number of cores to use (default=n.cores)}

\item{\code{...}}{Additional parameters passed to embedding functions, Rtsne::Rtsne() if 'L2', uwot::umap() if 'UMAP', embedKnnGraphUmap() if 'UMAP_graph'}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
embedding stored in self$embedding
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-clone"></a>}}
\if{latex}{\out{\hypertarget{method-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{Pagoda2$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
