% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bin_time.R
\name{bin_time}
\alias{bin_time}
\title{Assign fossil occurrences to time bins}
\usage{
bin_time(occdf, bins, method = "mid", reps = 100)
}
\arguments{
\item{occdf}{\code{dataframe}. A dataframe of the fossil occurrences you
wish to bin. This dataframe should contain the following named columns:
"max_ma" and "min_ma". These columns should contain \code{numeric} values.
If required, \code{numeric} ages can be generated from interval names via the
\code{\link[palaeoverse:look_up]{look_up()}}.}

\item{bins}{\code{dataframe}. A dataframe of the bins that you wish to
allocate fossil occurrences to such as that returned by
\code{\link[palaeoverse:time_bins]{time_bins()}}. This dataframe must
contain at least the following named columns: "bin", "max_ma" and "min_ma".
Columns "max_ma" and "min_ma" must be \code{numeric} values.}

\item{method}{\code{character}. The method desired for binning fossil
occurrences. Currently, five methods exist in this function: "mid",
"majority", "all", "random", and "point". See Details for a description of
each.}

\item{reps}{\code{numeric}. A non-negative \code{numeric} specifying the number
of replications for sampling. This argument is only useful in the case of
the "random" or "point" method being specified in the \code{method} argument.
Defaults to 100.}
}
\value{
For methods "mid", "majority" and "all", a \code{dataframe} of the
original input \code{occdf} with the following appended columns is returned:
occurrence id (\code{id}), number of bins that the occurrence age
range covers (\code{n_bins}), bin assignment (\code{bin_assignment}), and bin midpoint
(\code{bin_midpoint}). In the case of the "majority" method, an additional column
of the majority percentage overlap (\code{overlap_percentage}) is also appended.
For the "random" and "point" method, a
\code{list} is returned (of length reps) with each element a copy of
the \code{occdf} and appended columns (random: \code{bin_assignment} and
\code{bin_midpoint}; point: \code{bin_assignment} and \code{point_estimates}).
}
\description{
A function to assign fossil occurrences to specified time bins based on
different approaches commonly applied in palaeobiology.
}
\details{
Five approaches (methods) exist in the \code{bin_time()} function
for assigning occurrences to time bins:
\itemize{
\item Midpoint: The "mid" method is the simplest approach and uses the midpoint
of the fossil occurrence age range to bin the occurrence.
\item Majority: The "majority" method bins an occurrence into the bin which it
most overlaps with.
As part of this implementation, the majority percentage overlap of the
occurrence is also calculated and returned as an additional column in
\code{occdf}. If desired, these percentages can be used to further filter an
occurrence dataset.
\item All: The "all" method bins an occurrence into every bin its age range
covers. For occurrences with age ranges of more than one bin, the occurrence
row is duplicated. Each occurrence is assigned an ID in the column
\code{occdf$id} so that duplicates can be tracked. Additionally, \code{occdf$n_bins}
records the number of bins each occurrence appears within.
\item Random: The "random" method randomly samples X amount of bins (with
replacement) from the bins that the fossil occurrence age range covers with
equal probability regardless of bin length.
The \code{reps} argument determines the number of times the sample process is
repeated. All replications are stored as individual elements
within the returned list with an appended \code{bin_assignment} and
\code{bin_midpoint} column to the original input \code{occdf}.
\item Point: The "point" method randomly samples X amount of point age estimates
from the age range of the fossil occurrence. Sampling follows a
uniform probability distribution defined by the age range of the fossil
occurrence. As such, bins which cover more of the age range of the fossil
occurrence are more likely to be assigned. The \code{reps} argument determines
the number of times the sample process is repeated. All replications are
stored as individual elements within the returned list with an appended
\code{bin_assignment} and \code{point_estimates} column to the original input \code{occdf}.
}
}
\section{Developer(s)}{

Christopher D. Dean & Lewis A. Jones
}

\section{Reviewer(s)}{

William Gearty
}

\examples{
#Grab internal tetrapod data
occdf <- tetrapods[1:100, ]
bins <- time_bins()

#Assign via midpoint age of fossil occurrence data
ex1 <- bin_time(occdf = occdf, bins = bins, method = "mid")

#Assign to all bins that age range covers
ex2 <- bin_time(occdf = occdf, bins = bins, method = "all")

#Assign via majority overlap based on fossil occurrence age range
ex3 <- bin_time(occdf = occdf, bins = bins, method = "majority")

#Assign randomly to overlapping bins based on fossil occurrence age range
ex4 <- bin_time(occdf = occdf, bins = bins, method = "random", reps = 5)

#Assign point estimates based on fossil occurrence age range
ex5 <- bin_time(occdf = occdf, bins = bins, method = "point", reps = 5)
}
