\name{idn}
\alias{idn}
\alias{C_bd_simp}

\title{
Multidimensional evaluation on posets (Identification Function only)
}
\description{
Given a partial order (arguments \code{profiles} and/or
\code{zeta}) and a selected \code{threshold},
the function computes the identification function, as a S3 class object \code{parsec}. The identification function is computed by uniform sampling of the linear extensions of the input poset, through a C implementation of the Bubley - Dyer (1999) algorithm. \link{idn} is a simplified and faster version of \link{evaluation}, computing just the identification function.
}
\usage{
idn(
    profiles = NULL,
    threshold,
    error = 10^(-3),
    zeta = getzeta(profiles),
    weights = {
        if (!is.null(profiles)) 
            profiles$freq
        else rep(1, nrow(zeta))
    },
    linext = lingen(zeta),
    nit = floor({
        n <- nrow(zeta)
        n^5 * log(n) + n^4 * log(error^(-1))
    }),
    maxint = 2^31 - 1
)
}

\arguments{
  \item{profiles}{
an object of S3 class \code{wprof}.
}
  \item{threshold}{
a vector identifying the threshold. It can be a vector of indexes (numeric),
a vector of poset element names (character) or a boolean vector of length equal to the number of
elements.
}
  \item{error}{
the "distance" from uniformity in the sampling distribution of linear extensions.
}
  \item{zeta}{
the incidence matrix of the poset. An object of S3 class \code{incidence}.
By default, extracted from \code{profiles}.
}
  \item{weights}{
weights assigned to profiles. If the argument \code{profiles} is not \code{NULL},
 weights are by default set equal to profile frequencies, otherwise they are
set equal to 1.
}
  \item{linext}{
the linear extension initializing the sampling algorithm. By default, it is generated by \code{lingen(zeta)}. Alternatively, it can be provided by 
the user through a vector of elements positions.
}
  \item{nit}{
Number of iterations in the Bubley-Dyer algorithm, by default evaluated using a formula of Karzanov and Khachiyan
based on the number of poset elements and the argument \code{error} (see Bubley and Dyer, 1999).
}
  \item{maxint}{
Maximum integer. By default the maximum integer obtainable in a 32bit system.
This argument is used to group iterations and run the compiled
C code more times, so as to avoid memory indexing problems. User can
set a lower value to \code{maxint} in case of lower RAM availability.
}
}

\value{
\item{profiles}{an object of S3 class \code{wprof} reporting poset profiles
and their associated frequencies (number of statistical units in each profile).}

\item{number_of_profiles}{number of profiles.}

\item{number_of_variables}{number of variables.}

\item{incidence}{S3 class \code{incidence}, incidence matrix of the poset.}

\item{cover}{S3 class \code{cover}, cover matrix of the poset.}

\item{threshold}{boolean vector specifying whether a profile belongs to the threshold.}

\item{number_of_iterations}{number of iterations performed by the Bubley Dyer algorithm.}

\item{rank_dist}{matrix reporting by rows the relative
frequency distribution of the poverty ranks of each profile, over the set of sampled linear extensions.}

\item{thr_dist}{vector reporting the relative frequency a profile is used as threshold in the sampled linear extensions.
This result is useful for a posteriori valuation of the poset threshold.}

\item{prof_w}{vector of weights assigned to each profile.}

\item{edges_weights}{matrix of distances between profiles, used to evaluate the
measures of gap.}

\item{idn_f}{vector reporting the identification function, computed as the fraction of sampled linear extensions where a profile is in the downset of the threshold.}

\item{svr_abs}{\code{NA} use \link{evaluation} to obtain this result.}

\item{svr_rel}{\code{NA} use \link{evaluation} to obtain this result.}

\item{wea_abs}{\code{NA} use \link{evaluation} to obtain this result.}

\item{wea_rel}{\code{NA} use \link{evaluation} to obtain this result.}

\item{poverty_gap}{\code{NA} use \link{evaluation} to obtain this result.}

\item{wealth_gap}{\code{NA} use \link{evaluation} to obtain this result.}

\item{inequality}{\code{NA} use \link{evaluation} to obtain this result.}
}
\references{
Bubley R., Dyer M. (1999), Faster random generation of linear extensions, Discrete Math., 201, 81-88.

Fattore M., Arcagni A. (2013), Measuring multidimensional polarization with ordinal data, SIS 2013 Statistical Conference, BES-M3.1 - The BES and the challenges of constructing composite indicators dealing with equity and sustainability
}
\examples{
profiles <- var2prof(varlen = c(3, 2, 4))
threshold <- c("311", "112")

res <- idn(profiles, threshold, maxint = 10^5)

summary(res)
plot(res)
}