% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_compare.R
\name{plotCompare}
\alias{plotCompare}
\title{Plot a comparison results for fitted or validated subgroup identification models}
\usage{
plotCompare(..., type = c("boxplot", "density", "interaction"),
  avg.line = TRUE)
}
\arguments{
\item{...}{the fitted (model or validation) objects to be plotted. Must be either
objects returned from \code{fit.subgroup()} or \code{validate.subgroup()}}

\item{type}{type of plot. \code{"density"} results in a density plot for the results
across all observations (if \code{x} is from \code{fit.subgroup()}) or if \code{x} is from \code{validate.subgroup()}
across iterations of either the bootstrap or training/test re-fitting. For the latter
case the test results will be plotted. \code{"boxplot"} results in boxplots across all observations/iterations of either
the bootstrap or training/test re-fitting. For the latter
case the test results will be plotted. \code{"interaction"} creates an
interaction plot for the different subgroups (crossing lines here means a meaningful subgroup)}

\item{avg.line}{boolean value of whether or not to plot a line for the average
value in addition to the density (only valid for \code{type = "density"})}
}
\description{
Plots comparison of results for estimated subgroup treatment effects
}
\examples{
library(personalized)

set.seed(123)
n.obs  <- 1000
n.vars <- 50
x <- matrix(rnorm(n.obs * n.vars, sd = 3), n.obs, n.vars)


# simulate non-randomized treatment
xbetat   <- 0.5 + 0.5 * x[,21] - 0.5 * x[,41]
trt.prob <- exp(xbetat) / (1 + exp(xbetat))
trt01    <- rbinom(n.obs, 1, prob = trt.prob)

trt      <- 2 * trt01 - 1

# simulate response
delta <- 2 * (0.5 + x[,2] - x[,3] - x[,11] + x[,1] * x[,12])
xbeta <- x[,1] + x[,11] - 2 * x[,12]^2 + x[,13]
xbeta <- xbeta + delta * trt

# continuous outcomes
y <- drop(xbeta) + rnorm(n.obs, sd = 2)

# create function for fitting propensity score model
prop.func <- function(x, trt)
{
    # fit propensity score model
    propens.model <- cv.glmnet(y = trt,
                               x = x, family = "binomial")
    pi.x <- predict(propens.model, s = "lambda.min",
                    newx = x, type = "response")[,1]
    pi.x
}

subgrp.model <- fit.subgroup(x = x, y = y,
                           trt = trt01,
                           propensity.func = prop.func,
                           loss   = "sq_loss_lasso",
                           nfolds = 5)              # option for cv.glmnet


subgrp.modelg <- fit.subgroup(x = x, y = y,
                           trt = trt01,
                           propensity.func = prop.func,
                           loss   = "sq_loss_lasso_gam")

plotCompare(subgrp.model, subgrp.modelg)

}
\seealso{
\code{\link[personalized]{fit.subgroup}} for function which fits subgroup identification models and
\code{\link[personalized]{validate.subgroup}} for function which creates validation results.
}
