% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/validate_subgroup.R
\name{validate.subgroup}
\alias{validate.subgroup}
\title{Validating fitted subgroup identification models}
\usage{
validate.subgroup(model, B = 50L, method = c("training_test_replication",
  "boot_bias_correction"), train.fraction = 0.5,
  benefit.score.quantiles = c(0.1666667, 0.3333333, 0.5, 0.6666667,
  0.8333333), parallel = FALSE)
}
\arguments{
\item{model}{fitted model object returned by \code{fit.subgroup()} function}

\item{B}{integer. number of bootstrap replications or refitting replications.}

\item{method}{validation method. \code{"boot_bias_correction"} for the bootstrap
bias correction method of Harrell, et al (1996) or \code{"training_test_replication"}
for repeated training and test splitting of the data (\code{train.fraction} should be specified
for this option)}

\item{train.fraction}{fraction (between 0 and 1) of samples to be used for training in
training/test replication. Only used for \code{method = "training_test_replication"}}

\item{benefit.score.quantiles}{a vector of quantiles (between 0 and 1) of the benefit score values
for which to return bootstrapped information about the subgroups. ie if one of the quantile values is 0.5, the
median value of the benefit scores will be used as a cutoff to determine subgroups and summary statistics
will be returned about these subgroups}

\item{parallel}{Should the loop over replications be parallelized? If \code{FALSE}, then no, if \code{TRUE}, then yes.
If user sets \code{parallel = TRUE} and the fitted \code{fit.subgroup()} object uses the parallel version of
an internal model, say for \code{cv.glmnet()}, then the internal parallelization will be overridden so as
not to create a conflict of parallelism.}
}
\description{
Validates subgroup treatment effects for fitted
 subgroup identification model class of Chen, et al (2017)
}
\examples{
library(personalized)

set.seed(123)
n.obs  <- 500
n.vars <- 20
x <- matrix(rnorm(n.obs * n.vars, sd = 3), n.obs, n.vars)


# simulate non-randomized treatment
xbetat   <- 0.5 + 0.5 * x[,11] - 0.5 * x[,13]
trt.prob <- exp(xbetat) / (1 + exp(xbetat))
trt01    <- rbinom(n.obs, 1, prob = trt.prob)

trt      <- 2 * trt01 - 1

# simulate response
delta <- 2 * (0.5 + x[,2] - x[,3] - x[,11] + x[,1] * x[,12])
xbeta <- x[,1] + x[,11] - 2 * x[,12]^2 + x[,13]
xbeta <- xbeta + delta * trt

# continuous outcomes
y <- drop(xbeta) + rnorm(n.obs, sd = 2)

# create function for fitting propensity score model
prop.func <- function(x, trt)
{
    # fit propensity score model
    propens.model <- cv.glmnet(y = trt,
                               x = x, family = "binomial")
    pi.x <- predict(propens.model, s = "lambda.min",
                    newx = x, type = "response")[,1]
    pi.x
}

subgrp.model <- fit.subgroup(x = x, y = y,
                             trt = trt01,
                             propensity.func = prop.func,
                             loss   = "sq_loss_lasso",
                             nfolds = 5)    # option for cv.glmnet

subgrp.model$subgroup.trt.effects

x.test <- matrix(rnorm(10 * n.obs * n.vars, sd = 3), 10 * n.obs, n.vars)


# simulate non-randomized treatment
xbetat.test   <- 0.5 + 0.5 * x.test[,11] - 0.5 * x.test[,13]
trt.prob.test <- exp(xbetat.test) / (1 + exp(xbetat.test))
trt01.test    <- rbinom(10 * n.obs, 1, prob = trt.prob.test)

trt.test      <- 2 * trt01.test - 1

# simulate response
delta.test <- 2 * (0.5 + x.test[,2] - x.test[,3] - x.test[,11] + x.test[,1] * x.test[,12])
xbeta.test <- x.test[,1] + x.test[,11] - 2 * x.test[,12]^2 + x.test[,13]
xbeta.test <- xbeta.test + delta.test * trt.test

y.test <- drop(xbeta.test) + rnorm(10 * n.obs, sd = 2)

valmod <- validate.subgroup(subgrp.model, B = 10,
                            method = "training_test",
                            train.fraction = 0.75)
valmod

bene.score.test <- subgrp.model$predict(x.test)

mean(y.test[bene.score.test > 0 & trt01.test == 1]) -
       mean(y.test[bene.score.test > 0 & trt01.test == 0])
mean(y.test[bene.score.test <= 0 & trt01.test == 0]) -
       mean(y.test[bene.score.test <= 0 & trt01.test == 1])

quantile(valmod$boot.results[[1]][,1], c(0.025, 0.975))
quantile(valmod$boot.results[[1]][,2], c(0.025, 0.975))
}
\references{
Chen, S., Tian, L., Cai, T. and Yu, M. (2017), A general statistical framework for subgroup identification
and comparative treatment scoring. Biometrics. doi:10.1111/biom.12676

Harrell, F. E., Lee, K. L., and Mark, D. B. (1996). Tutorial in biostatistics multivariable prognostic models: issues in developing models,
evaluating assumptions and adequacy, and measuring and reducing errors. Statistics in medicine, 15, 361-387.
doi:10.1002/(SICI)1097-0258(19960229)15:4<361::AID-SIM168>3.0.CO;2-4
}
\seealso{
\code{\link[personalized]{fit.subgroup}} for function which fits subgroup identification models,
\code{\link[personalized]{plot.subgroup_validated}} for plotting of validation results, and
\code{\link[personalized]{print.subgroup_validated}} for arguments for printing options for \code{validate.subgroup()}.
}
