% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_subgroup_fitted.R,
%   R/plot_subgroup_validated.R
\name{plot.subgroup_fitted}
\alias{plot.subgroup_fitted}
\alias{plot.subgroup_validated}
\title{Plotting results for fitted subgroup identification models}
\usage{
\method{plot}{subgroup_fitted}(
  x,
  type = c("boxplot", "density", "interaction", "conditional"),
  avg.line = TRUE,
  ...
)

\method{plot}{subgroup_validated}(
  x,
  type = c("boxplot", "density", "interaction", "conditional", "stability"),
  avg.line = TRUE,
  ...
)
}
\arguments{
\item{x}{fitted object returned by \code{validate.subgroup()} or \code{fit.subgroup()} function}

\item{type}{type of plot. \code{"density"} results in a density plot for the results
across all observations (if \code{x} is from \code{fit.subgroup()}) or if \code{x} is from \code{validate.subgroup()}
across iterations of either the bootstrap or training/test re-fitting. For the latter
case the test results will be plotted. \code{"boxplot"} results in boxplots across all observations/iterations of either
the bootstrap or training/test re-fitting. For the latter
case the test results will be plotted. \code{"interaction"} creates an
interaction plot for the different subgroups (crossing lines here means a meaningful subgroup). For the interaction plot,
the intervals around each point represent +1 one SE
\code{"conditional"} For subgroup_fitted objects, plots smoothed (via a GAM smoother) means of the outcomes as a function of the estimated benefit score
separately for the treated and untreated groups. For subgroup_validated objects, boxplots of summary statistics
within subgroups will be plotted as subgroups are defined by different cutoffs of the benefit scores.
These cutoffs can be specified via the \code{benefit.score.quantiles} argument of
\code{\link[personalized]{validate.subgroup}}.}

\item{avg.line}{boolean value of whether or not to plot a line for the average
value in addition to the density (only valid for \code{type = "density"})}

\item{...}{not used}
}
\description{
Plots results for estimated subgroup treatment effects

Plots validation results for estimated subgroup treatment effects
}
\examples{
library(personalized)

set.seed(123)
n.obs  <- 250
n.vars <- 15
x <- matrix(rnorm(n.obs * n.vars, sd = 3), n.obs, n.vars)


# simulate non-randomized treatment
xbetat   <- 0.5 + 0.5 * x[,11] - 0.5 * x[,13]
trt.prob <- exp(xbetat) / (1 + exp(xbetat))
trt01    <- rbinom(n.obs, 1, prob = trt.prob)

trt      <- 2 * trt01 - 1

# simulate response
delta <- 2 * (0.5 + x[,2] - x[,3] - x[,11] + x[,1] * x[,12])
xbeta <- x[,1] + x[,11] - 2 * x[,12]^2 + x[,13]
xbeta <- xbeta + delta * trt

# continuous outcomes
y <- drop(xbeta) + rnorm(n.obs, sd = 2)

# create function for fitting propensity score model
prop.func <- function(x, trt)
{
    # fit propensity score model
    propens.model <- cv.glmnet(y = trt,
                               x = x, family = "binomial")
    pi.x <- predict(propens.model, s = "lambda.min",
                    newx = x, type = "response")[,1]
    pi.x
}

subgrp.model <- fit.subgroup(x = x, y = y,
                           trt = trt01,
                           propensity.func = prop.func,
                           loss   = "sq_loss_lasso",
                           nfolds = 5)              # option for cv.glmnet

subgrp.model$subgroup.trt.effects

plot(subgrp.model)

plot(subgrp.model, type = "boxplot")

plot(subgrp.model, type = "interaction")

plot(subgrp.model, type = "conditional")

valmod <- validate.subgroup(subgrp.model, B = 3,
                          method = "training_test",
                          benefit.score.quantiles = c(0.25, 0.5, 0.75),
                          train.fraction = 0.75)

plot(valmod)


plot(valmod, type = "interaction")

# see how summary statistics of subgroups change
# when the subgroups are defined based on different cutoffs
# (25th quantile of bene score, 50th, and 75th)
plot(valmod, type = "conditional")

# visualize the frequency of particular variables
# of being selected across the resampling iterations with
# 'type = "stability"'
# not run:
# plot(valmod, type = "stability")

}
\seealso{
\code{\link[personalized]{fit.subgroup}} for function which fits subgroup identification models.

\code{\link[personalized]{validate.subgroup}} for function which creates validation results
and \code{\link[personalized]{fit.subgroup}} for function which fits subgroup identification models.
}
