#' @title get_reverse_depends
#' @author Srikanth KS
#' @description Get reverse dependencies
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer) Depth of recursive dependency
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_reverse_depends}}, \code{\link{get_reverse_imports}},
#'   \code{\link{get_reverse_linkingto}}, \code{\link{get_reverse_suggests}},
#'   \code{\link{get_reverse_enhances}}, \code{\link{get_all_reverse_dependencies}},
#'   \code{\link{get_depends}}
#' @examples
#' pkggraph::init()
#' pkggraph::get_reverse_depends("utils")
#' @export
get_reverse_depends <- function(packages, level = 1L){
  get_dependencies(packages    = packages
                   , type      = "Depends"
                   , level     = level
                   , reverse   = TRUE)
}

#' @title get_reverse_imports
#' @author Srikanth KS
#' @description Get reverse dependencies
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer) Depth of recursive dependency
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_reverse_depends}},
#'   \code{\link{get_reverse_imports}}, \code{\link{get_reverse_linkingto}},
#'   \code{\link{get_reverse_suggests}}, \code{\link{get_reverse_enhances}},
#'   \code{\link{get_all_reverse_dependencies}}, \code{\link{get_imports}}
#' @examples
#' pkggraph::init()
#' pkggraph::get_reverse_imports("Rcpp")
#' @export
get_reverse_imports <- function(packages, level = 1L){
  get_dependencies(packages    = packages
                   , type      = "Imports"
                   , level     = level
                   , reverse   = TRUE)
}

#' @title get_reverse_linkingto
#' @author Srikanth KS
#' @description Get reverse dependencies
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer) Depth of recursive dependency
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_reverse_depends}},
#'   \code{\link{get_reverse_imports}}, \code{\link{get_reverse_linkingto}},
#'   \code{\link{get_reverse_suggests}}, \code{\link{get_reverse_enhances}},
#'   \code{\link{get_all_reverse_dependencies}}, \code{\link{get_linkingto}}
#' @examples
#' pkggraph::init()
#' pkggraph::get_reverse_linkingto("BH")
#' @export
get_reverse_linkingto <- function(packages, level = 1L){
  get_dependencies(packages    = packages
                   , type      = "LinkingTo"
                   , level     = level
                   , reverse   = TRUE)
}

#' @title get_reverse_suggests
#' @author Srikanth KS
#' @description Get reverse dependencies
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer) Depth of recursive dependency
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_reverse_depends}}, \code{\link{get_reverse_imports}},
#'   \code{\link{get_reverse_linkingto}}, \code{\link{get_reverse_suggests}},
#'   \code{\link{get_reverse_enhances}}, \code{\link{get_all_reverse_dependencies}},
#'   \code{\link{get_suggests}}
#' @examples
#' pkggraph::init()
#' pkggraph::get_reverse_suggests("purrr")
#' @export
get_reverse_suggests <- function(packages, level = 1L){
  get_dependencies(packages    = packages
                   , type      = "Suggests"
                   , level     = level
                   , reverse   = TRUE)
}

#' @title get_reverse_enhances
#' @description Get reverse dependencies
#' @author Srikanth KS
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer) Depth of recursive dependency
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_reverse_depends}},
#'   \code{\link{get_reverse_imports}}, \code{\link{get_reverse_linkingto}},
#'   \code{\link{get_reverse_suggests}}, \code{\link{get_reverse_enhances}},
#'   \code{\link{get_all_reverse_dependencies}}, \code{\link{get_enhances}}
#' @examples
#' pkggraph::init()
#' pkggraph::get_reverse_enhances("synchronicity")
#' @export
get_reverse_enhances <- function(packages, level = 1L){
  get_dependencies(packages    = packages
                   , type      = "Enhances"
                   , level     = level
                   , reverse   = TRUE)
}

#' @title get_all_reverse_dependencies
#' @author Srikanth KS
#' @description Get all reverse dependencies
#' @param packages (non-empty character vector) Package names
#' @param level (positive integer, Default = 1L) Depth of recursive dependency
#' @param relation (character vector) Types of relations. Must be a subset of
#'   c("Depends", "Imports", "LinkingTo", "Suggests", "Enhances")
#' @param strict (logical, Default: TRUE) Whether to consider all packages
#'   (alternately only 'relation' specific packages) when computing dependencies
#'   for the next level
#' @param ignore package names to ignore
#' @return A tibble with three columns: `pkg_1`, `relation` and `pkg_2`
#' @seealso \code{\link{get_all_dependencies}}
#' @examples
#' pkggraph::init()
#' # general use
#' pkggraph::get_all_reverse_dependencies("mlr")
#' # specify two levels
#' pkggraph::get_all_reverse_dependencies("mlr", level = 2)
#' # specify relation(s)
#' pkggraph::get_all_reverse_dependencies("mlr", level = 2, relation = "Imports")
#' # setting strict to TRUE to only consider 'Imports' of the previous level
#' pkggraph::get_all_reverse_dependencies("mlr"
#'                                        , level    = 2
#'                                        , relation = "Imports"
#'                                        , strict   = TRUE)
#' @export
get_all_reverse_dependencies <- function(packages
                                         , level = 1L
                                         , relation = c("Depends"
                                                        , "Imports"
                                                        , "LinkingTo"
                                                        , "Suggests"
                                                        , "Enhances"
                                         )
                                         , strict = FALSE
                                         , ignore = c("datasets"
                                                      , "utils"
                                                      , "grDevices"
                                                      , "graphics"
                                                      , "stats"
                                                      , "methods"
                                         )){
  # assertions ----
  if(!exists("deptable")){
    stop("Unable to find `deptable`. Please run `pkggraph::init()`.")
  }
  stopifnot(is.character(packages) && length(packages) > 0)
  stopifnot(length(level) == 1 &&
              level == as.integer(level) &&
              level >= 1L)
  stopifnot(length(relation) > 0 && is.character(relation))
  stopifnot(relation %in% c("Depends"
                            , "Imports"
                            , "LinkingTo"
                            , "Suggests"
                            , "Enhances"))
  stopifnot(is.logical(strict) && length(strict) == 1)
  stopifnot(is.character(ignore))


  # consider 'strict' ----
  if(!strict){
    relation2 <- c("Depends"
                   , "Imports"
                   , "LinkingTo"
                   , "Suggests"
                   , "Enhances")
  } else {
    relation2 <- relation
  }

  # list of dependency frames ----
  ell <- length(relation2)
  for(it in seq_along(1:level)){

    direct_list <- vector("list", length = ell)
    for(it2 in seq_along(1:ell)){
      direct_list[[it2]] <- get_dependencies(packages  = packages
                                             , type    = relation2[it2]
                                             , assert  = FALSE
                                             , reverse = TRUE
                                             )
    }
    res      <- dplyr::bind_rows(direct_list)
    packages <- union(res[["pkg_1"]], res[["pkg_2"]])
  }

  # filter depending on 'strict' ----
  if(!strict){
    res <- res[res$relation %in% relation, ]
  }

  # handle ignore
  if(length(ignore) > 0){
    res <- res[!(res$pkg_1 %in% ignore) & !(res$pkg_2 %in% ignore),]
  }

  # arrange by relation and return ----
  return(dplyr::arrange_(res, "relation"))
}

