## ----, echo=FALSE--------------------------------------------------------
library(knitr)
knit_hooks$set(source = function(x, options){
  if (!is.null(options$verbatim) && options$verbatim){
    opts = gsub(",\\s*verbatim\\s*=\\s*TRUE\\s*", "", options$params.src)
    opts = gsub(",\\s*eval\\s*=\\s*FALSE\\s*", "", opts)
    bef = sprintf('\n\n    ```{r %s}\n', opts, "\n")
    stringr::str_c(
      bef, 
      knitr:::indent_block(paste(x, collapse = '\n'), "    "), 
      "\n    ```\n"
    )
  } else {
    stringr::str_c("\n\n```", tolower(options$engine), "\n", 
      paste(x, collapse = '\n'), "\n```\n\n"
    )
  }
})

## ----load, eval = FALSE--------------------------------------------------
#  install.packages("plotROC")
#  devtools::install_github("sachsmc/plotROC")
#  library(plotROC)

## ----shiny, eval = FALSE-------------------------------------------------
#  shiny_plotROC()

## ----dataset, echo = -1--------------------------------------------------
library(plotROC)
D.ex <- rbinom(100, size = 1, prob = .5)
M.ex <- rnorm(100, mean = D.ex)

## ----calc----------------------------------------------------------------
roc.estimate <- calculate_roc(M.ex, D.ex)
str(roc.estimate)

## ----test-a--------------------------------------------------------------
single.rocplot <- ggroc(roc.estimate, label = "Example")

## ----inter, eval = FALSE-------------------------------------------------
#  plot_interactive_roc(single.rocplot)

## ----int-no, fig.keep='none', results = 'asis', eval = FALSE, verbatim = TRUE----
#  cat(
#    export_interactive_roc(single.rocplot,
#                          prefix = "a")
#    )

## ----int-yes, fig.keep='none', results = 'asis', fig.width=6, fig.height=6----
cat(
  export_interactive_roc(single.rocplot, 
                        prefix = "a")
  )

## ----print, fig.width = 6, fig.height = 6, fig.cap = "Illustration of ROC curve plot generated by plotROC for use in print. "----
plot_journal_roc(single.rocplot)

## ----multistart----------------------------------------------------------
D.ex <- rbinom(100, 1, .5)

paired.data <- data.frame(M1 = rnorm(100, mean = D.ex), 
                       M2 = rnorm(100, mean = D.ex, sd = .4), 
                       M3 = runif(100), D = D.ex)

estimate.list <- calculate_multi_roc(paired.data, c("M1", "M2", "M3"), "D")

## ----multi2, fig.width = 6, fig.height = 6, fig.cap = "Illustration of plot with multiple curves. "----
multi.rocplot <- multi_ggroc(estimate.list, label = c("M1", "M2", "M3"))
plot_journal_roc(multi.rocplot)

## ----multi3, message = FALSE, warning = FALSE, fig.keep='none', results = 'asis', echo = TRUE, fig.width=6, fig.height=6----
colorplot <- multi_ggroc(estimate.list, 
                         xlabel = "1 - Specificity", 
                         ylabel = "Sensitivity")
cat(
  export_interactive_roc(colorplot, lty = rep(1, 3), prefix = "multi3",
                         color = c("black", "purple", "orange"), 
                         legend = TRUE)
  )

## ----test-a-ci-----------------------------------------------------------
roc.ci <- calculate_roc(paired.data$M1, paired.data$D, ci = TRUE, alpha = 0.05)
ci.rocplot <- ggroc(roc.ci, label = "CI Example", ci = TRUE)

## ----int-nob, fig.keep='none', results = 'asis', echo = TRUE, fig.width=6, fig.height=6----
cat(
  export_interactive_roc(ci.rocplot, 
                         prefix = "aci")
  )

## ----printci, fig.width = 6, fig.height = 6, fig.cap = "Illustration of plot with exact confidence regions."----
plot_journal_roc(ci.rocplot, n.cuts = 10, 
                 ci.at = c(-.5, .5, 2.1))

## ----print2, warning = FALSE, message = FALSE, fig.width = 6, fig.height = 6, fig.cap = "Using ggplot2 themes and annotations with plotROC objects."----
library(ggplot2)
plot_journal_roc(ci.rocplot, n.cuts = 10, 
                 ci.at = c(-.5, .5, 2.1)) + 
  theme_grey() + 
  geom_abline(intercept = 0, slope = 1, color = "white") + 
  ggtitle("Themes and annotations") + 
  annotate("text", x = .75, y = .25, 
           label = "AUC = 0.80") +
  scale_x_continuous("1 - Specificity", breaks = seq(0, 1, by = .1))

## ----rocr, fig.keep='none', results = 'asis', echo = TRUE, fig.width=6, fig.height=6----
library(ROCR)
D.ex <- rbinom(100, 1, .5)
M.ex <- rnorm(100, mean = D.ex, sd = .5)

rocr.est <- performance(prediction(M.ex, D.ex), "tpr", "fpr")
rocr.plot <- ggroc(rocr.est, label = "ROCR object")

cat(
  export_interactive_roc(rocr.plot, prefix = "rocr")
  )

## ----binormalsetup-------------------------------------------------------
mu1 <- mean(M.ex[D.ex == 1])
mu0 <- mean(M.ex[D.ex == 0])
s1 <- sd(M.ex[D.ex == 1])
s0 <- sd(M.ex[D.ex == 0])
c.ex <- seq(min(M.ex), max(M.ex), length.out = 300)

binorm.roc <- data.frame(c = c.ex, 
                             FPF = pnorm((mu0 - c.ex)/s0), 
                             TPF = pnorm((mu1 - c.ex)/s1)
                             )

## ----binormal, fig.keep='none', results = 'asis', echo = TRUE, fig.width=6, fig.height=6----
binorm.plot <- ggroc(binorm.roc, label = "Binormal")
cat(
  export_interactive_roc(binorm.plot, prefix = "binorm")
  )

