% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gradient.R
\name{Grad}
\alias{Grad}
\title{Gradient computation with parallel capabilities}
\usage{
Grad(
  FUN,
  x,
  elementwise = NA,
  vectorised = NA,
  multivalued = NA,
  deriv.order = 1L,
  side = 0,
  acc.order = 2,
  stencil = NULL,
  h = NULL,
  zero.tol = sqrt(.Machine$double.eps),
  h0 = NULL,
  control = list(),
  f0 = NULL,
  cores = 1,
  preschedule = TRUE,
  cl = NULL,
  func = NULL,
  method = NULL,
  method.args = list(),
  ...
)
}
\arguments{
\item{FUN}{A function returning a numeric scalar or a vector whose derivatives are to be
computed. If the function returns a vector, the output will be a Jacobian.}

\item{x}{Numeric vector or scalar: the point(s) at which the derivative is estimated.
\code{FUN(x)} must be finite.}

\item{elementwise}{Logical: is the domain effectively 1D, i.e. is this a mapping
\eqn{\mathbb{R} \mapsto \mathbb{R}}{R -> R} or
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^n}{R^n -> R^n}. If \code{NA},
compares the output length ot the input length.}

\item{vectorised}{Logical: if \code{TRUE}, the function
is assumed to be vectorised: it will accept a vector of parameters and return
a vector of values of the same length. Use \code{FALSE} or \code{"no"}  for
functions that take vector arguments and return outputs of arbitrary length (for
\eqn{\mathbb{R}^n \mapsto \mathbb{R}^k}{R^n -> R^k} functions). If \code{NA},
checks the output length and assumes vectorisation if it matches the input length;
this check is necessary and potentially slow.}

\item{multivalued}{Logical: if \code{TRUE}, the function is assumed to return vectors longer
than 1. Use \code{FALSE} for element-wise functions. If \code{NA}, attempts inferring it from
the function output.}

\item{deriv.order}{Integer or vector of integers indicating the desired derivative order,
\eqn{\mathrm{d}^m / \mathrm{d}x^m}{d^m/dx^m}, for each element of \code{x}.}

\item{side}{Integer scalar or vector indicating the type of finite difference:
\code{0} for central, \code{1} for forward, and \code{-1} for backward differences.
Central differences are recommended unless computational cost is prohibitive.}

\item{acc.order}{Integer or vector of integers specifying the desired accuracy order
for each element of \code{x}.
The final error will be of the order \eqn{O(h^{\mathrm{acc.order}})}{O(h^acc.order)}.}

\item{stencil}{Optional custom vector of points for function evaluation.
Must include at least \code{m+1} points for the \code{m}-th order derivative.}

\item{h}{Numeric or character specifying the step size(s) for the numerical
difference or a method of automatic step determination (\code{"CR"}, \code{"CRm"},
\code{"DV"}, or \code{"SW"} to be used in \code{\link[=gradstep]{gradstep()}}). The default value is
described in \code{?GenD}.}

\item{zero.tol}{Small positive integer: if \code{abs(x) >= zero.tol}, then, the automatically
guessed step size is relative (\code{x} multiplied by the step), unless an auto-selection
procedure is requested; otherwise, it is absolute.}

\item{h0}{Numeric scalar of vector: initial step size for automatic search with
\code{gradstep()}.}

\item{control}{A named list of tuning parameters passed to \code{gradstep()}.}

\item{f0}{Optional numeric: if provided, used to determine the vectorisation type
to save time. If FUN(x) must be evaluated (e.g. second derivatives), saves one evaluation.}

\item{cores}{Integer specifying the number of CPU cores used for parallel computation.
Recommended to be set to the number of physical cores on the machine minus one.}

\item{preschedule}{Logical: if \code{TRUE}, disables pre-scheduling for \code{mclapply()}
or enables load balancing with \code{parLapplyLB()}. Recommended for functions that
take less than 0.1 s per evaluation.}

\item{cl}{An optional user-supplied \code{cluster} object  (created by \code{makeCluster}
or similar functions). If not \code{NULL}, the code uses \code{parLapply()} (if \code{preschedule}
is \code{TRUE}) or \code{parLapplyLB()} on that cluster on Windows, and \code{mclapply}
(fork cluster) on everything else.}

\item{func}{For compatibility with \code{numDeriv::grad()} only. If instead of
\code{FUN}, \code{func} is used, it will be reassigned to \code{FUN} with a warning.}

\item{method}{For compatibility with \code{numDeriv::grad()} only. Supported values:
\code{"simple"} and \code{"Richardson"}. Non-null values result in a warning.}

\item{method.args}{For compatibility with \code{numDeriv::grad()} only. Check
\code{?numDeriv::grad} for a list of values. Non-empty lists result in a warning.}

\item{...}{Ignored.}
}
\value{
Numeric vector of the gradient. If \code{FUN} returns a vector,
a warning is issued suggesting the use of \code{\link[=Jacobian]{Jacobian()}}.
}
\description{
Computes numerical derivatives and gradients of scalar-valued functions using
finite differences. This function supports both two-sided (central, symmetric) and
one-sided (forward or backward) derivatives. It can utilise parallel processing
to accelerate computation of gradients for slow functions or
to attain higher accuracy faster.
}
\details{
This function aims to be 100\% compatible with the syntax of \code{numDeriv::Grad()},
but there might be differences in the step size because some choices made in
\code{numDeriv} are not consistent with theory.

There is one feature of the default step size in \code{numDeriv} that deserves
an explanation. In that package (but not in \code{pnd}),

\itemize{
\item If \code{method = "simple"}, then, simple forward differences are used with
a fixed step size \code{eps}, which we denote by \eqn{\varepsilon}{eps}.
\item If \code{method = "Richardson"}, then, central differences are used with
a fixed step
\eqn{h := |d\cdot x| + \varepsilon (|x| < \mathrm{zero.tol})}{h := |d*x| + eps*(|x| < zero.tol)},
where \code{d = 1e-4} is the relative step size and \code{eps} becomes an extra
addition to the step size for the argument that are closer to zero than \code{zero.tol}.
}
We believe that the latter may lead to mistakes when the user believes that they can set
the step size for near-zero arguments, whereas in reality, a combination of \code{d} and \code{eps}
is used.

Here is the synopsis of the old arguments:
\describe{
\item{side}{\code{numDeriv} uses \code{NA} for handling two-sided differences.
The \code{pnd} equivalent is \code{0}, and \code{NA} is replaced with \code{0}.}
\item{eps}{If \code{numDeriv} \code{method = "simple"}, then, \code{eps = 1e-4} is
the absolute step size and forward differences are used.
If \code{method = "Richardson"}, then, \code{eps = 1e-4} is the absolute increment of the step
size for small arguments below the zero tolerance.}
\item{d}{If \code{numDeriv} \code{method = "Richardson"}, then, \code{d*abs(x)} is the
step size for arguments above the zero tolerance and the baseline step size for
small arguments that gets incremented by \code{eps}.}
\item{r}{The number of Richardson extrapolations that successively reduce the initial step size.
For two-sided differences, each extrapolation increases the accuracy order by 2.}
\item{v}{The reduction factor in Richardson extrapolations.}
}

Here are the differences in the new compatible implementation.
\describe{
\item{eps}{If \code{numDeriv} \code{method = "simple"}, then,
\code{ifelse(x!=0, abs(x), 1) * sqrt(.Machine$double.eps) * 2} is used because
one-sided differences require a smaller step size to reduce the truncation error.
If \code{method = "Richardson"}, then, \code{eps = 1e-5}.}
\item{d}{If \code{numDeriv} \code{method = "Richardson"}, then, \code{d*abs(x)} is the
step size for arguments above the zero tolerance and the baseline step size for
small arguments that gets incremented by \code{eps}.}
\item{r}{The number of Richardson extrapolations that successively reduce the initial step size.
For two-sided differences, each extrapolation increases the accuracy order by 2.}
\item{v}{The reduction factor in Richardson extrapolations.}
}

\code{Grad} does an initial check (if \code{f0 = FUN(x)} is not provided)
and calls \code{\link[=GenD]{GenD()}} with a set of appropriate parameters (\code{multivalued = FALSE}
if the check succeds). In case of parameter mismatch, throws and error.
}
\examples{
f <- function(x) sum(sin(x))
g1 <- Grad(FUN = f, x = 1:4)
g2 <- Grad(FUN = f, x = 1:4, h = 7e-6)
g2 - g1  # Tiny differences due to different step sizes
g.auto <- Grad(FUN = f, x = 1:4, h = "SW")
print(g.auto)
attr(g.auto, "step.search")$exitcode  # Success

# Gradients for vectorised functions -- e.g. leaky ReLU
LReLU <- function(x) ifelse(x > 0, x, 0.01*x)
Grad(LReLU, seq(-1, 1, 0.1))
}
\seealso{
\code{\link[=GenD]{GenD()}}, \code{\link[=Jacobian]{Jacobian()}}
}
