% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/object_ops.R
\name{x_write_disk}
\alias{x_write_disk}
\title{Write an \emph{agent}, \emph{informant}, \emph{multiagent}, or table scan to disk}
\usage{
x_write_disk(
  x,
  filename,
  path = NULL,
  keep_tbl = FALSE,
  keep_extracts = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{x}{\emph{One of several types of objects}

\verb{<object>} // \strong{required}

An \emph{agent} object of class \code{ptblank_agent}, an \emph{informant} of class
\code{ptblank_informant}, or an table scan of class \code{ptblank_tbl_scan}.}

\item{filename}{\emph{File name}

\verb{scalar<character>} // \strong{required}

The filename to create on disk for the \code{agent}, \code{informant}, or table scan.}

\item{path}{\emph{File path}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

An optional path to which the file should be saved (this is automatically
combined with \code{filename}).}

\item{keep_tbl}{\emph{Keep data table inside object}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

An option to keep a data table that is associated with the
\emph{agent} or \emph{informant} (which is the case when the \emph{agent}, for example, is
created using \verb{create_agent(tbl = <data table, ...)}). The default is
\code{FALSE} where the data table is removed before writing to disk. For
database tables of the class \code{tbl_dbi} and for Spark DataFrames
(\code{tbl_spark}) the table is always removed (even if \code{keep_tbl} is set to
\code{TRUE}).}

\item{keep_extracts}{\emph{Keep data extracts inside object}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

An option to keep any collected extract data for failing rows. Only applies
to \emph{agent} objects. By default, this is \code{FALSE} (i.e., extract data is
removed).}

\item{quiet}{\emph{Inform (or not) upon file writing}

\verb{scalar<logical>} // \emph{default:} \code{FALSE}

Should the function \emph{not} inform when the file is written?}
}
\value{
Invisibly returns \code{TRUE} if the file has been written.
}
\description{
Writing an \emph{agent}, \emph{informant}, \emph{multiagent}, or even a table scan to disk
with \code{x_write_disk()} can be useful for keeping data validation intel or
table information close at hand for later retrieval (with \code{\link[=x_read_disk]{x_read_disk()}}).
By default, any data table that the \emph{agent} or \emph{informant} may have held
before being committed to disk will be expunged (not applicable to any table
scan since they never hold a table object). This behavior can be changed by
setting \code{keep_tbl} to \code{TRUE} but this only works in the case where the table
is not of the \code{tbl_dbi} or the \code{tbl_spark} class.
}
\details{
It is recommended to set up a table-prep formula so that the \emph{agent} and
\emph{informant} can access refreshed data after being read from disk through
\code{\link[=x_read_disk]{x_read_disk()}}. This can be done initially with the \code{tbl} argument of
\code{\link[=create_agent]{create_agent()}}/\code{\link[=create_informant]{create_informant()}} by passing in a table-prep formula or a
function that can obtain the target table when invoked. Alternatively, we can
use the \code{\link[=set_tbl]{set_tbl()}} with a similarly crafted \code{tbl} expression to ensure that
an \emph{agent} or \emph{informant} can retrieve a table at a later time.
}
\section{Examples}{

\subsection{A: Writing an \code{agent} to disk}{

Let's go through the process of (1) developing an agent with a validation
plan (to be used for the data quality analysis of the \code{\link{small_table}}
dataset), (2) interrogating the agent with the \code{\link[=interrogate]{interrogate()}} function, and
(3) writing the agent and all its intel to a file.

Creating an \code{action_levels} object is a common workflow step when creating a
pointblank agent. We designate failure thresholds to the \code{warn}, \code{stop}, and
\code{notify} states using \code{\link[=action_levels]{action_levels()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )
}\if{html}{\out{</div>}}

Now, let's create a pointblank \code{agent} object and give it the \code{al} object
(which serves as a default for all validation steps which can be overridden).
The data will be referenced in the \code{tbl} argument with a leading \code{~}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <- 
  create_agent(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`x_write_disk()`",
    actions = al
  )
}\if{html}{\out{</div>}}

Then, as with any \code{agent} object, we can add steps to the validation plan by
using as many validation functions as we want. After that, use
\code{\link[=interrogate]{interrogate()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <-
  agent \%>\% 
  col_exists(columns = c(date, date_time)) \%>\%
  col_vals_regex(
    columns = b,
    regex = "[0-9]-[a-z]\{3\}-[0-9]\{3\}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(columns = d, value = 100) \%>\%
  col_vals_lte(columns = c, value = 5) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

The \code{agent} can be written to a file with the \code{x_write_disk()} function.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_write_disk(
  agent,
  filename = "agent-small_table.rds"
)
}\if{html}{\out{</div>}}

We can read the file back as an agent with the \code{\link[=x_read_disk]{x_read_disk()}} function and
we'll get all of the intel along with the restored agent.

If you're consistently writing agent reports when periodically checking data,
we could make use of the \code{\link[=affix_date]{affix_date()}} or \code{\link[=affix_datetime]{affix_datetime()}} depending on
the granularity you need. Here's an example that writes the file with the
format: \code{"<filename>-YYYY-mm-dd_HH-MM-SS.rds"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_write_disk(
  agent,
  filename = affix_datetime(
    "agent-small_table.rds"
  )
)
}\if{html}{\out{</div>}}
}

\subsection{B: Writing an \code{informant} to disk}{

Let's go through the process of (1) creating an informant object that
minimally describes the \code{\link{small_table}} dataset, (2) ensuring that data is
captured from the target table using the \code{\link[=incorporate]{incorporate()}} function, and (3)
writing the informant to a file.

Create a pointblank \code{informant} object with \code{\link[=create_informant]{create_informant()}} and the
\code{\link{small_table}} dataset. Use \code{\link[=incorporate]{incorporate()}} so that info snippets are
integrated into the text.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{informant <- 
  create_informant(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`x_write_disk()`"
  ) \%>\%
  info_snippet(
    snippet_name = "high_a",
    fn = snip_highest(column = "a")
  ) \%>\%
  info_snippet(
    snippet_name = "low_a",
    fn = snip_lowest(column = "a")
  ) \%>\%
  info_columns(
    columns = a,
    info = "From \{low_a\} to \{high_a\}."
  ) \%>\%
  info_columns(
    columns = starts_with("date"),
    info = "Time-based values."
  ) \%>\%
  info_columns(
    columns = date,
    info = "The date part of `date_time`."
  ) \%>\%
  incorporate()
}\if{html}{\out{</div>}}

The \code{informant} can be written to a file with \code{x_write_disk()}. Let's do this
with \code{\link[=affix_date]{affix_date()}} so that the filename has a datestamp.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_write_disk(
  informant,
  filename = affix_date(
    "informant-small_table.rds"
  )
)
}\if{html}{\out{</div>}}

We can read the file back into a new informant object (in the same state as
when it was saved) by using \code{\link[=x_read_disk]{x_read_disk()}}.
}

\subsection{C: Writing a multiagent to disk}{

Let's create one more pointblank agent object, provide it with some
validation steps, and \code{\link[=interrogate]{interrogate()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent_b <-
  create_agent(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`x_write_disk()`",
    actions = al
  ) \%>\%
  col_vals_gt(
    columns = b,
    value = g,
    na_pass = TRUE,
    label = "b > g"
  ) \%>\%
  col_is_character(
    columns = c(b, f),
    label = "Verifying character-type columns" 
  ) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

Now we can combine the earlier \code{agent} object with the newer \code{agent_b} to
create a \code{multiagent}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{multiagent <- create_multiagent(agent, agent_b)
}\if{html}{\out{</div>}}

The \code{multiagent} can be written to a file with the \code{x_write_disk()} function.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_write_disk(
  multiagent,
  filename = "multiagent-small_table.rds"
)
}\if{html}{\out{</div>}}

We can read the file back as a multiagent with the \code{\link[=x_read_disk]{x_read_disk()}} function
and we'll get all of the constituent agents and their associated intel back
as well.
}

\subsection{D: Writing a table scan to disk}{

We can get a report that describes all of the data in the \code{storms} dataset.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl_scan <- scan_data(tbl = dplyr::storms)
}\if{html}{\out{</div>}}

The table scan object can be written to a file with \code{x_write_disk()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_write_disk(
  tbl_scan,
  filename = "tbl_scan-storms.rds"
)
}\if{html}{\out{</div>}}
}
}

\section{Function ID}{

9-1
}

\seealso{
Other Object Ops: 
\code{\link{activate_steps}()},
\code{\link{deactivate_steps}()},
\code{\link{export_report}()},
\code{\link{remove_steps}()},
\code{\link{set_tbl}()},
\code{\link{x_read_disk}()}
}
\concept{Object Ops}
