#' Read a YAML file to create a new agent with a validation plan
#'
#' @description 
#' With `agent_yaml_read()` we can read a **pointblank** YAML file that
#' describes a validation plan to be carried out by an *agent* (typically
#' generated by the [agent_yaml_write()] function. What's returned is a new
#' *agent* with that validation plan, ready to interrogate the target table at
#' will (using the table-reading function stored as the `read_fn`). The agent
#' can be given more validation steps if needed before using [interrogate()] or
#' taking part in any other agent ops (e.g., writing to disk with outputs intact
#' via [agent_write()] or again to **pointblank** YAML with
#' [agent_yaml_write()]).
#'
#' To get a picture of how `agent_yaml_read()` is interpreting the validation
#' plan specified in the **pointblank** YAML, we can use the
#' [agent_yaml_show_exprs()] function. That function shows us (in the console)
#' the **pointblank** expressions for generating the described validation plan.
#' 
#' @param path A path to a YAML file that specifies a validation plan for an
#'   *agent*.
#'   
#' @examples
#' # Let's go through the process of
#' # developing an agent with a validation
#' # plan (to be used for the data quality
#' # analysis of the `small_table` dataset),
#' # and then offloading that validation
#' # plan to a pointblank YAML file; this
#' # will be read in with `agent_yaml_read()`
#' 
#' # We ought to think about what's
#' # tolerable in terms of data quality so
#' # let's designate proportional failure
#' # thresholds to the `warn`, `stop`, and
#' # `notify` states using `action_levels()`
#' al <- 
#'   action_levels(
#'     warn_at = 0.10,
#'     stop_at = 0.25,
#'     notify_at = 0.35
#'   )
#' 
#' # Now create a pointblank `agent` object
#' # and give it the `al` object (which
#' # serves as a default for all validation
#' # steps which can be overridden); the
#' # data will be referenced in a `read_fn`
#' # (a requirement for writing to YAML)
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     name = "example",
#'     actions = al
#'   )
#' 
#' # Then, as with any `agent` object, we
#' # can add steps to the validation plan by
#' # using as many validation functions as we
#' # want
#' agent <-
#'   agent %>% 
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `agent_yaml_write()`
#' # agent_yaml_write(agent, filename = "x.yml")
#' 
#' # The 'x.yml' file is available in the package
#' # through `system.file()`
#' yml_file <- 
#'   system.file("x.yml", package = "pointblank")
#' 
#' # We can view the YAML file in the console
#' # with the `agent_yaml_string()` function
#' agent_yaml_string(path = yml_file)
#' 
#' # The YAML can also be printed in the console
#' # by supplying the agent as the input
#' agent_yaml_string(agent = agent)
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `agent_yaml_read()` function
#' agent <- agent_yaml_read(path = yml_file)
#' 
#' class(agent)
#' 
#' # We can interrogate the data (which
#' # is accessible through the `read_fn`)
#' # with `interrogate()` and get an
#' # agent with intel, or, we can
#' # interrogate directly from the YAML
#' # file with `agent_yaml_interrogate()`
#' agent <- 
#'   agent_yaml_interrogate(path = yml_file)
#' 
#' class(agent)
#' 
#' @family pointblank YAML
#' @section Function ID:
#' 7-2
#' 
#' @export
agent_yaml_read <- function(path) {
  expr_from_agent_yaml(path = path, interrogate = FALSE) %>%
    rlang::parse_expr() %>%
    rlang::eval_tidy()
}

#' Read a YAML file to interrogate a target table immediately
#'
#' The `agent_yaml_interrogate()` function operates much like the
#' [agent_yaml_read()] function (reading a **pointblank** YAML file and
#' generating an *agent* with a validation plan in place). The key difference is
#' that this function takes things a step function and interrogates the target
#' table (defined by table-reading, `read_fn`, function that is required in the
#' YAML file). The additional auto-invocation of [interrogate()] uses the
#' default options of that function. As with [agent_yaml_read()] the agent is
#' returned except, this time, it has intel from the interrogation.
#'
#' @param path A path to a YAML file that specifies a validation plan for an
#'   *agent*.
#'
#' @examples
#' # Let's go through the process of
#' # developing an agent with a validation
#' # plan (to be used for the data quality
#' # analysis of the `small_table` dataset),
#' # and then offloading that validation
#' # plan to a pointblank YAML file; this
#' # will later be read in as a new agent and
#' # the target data will be interrogated
#' # (one step) with `agent_yaml_interrogate()`
#' 
#' # We ought to think about what's
#' # tolerable in terms of data quality so
#' # let's designate proportional failure
#' # thresholds to the `warn`, `stop`, and
#' # `notify` states using `action_levels()`
#' al <- 
#'   action_levels(
#'     warn_at = 0.10,
#'     stop_at = 0.25,
#'     notify_at = 0.35
#'   )
#' 
#' # Now create a pointblank `agent` object
#' # and give it the `al` object (which
#' # serves as a default for all validation
#' # steps which can be overridden); the
#' # data will be referenced in a `read_fn`
#' # (a requirement for writing to YAML)
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     name = "example",
#'     actions = al
#'   )
#' 
#' # Then, as with any `agent` object, we
#' # can add steps to the validation plan by
#' # using as many validation functions as we
#' # want
#' agent <-
#'   agent %>% 
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `agent_yaml_write()`
#' # agent_yaml_write(agent, filename = "x.yml")
#' 
#' # The 'x.yml' file is available in the package
#' # through `system.file()`
#' yml_file <- 
#'   system.file("x.yml", package = "pointblank")
#' 
#' # We can view the YAML file in the console
#' # with the `agent_yaml_string()` function
#' agent_yaml_string(path = yml_file)
#' 
#' # The YAML can also be printed in the console
#' # by supplying the agent as the input
#' agent_yaml_string(agent = agent)
#' 
#' # We can interrogate the data (which
#' # is accessible through the `read_fn`)
#' # through direct use of the YAML file
#' # with `agent_yaml_interrogate()`
#' agent <- 
#'   agent_yaml_interrogate(path = yml_file)
#' 
#' class(agent)
#'
#' # If it's desired to only create a new
#' # agent with the validation plan in place
#' # (stopping short of interrogating the data),
#' # then the `agent_yaml_read()` function
#' # will be useful
#' agent <- agent_yaml_read(path = yml_file)
#' 
#' class(agent)
#'
#' @family pointblank YAML
#' @section Function ID:
#' 7-3
#'
#' @export
agent_yaml_interrogate <- function(path) {
  expr_from_agent_yaml(path = path, interrogate = TRUE) %>%
    rlang::parse_expr() %>%
    rlang::eval_tidy()
}

#' Display *pointblank* expressions using a YAML file with a validation plan
#'
#' The `agent_yaml_show_exprs()` function follows the specifications of a
#' **pointblank** YAML file to generate and show the **pointblank** expressions
#' for generating the described validation plan. The expressions are shown in
#' the console, providing an opportunity to copy the statements and extend as
#' needed. A **pointblank** YAML file can itself be generated by using the
#' [agent_yaml_write()] function with a pre-existing *agent*, or, it can be
#' carefully written by hand.
#'
#' @param path A path to a YAML file that specifies a validation plan for an
#'   *agent*.
#' 
#' @examples 
#' # Let's create a validation plan for the
#' # data quality analysis of the `small_table`
#' # dataset; we need an agent and its
#' # table-reading function enables retrieval
#' # of the target table
#' agent <- 
#'   create_agent(
#'     read_fn = ~small_table,
#'     name = "example",
#'     actions = action_levels(
#'       warn_at = 0.10,
#'       stop_at = 0.25,
#'       notify_at = 0.35
#'     )
#'   ) %>%
#'   col_exists(vars(date, date_time)) %>%
#'   col_vals_regex(
#'     vars(b), "[0-9]-[a-z]{3}-[0-9]{3}"
#'   ) %>%
#'   rows_distinct() %>%
#'   col_vals_gt(vars(d), 100) %>%
#'   col_vals_lte(vars(c), 5)
#'
#' # The agent can be written to a pointblank
#' # YAML file with `agent_yaml_write()`
#' # agent_yaml_write(agent, filename = "x.yml")
#' 
#' # The 'x.yml' file is available in the package
#' # through `system.file()`
#' yml_file <- 
#'   system.file("x.yml", package = "pointblank")
#' 
#' # At a later time, the YAML file can
#' # be read into a new agent with the
#' # `agent_yaml_read()` function
#' agent <- agent_yaml_read(path = yml_file)
#' 
#' class(agent)
#' 
#' # To get a sense of which expressions are
#' # being used to generate the new agent, we
#' # can use `agent_yaml_show_exprs()`
#' agent_yaml_show_exprs(path = yml_file)
#'   
#' @family pointblank YAML
#' @section Function ID:
#' 7-5
#'
#' @export
agent_yaml_show_exprs <- function(path) {
  message(expr_from_agent_yaml(path = path, interrogate = FALSE))
}

expr_from_agent_yaml <- function(path,
                                 interrogate = FALSE) {

  # Read the YAML file with `yaml::read_yaml()`
  y <- yaml::read_yaml(file = path)
  
  # Get the `name` and `read_fn` fields from the YAML
  # file and create argument strings
  read_fn <- paste0("  read_fn = ", y$read_fn)
  name <- paste0("  name = \"", y$name, "\"")
  
  # Create argument strings for the `actions` and
  # `end_fns` arguments (which could be NULL)
  actions <- make_action_levels_str(y$actions)
  end_fns <- make_end_fns_str(y$end_fns)
  
  # The `embed_report` and `reporting_lang` values are
  # taken from the YAML (if they exist) and transformed
  # to argument strings
  if (!is.null(y$embed_report) && y$embed_report) {
    embed_report <- paste0("  embed_report = ", y$embed_report)
  } else {
    embed_report <- NULL
  }
  if (!is.null(y$reporting_lang) && y$reporting_lang != "en") {
    reporting_lang <- paste0("  reporting_lang = \"", y$reporting_lang, "\"")
  } else {
    reporting_lang <- NULL
  }
  
  # Generate all of the validation steps that make up
  # the agent's validation plan
  validation_steps <- make_validation_steps(y$steps)
  
  # Generate the expression string
  expr_str <-
    paste0(
      "create_agent(\n",
      paste(
        c(read_fn, name, actions, end_fns, embed_report, reporting_lang),
        collapse = ",\n"
      ),
      "\n) ",
      validation_steps
    )

  # Add the `interrogate()` statement if needed (this is
  # for the `agent_yaml_interrogate()` function)
  if (interrogate) {
    expr_str <- paste0(expr_str, "%>%\ninterrogate()")
  }
  
  expr_str
}

make_action_levels_str <- function(al) {
  
  if (is.null(al)) {
    return(NULL)
  }

  top_args <- c()
  
  if (!is.null(al$warn_fraction) || !is.null(al$warn_count)) {
    top_args <- c(top_args, paste0("warn_at = ", c(al$warn_fraction, al$warn_count)))
  }
  if (!is.null(al$stop_fraction) || !is.null(al$stop_count)) {
    top_args <- c(top_args, paste0("stop_at = ", c(al$stop_fraction, al$stop_count)))
  }
  if (!is.null(al$notify_fraction) || !is.null(al$notify_count)) {
    top_args <- c(top_args, paste0("notify_at = ", c(al$notify_fraction, al$notify_count)))
  }

  fns_args <- c()
  
  if (!is.null(al$fns$warn)) {
    fns_args <- c(fns_args, paste0("warn = ", al$fns$warn))
  }
  if (!is.null(al$fns$stop)) {
    fns_args <- c(fns_args, paste0("stop = ", al$fns$stop))
  }
  if (!is.null(al$fns$notify)) {
    fns_args <- c(fns_args, paste0("notify = ", al$fns$notify))
  }
  
  if (length(fns_args) > 0) {
    fns_args <- paste0("    fns = list(\n", paste0("      ", fns_args), "\n    )")
  }
  
  paste0(
    "  actions = action_levels(\n",
    paste0(
      c(paste0("    ", top_args, collapse = ",\n"), fns_args),
      collapse = ",\n"
    ),
    "\n  )"
  )
}

make_end_fns_str <- function(end_fns) {
  
  if (is.null(end_fns)) {
    return(NULL)
  }
  
  paste0(
    "  end_fns = list(\n",
      c(paste0("    ", end_fns, collapse = ",\n")),
    "\n  )"
  )
}

make_validation_steps <- function(steps) {
  
  if (length(steps) == 0) return("")
  
  str_exprs <- 
    lapply(
      seq_along(steps),
      FUN = function(x) { 
        
        step_i <- steps[[x]]
        step_fn <- names(step_i)
        
        args <- 
          vapply(
            seq_along(step_i[[1]]),
            FUN.VALUE = character(1), 
            FUN = function(x) {
              
              arg_name <- names(step_i[[1]][x])
              val <- step_i[[1]][[x]]
              
              if (arg_name == "fns") {
                return(paste("  ", val, collapse = ",\n"))
              }
              
              # Return empty string if seeing default values
              if (arg_name == "active" && val) {
                return("")
              }
              if (arg_name == "preconditions" && is.null(val)) {
                return("")
              }
              if (arg_name == "na_pass" && !val) {
                return("")
              }
              if (arg_name == "inclusive" && all(val)) {
                return("")
              }
              
              if (is.null(val[1])) {
                
                val <- "NULL"
                
              } else if (!is.null(val[1]) && is.logical(val[1])) {
                
                val <- as.character(val)
                
              } else if (!is.null(val[1]) && arg_name == "actions") {

                return(make_action_levels_str(val))
                
              } else if (!is.null(val[1]) && arg_name == "schema") {

                vals <- 
                  vapply(
                    val,
                    FUN.VALUE = character(1),
                    USE.NAMES = FALSE,
                    FUN = function(x) {
                      if (length(x) > 1) {
                        val <- paste0("c(", paste0(paste0("\"", as.character(x), "\""), collapse = ", "), ")")
                      } else {
                        val <- paste0("\"", as.character(x), "\"")
                      }
                      val
                    }
                  )

                val <- 
                  paste0("  schema = col_schema(\n",
                    paste("   ", names(val), "=", vals, collapse = ",\n"),
                    "\n  )"
                  )
                
                return(val)
                
              } else if (!is.null(val[1]) &&
                         is.character(val[1]) &&
                         !grepl("^vars\\(.*?\\)$", val[1]) &&
                         !(arg_name %in% c("preconditions", "expr", "schema"))) {
                
                val <- paste0("\"", val, "\"")
              }
              
              if (length(val) > 1) {
                val <- paste0("c(", paste(as.character(val), collapse = ", "), ")")
              } else {
                val <- as.character(val)
              }
              
              paste(" ", arg_name[1], "=", val[1])
            }
          )
        
        args <- args[args != ""]
        
        args %>% 
          paste(collapse = ",\n") %>%
          paste0("%>%\n", step_fn, "(\n", ., "\n)")
      }
    ) %>% 
    unlist() %>%
    paste(collapse = " ") %>%
    paste0(., " ")
  
  gsub("rows_distinct(\n  columns = NULL\n)", "rows_distinct()", str_exprs, fixed = TRUE)
}
