% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MDP_policy_functions.R
\name{MDP_policy_functions}
\alias{MDP_policy_functions}
\alias{q_values_MDP}
\alias{MDP_policy_evaluation}
\alias{greedy_MDP_action}
\alias{random_MDP_policy}
\alias{manual_MDP_policy}
\alias{greedy_MDP_policy}
\title{Functions for MDP Policies}
\usage{
q_values_MDP(model, U = NULL)

MDP_policy_evaluation(
  pi,
  model,
  U = NULL,
  k_backups = 1000,
  theta = 0.001,
  verbose = FALSE
)

greedy_MDP_action(s, Q, epsilon = 0, prob = FALSE)

random_MDP_policy(model, prob = NULL)

manual_MDP_policy(model, actions)

greedy_MDP_policy(Q)
}
\arguments{
\item{model}{an MDP problem specification.}

\item{U}{a vector with value function representing the state utilities
(expected sum of discounted rewards from that point on).
If \code{model} is a solved model, then the state
utilities are taken from the solution.}

\item{pi}{a policy as a data.frame with at least columns for states and action.}

\item{k_backups}{number of look ahead steps used for approximate policy evaluation
used by the policy iteration method. Set k_backups to \code{Inf} to only use
\eqn{\theta} as the stopping criterion.}

\item{theta}{stop when the largest change in a state value is less
than \eqn{\theta}.}

\item{verbose}{logical; should progress and approximation errors be printed.}

\item{s}{a state.}

\item{Q}{an action value function with Q-values as a state by action matrix.}

\item{epsilon}{an \code{epsilon > 0} applies an epsilon-greedy policy.}

\item{prob}{probability vector for random actions for \code{random_MDP_policy()}.
a logical indicating if action probabilities should be returned for
\code{greedy_MDP_action()}.}

\item{actions}{a vector with the action (either the action label or the
numeric id) for each state.}
}
\value{
\code{q_values_MDP()} returns a state by action matrix specifying the Q-function,
i.e., the action value for executing each action in each state. The Q-values
are calculated from the value function (U) and the transition model.

\code{MDP_policy_evaluation()} returns a vector with (approximate)
state values (U).

\code{greedy_MDP_action()} returns the action with the highest q-value
for state \code{s}. If \code{prob = TRUE}, then a vector with
the probability for each action is returned.

\code{random_MDP_policy()} returns a data.frame with the columns state and action to define a policy.

\code{manual_MDP_policy()} returns a data.frame with the columns state and action to define a policy.

\code{greedy_MDP_policy()} returns the greedy policy given \code{Q}.
}
\description{
Implementation several functions useful to deal with MDP policies.
}
\details{
Implemented functions are:
\itemize{
\item \code{q_values_MDP()} calculates (approximates)
Q-values for a given model using the Bellman
optimality equation:

\deqn{q(s,a) = \sum_{s'} T(s'|s,a) [R(s,a) + \gamma U(s')]}

Q-values can be used as the input for several other functions.
\item \code{MDP_policy_evaluation()} evaluates a policy \eqn{\pi} for a model and returns
(approximate) state values by applying the Bellman equation as an update
rule for each state and iteration \eqn{k}:

\deqn{U_{k+1}(s) =\sum_a \pi{a|s} \sum_{s'} T(s' | s,a) [R(s,a) + \gamma U_k(s')]}

In each iteration, all states are updated. Updating is stopped after
\code{k_backups} iterations or after the
largest update \eqn{||U_{k+1} - U_k||_\infty < \theta}.
\item \code{greedy_MDP_action()} returns the action with the largest Q-value given a
state.
\item \code{random_MDP_policy()}, \code{manual_MDP_policy()}, and \code{greedy_MDP_policy()}
generates different policies. These policies can be added to a problem
using \code{\link[=add_policy]{add_policy()}}.
}
}
\examples{
data(Maze)
Maze

# create several policies:
# 1. optimal policy using value iteration
maze_solved <- solve_MDP(Maze, method = "value_iteration")
maze_solved
pi_opt <- policy(maze_solved)
pi_opt
gridworld_plot_policy(add_policy(Maze, pi_opt), main = "Optimal Policy")

# 2. a manual policy (go up and in some squares to the right)
acts <- rep("up", times = length(Maze$states))
names(acts) <- Maze$states
acts[c("s(1,1)", "s(1,2)", "s(1,3)")] <- "right"
pi_manual <- manual_MDP_policy(Maze, acts)
pi_manual
gridworld_plot_policy(add_policy(Maze, pi_manual), main = "Manual Policy")

# 3. a random policy
set.seed(1234)
pi_random <- random_MDP_policy(Maze)
pi_random
gridworld_plot_policy(add_policy(Maze, pi_random), main = "Random Policy")

# 4. an improved policy based on one policy evaluation and
#   policy improvement step.
u <- MDP_policy_evaluation(pi_random, Maze)
q <- q_values_MDP(Maze, U = u)
pi_greedy <- greedy_MDP_policy(q)
pi_greedy
gridworld_plot_policy(add_policy(Maze, pi_greedy), main = "Greedy Policy")

#' compare the approx. value functions for the policies (we restrict
#'    the number of backups for the random policy since it may not converge)
rbind(
  random = MDP_policy_evaluation(pi_random, Maze, k_backups = 100),
  manual = MDP_policy_evaluation(pi_manual, Maze),
  greedy = MDP_policy_evaluation(pi_greedy, Maze),
  optimal = MDP_policy_evaluation(pi_opt, Maze)
)

# For many functions, we first add the policy to the problem description
#   to create a "solved" MDP
maze_random <- add_policy(Maze, pi_random)
maze_random

# plotting
plot_value_function(maze_random)
gridworld_plot_policy(maze_random)

# compare to a benchmark
regret(maze_random, benchmark = maze_solved)

# calculate greedy actions for state 1
q <- q_values_MDP(maze_random)
q
greedy_MDP_action(1, q, epsilon = 0, prob = FALSE)
greedy_MDP_action(1, q, epsilon = 0, prob = TRUE)
greedy_MDP_action(1, q, epsilon = .1, prob = TRUE)
}
\references{
Sutton, R. S., Barto, A. G. (2020). Reinforcement Learning: An Introduction.
Second edition. The MIT Press.
}
\seealso{
Other MDP: 
\code{\link{MDP}()},
\code{\link{MDP2POMDP}},
\code{\link{accessors}},
\code{\link{actions}()},
\code{\link{add_policy}()},
\code{\link{gridworld}},
\code{\link{reachable_and_absorbing}},
\code{\link{regret}()},
\code{\link{simulate_MDP}()},
\code{\link{solve_MDP}()},
\code{\link{transition_graph}()},
\code{\link{value_function}()}
}
\author{
Michael Hahsler
}
\concept{MDP}
