% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bitwise.r
\name{bitwise.dist}
\alias{bitwise.dist}
\title{Calculate dissimilarity or Euclidean distance for genlight objects}
\usage{
bitwise.dist(
  x,
  percent = TRUE,
  mat = FALSE,
  missing_match = TRUE,
  scale_missing = FALSE,
  euclidean = FALSE,
  differences_only = FALSE,
  threads = 0L
)
}
\arguments{
\item{x}{a \link[adegenet:genlight]{genlight} or \link[=snpclone-class]{snpclone} object.}

\item{percent}{\code{logical}. Should the distance be represented from 0 to
1? Default set to \code{TRUE}. \code{FALSE} will return the distance
represented as integers from 1 to n where n is the number of loci.
This option has no effect if \code{euclidean = TRUE}}

\item{mat}{\code{logical}. Return a matrix object. Default set to
\code{FALSE}, returning a dist object. \code{TRUE} returns a matrix object.}

\item{missing_match}{\code{logical}. Determines whether two samples differing
by missing data in a location should be counted as matching at that
location. Default set to \code{TRUE}, which forces missing data to match
with anything. \code{FALSE} forces missing data to not match with any other
information, \strong{including other missing data}.}

\item{scale_missing}{A logical. If \code{TRUE}, comparisons with missing
data is scaled up proportionally to the number of columns used by
multiplying the value by \code{m / (m - x)} where m is the number of
loci and x is the number of missing sites. This option matches the behavior
of base R's \code{\link[=dist]{dist()}} function.
Defaults to \code{FALSE}.}

\item{euclidean}{\code{logical}. if \code{TRUE}, the Euclidean distance will
be calculated.}

\item{differences_only}{\code{logical}. When \code{differences_only = TRUE},
the output will reflect the number of different loci. The default setting,
\code{differences_only = FALSE}, reflects the number of different alleles.
Note: this has no effect on haploid organisms since 1 locus = 1 allele.
This option is NOT recommended.}

\item{threads}{The maximum number of parallel threads to be used within this
function. A value of 0 (default) will attempt to use as many threads as
there are available cores/CPUs. In most cases this is ideal. A value of 1
will force the function to run serially, which may increase stability on
some systems. Other values may be specified, but should be used with
caution.}
}
\value{
A dist object containing pairwise distances between samples.
}
\description{
This function calculates both dissimilarity and Euclidean distances for
\link[adegenet:genlight]{genlight} or \link[=snpclone-class]{snpclone} objects.
}
\details{
The default distance calculated here is quite simple and goes by
many names depending on its application. The most familiar name might be
the Hamming distance, or the number of differences between two strings.

As of poppr version 2.8.0, this function now also calculates Euclidean
distance and is considerably faster and more memory-efficient than the
standard \code{dist()} function.
}
\note{
This function is optimized for \link[adegenet:genlight]{genlight} and
\link[=snpclone-class]{snpclone} objects. This does not mean that it is a
catch-all optimization for SNP data. Three assumptions must be met for this
function to work:
\enumerate{
\item SNPs are bi-allelic
\item Samples are haploid or diploid
\item All samples have the same ploidy
}

If the user supplies a \link[adegenet:genind]{genind} or
\link[=genclone-class]{genclone} object, \code{\link[=prevosti.dist]{prevosti.dist()}} will be used for
calculation.
}
\examples{
set.seed(999)
x <- glSim(n.ind = 10, n.snp.nonstruc = 5e2, n.snp.struc = 5e2, ploidy = 2)
x
# Assess fraction of different alleles
system.time(xd <- bitwise.dist(x, threads = 1L))
xd

# Calculate Euclidean distance
system.time(xdt <- bitwise.dist(x, euclidean = TRUE, scale_missing = TRUE, threads = 1L))
xdt

\dontrun{

# This function is more efficient in both memory and speed than [dist()] for
# calculating Euclidean distance on genlight objects. For example, we can
# observe a clear speed increase when we attempt a calculation on 100k SNPs
# with 10\% missing data:

set.seed(999)
mat <- matrix(sample(c(0:2, NA), 
                     100000 * 50, 
                     replace = TRUE, 
                     prob = c(0.3, 0.3, 0.3, 0.1)),
              nrow = 50)
glite <- new("genlight", mat, ploidy = 2)

# Default Euclidean distance 
system.time(dist(glite))

# Bitwise dist
system.time(bitwise.dist(glite, euclidean = TRUE, scale_missing = TRUE))

}
}
\seealso{
\code{\link[=diss.dist]{diss.dist()}}, \link[=snpclone-class]{snpclone},
\link[adegenet:genlight]{genlight}, \code{\link[=win.ia]{win.ia()}},  \code{\link[=samp.ia]{samp.ia()}}
}
\author{
Zhian N. Kamvar, Jonah C. Brooks
}
