% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setup.R
\name{unequal_clusters}
\alias{unequal_clusters}
\title{Setup unbalanced cluster sizes}
\usage{
unequal_clusters(..., func = NULL, trunc = 1, replace = 1)
}
\arguments{
\item{...}{Any number of separate numeric arguments specifying
each cluster's size}

\item{func}{A function that generates cluster sizes, used instead of \code{...}. See \emph{Details}.}

\item{trunc}{Cutoff for values generated by \code{func}, \code{x < trunc} are replaced,
used to avoid negative or 0 values.}

\item{replace}{Indicates what value to replace cluster sizes less than \code{trunc} with.}
}
\value{
An object of type 'plcp_unequal_clusters'
}
\description{
Helps specifying unequal cluster sizes with \code{\link{study_parameters}}
}
\details{
If \code{func} is used together with a function that generates random draws, e.g.
\code{rnorm} or \code{rpois}, then cluster sizes  (and possibly the number of clusters),
will be treated as a random variable. The expected power is then reported by averaging over
multiple realizations of the random variables.

Unless \code{per_treatment} is used, then the same realization of random cluster sizes
will be used in both groups. To use independent realizations from the same distribution for
each treatment group, simply combine the \code{unequal_clusters} with \code{per_treatment}.
}
\examples{
library(dplyr)
n2 <- unequal_clusters(5, 10, 15, 40)
p <- study_parameters(n1 = 11,
                      n2 = n2,
                      n3 = 6,
                      T_end = 10,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      sigma_error = 1,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      cohend = -0.8)

# verify cluster sizes
d <- simulate_data(p)
d \%>\%
    filter(time == 0) \%>\%
    group_by(treatment, cluster) \%>\%
    summarise(n = n())

# Poisson distributed cluster sizes, same in both groups
n2 <- unequal_clusters(func = rpois(n = 5, lambda = 5))
p <- study_parameters(n1 = 11,
                      n2 = n2,
                      T_end = 10,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      sigma_error = 1,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      cohend = -0.8)

# Independent draws from same dist
n2 <- unequal_clusters(func = rpois(n = 5, lambda = 5))
p <- study_parameters(n1 = 11,
                      n2 = per_treatment(n2, n2),
                      T_end = 10,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      sigma_error = 1,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      cohend = -0.8)

# Use per_treatment() to specify per treatment ------------------------------
n2 <- per_treatment(unequal_clusters(2, 2, 2, 2, 3, 4, 5),
                     unequal_clusters(10, 15))
p <- study_parameters(n1 = 11,
                      n2 = n2,
                      n3 = 3,
                      T_end = 10,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      cohend = -0.8)

# verify cluster sizes
d <- simulate_data(p)
d \%>\%
    filter(time == 0) \%>\%
    group_by(treatment, cluster) \%>\%
    summarise(n = n())
}
\seealso{
\code{\link{per_treatment}}
}
