% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setup.R
\name{study_parameters}
\alias{study_parameters}
\title{Setup study parameters}
\usage{
study_parameters(n1, n2, n3 = 1, T_end = NULL, fixed_intercept = 0L,
  fixed_slope = 0L, sigma_subject_intercept = NULL,
  sigma_subject_slope = NULL, sigma_cluster_intercept = NULL,
  sigma_cluster_slope = NULL, sigma_error = 10, cor_subject = 0L,
  cor_cluster = 0L, cor_within = 0L, var_ratio = NULL, icc_slope = NULL,
  icc_pre_subject = NULL, icc_pre_cluster = NULL, effect_size = 0L,
  cohend = NULL, partially_nested = FALSE, dropout = 0L,
  deterministic_dropout = TRUE)
}
\arguments{
\item{n1}{Number of level 1 units, e.g. measurements per subject.}

\item{n2}{Number of level 2 units per level 3 unit, e.g. subjects per cluster.
Unbalanced cluster sizes are supported, see \code{\link{unequal_clusters}}.}

\item{n3}{Number of level 3 units per treatment, can be different in each
treatment arm, see \code{\link{per_treatment}}.}

\item{T_end}{Time point of the last measurement. If \code{NULL} it will be set
to \code{n1 - 1}.}

\item{fixed_intercept}{Average baseline value, assumed to be equal for both groups.}

\item{fixed_slope}{Overall change per unit time, in the control group.}

\item{sigma_subject_intercept}{Subject-level random intercept.}

\item{sigma_subject_slope}{Subject-level random slope.}

\item{sigma_cluster_intercept}{Cluster-level random intercept.}

\item{sigma_cluster_slope}{Cluster-level random slope.}

\item{sigma_error}{Within-subjects (residual) variation.}

\item{cor_subject}{Correlation between the subject-level random intercept
and slopes.}

\item{cor_cluster}{Correlation between the cluster-level random intercept
and slopes.}

\item{cor_within}{Correlation of the level 1 residual. Currently ignored in
the analytical power calculations.}

\item{var_ratio}{Ratio of the random
slope variance to the within-subject variance.}

\item{icc_slope}{Proportion of slope variance
at the cluster level.}

\item{icc_pre_subject}{Amount of baseline
variance at the subject level. N.B. the variance at the subject-level also
included the cluster-level variance. If there's no random slopes, this would
be the subject-level ICC, i.e. correlation between time points.}

\item{icc_pre_cluster}{Amount of baseline
variance at the cluster level.}

\item{effect_size}{The treatment effect. Either a \code{numeric} indicating the mean
difference (unstandardized) between the treatments at posttest, or a standardized effect
using the \code{\link{cohend}} helper function.}

\item{cohend}{\emph{Deprecated}; now act as a shortcut to \code{\link{cohend}} helper function.
Equivalent to using \code{effect_size = cohend(cohend, standardizer = "pretest_SD", treatment = "control")}}

\item{partially_nested}{\code{logical}; indicates if there's clustering in both
arms or only in the treatment arm.}

\item{dropout}{Dropout process, see \code{\link{dropout_weibull}} or
\code{\link{dropout_manual}}. Assumed to be 0 if \code{NULL}.}

\item{deterministic_dropout}{\code{logical}; if \code{FALSE} the input to
\code{dropout} will be treated as random and dropout will be sampled
from a multinomial distribution. N.B.: the random dropout will be
sampled independently in both treatment arms.}
}
\value{
A \code{list} or \code{data.frame} of parameters values, either of
class \code{plcp} or \code{plcp_multi} if multiple parameters are compared.
}
\description{
Setup the parameters for calculating power for longitudinal multilevel studies
comparing two groups. Ordinary two-level models (subjects with repeated measures),
and longitudinal three-level models with clustering due to therapists, schools, provider etc,
are supported. Random slopes at the subject level and cluster level are
possible. Cluster sizes can be unbalanced, and vary by treatment.
Partially nested designs are supported. Missing data can also be accounted
for.
}
\details{
\bold{Comparing a combination of parameter values}

It is possible to setup a grid of parameter combinations by entering the values
as vectors. All unique combinations of the inputs will be returned. This is
useful if you want see how different values of the parameters affect power.
See also the convenience function \code{\link{get_power_table}}.

\bold{Standardized and unstandardized inputs}

All parameters of the models can be specified. However, many of the raw
parameter values in a multilevel/LMM do no directly affect the power of the
test of the \code{treatment:time}-coefficient. Power will depend greatly on the relative
size of the parameters, therefore, it is possible to setup your calculations
using only standardized inputs, or by a combination of raw inputs and
standardized inputs. For instance, if \code{sigma_subject_slope} and
\code{icc_slope} is specified, the \code{sigma_cluster_slope} will be
solved for. Only the cluster-level parameters can be solved when standardized and
raw values are mixed. \code{sigma_error} is 10 by default. More information regarding
the standardized inputs are available in the two-level and three-level vignettes.

\bold{Difference between 0 and NA}

For the variance components \code{0} and \code{NA/NULL} have different meanings.
A parameter that is 0 is still kept in the model, e.g. if \code{icc_pre_cluster = 0}
a random intercept is estimated at the cluster level, but the true value is 0.
If the argument is either \code{NULL} or \code{NA} it is excluded from the model.
This choice will matter when running simulations, or if Satterthwaite \emph{dfs} are used.

The default behavior if a parameters is not specified is that \code{cor_subject} and
\code{cor_cluster} is 0, and the other variance components are \code{NULL}.

\bold{Effect size and Cohen's d}

The argument \code{effect_size} let's you specify the average difference in change
between the treatment groups. You can either pass a \code{numeric} value to define
the raw difference in means at posttest, or use a standardized effect size, see
\code{\link{cohend}} for more details on the standardized effects.

The argument \code{cohend} is kept for legacy reasons, and is equivalent to using
\code{effect_size = cohend(cohend, standardizer = "pretest_SD", treatment = "control")}.

\bold{Two- or three-level models}

If either \code{sigma_cluster_slope} or \code{icc_slope} and
 \code{sigma_cluster_intercept} or \code{icc_pre_cluster} is
\code{NULL} it will be assumed a two-level design is wanted.

\bold{Unequal cluster sizes and unbalanced allocation}

It is possible to specify different cluster sizes using
\code{\link{unequal_clusters}}. Cluster sizes can vary between treatment arms
by also using \code{\link{per_treatment}}. The number of clusters per treatment can
also be set by using \code{\link{per_treatment}}. Moreover, cluster
sizes can be sampled from a distribution, and treated as a random variable.
See \code{\link{per_treatment}} and \code{\link{unequal_clusters}} for examples of their use.

\bold{Missing data and dropout}

Accounting for missing data in the power calculations is possible. Currently,
\code{dropout} can be specified using either \code{\link{dropout_weibull}} or
\code{\link{dropout_manual}}. It is possible to have different dropout
patterns per treatment group using \code{\link{per_treatment}}. See their
respective help pages for examples of their use.

If \code{deterministic_dropout = TRUE} then the proportion of dropout is treated is fixed.
However, exactly which subjects dropout is randomly sampled within treatments. Thus,
clusters can become slightly unbalanced, but generally power varies little over realizations.

For \emph{random dropout}, \code{deterministic_dropout = FALSE}, the proportion
of dropout is converted to the probability of having exactly \emph{i} measurements,
and the actual dropout is sampled from a multinomial distribution. In this case, the proportion of
dropout varies over the realizations from the multinomial distribution, but will
match the dropout proportions in expectation. The random dropout in
each treatment group is sampled from independent multinomial distributions.

Generally, power based on fixed dropout is a good approximation of random dropout.
}
\examples{
# Three level model with both subject- and cluster-level random slope
# Power calculation using standardized inputs
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      var_ratio = 0.03,
                      icc_slope = 0.05,
                      effect_size = cohend(-0.8))

get_power(p)

# The same calculation with all parameters specified directly
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      T_end = 10,
                      fixed_intercept = 37,
                      fixed_slope = -0.65,
                      sigma_subject_intercept = 2.8,
                      sigma_subject_slope = 0.4726944,
                      sigma_cluster_intercept = 0,
                      sigma_cluster_slope = 0.1084435,
                      sigma_error = 2.8,
                      cor_subject = -0.5,
                      cor_cluster = 0,
                      effect_size = cohend(-0.8))
get_power(p)

# Standardized and unstandardized inputs
p <- study_parameters(n1 = 11,
                      n2 = 5,
                      n3 = 4,
                      sigma_subject_intercept = 2.8,
                      icc_pre_cluster = 0.07,
                      sigma_subject_slope = 0.47,
                      icc_slope = 0.05,
                      sigma_error = 2.8,
                      effect_size = cohend(-0.8))

get_power(p)

## Two-level model with subject-level random slope
p <- study_parameters(n1 = 11,
                      n2 = 40,
                      icc_pre_subject = 0.5,
                      var_ratio = 0.03,
                      effect_size = cohend(-0.8))
get_power(p)

# add missing data
p <- update(p, dropout = dropout_weibull(0.2, 1))
get_power(p)

## Comparing a combination of values
p <- study_parameters(n1 = 11,
                      n2 = c(5, 10),
                      n3 = c(2, 4),
                      icc_pre_subject = 0.5,
                      icc_pre_cluster = 0,
                      var_ratio = 0.03,
                      icc_slope = c(0, 0.05),
                      effect_size = cohend(c(-0.5, -0.8))
                      )

get_power(p)
}
\seealso{
\code{\link{cohend}}, \code{\link{get_power}}, \code{\link{simulate.plcp}}
}
