% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/add_cbc_solver.R
\name{add_cbc_solver}
\alias{add_cbc_solver}
\title{Add a \emph{CBC} solver}
\usage{
add_cbc_solver(
  x,
  gap = 0.1,
  time_limit = .Machine$integer.max,
  presolve = TRUE,
  threads = 1,
  first_feasible = FALSE,
  start_solution = NULL,
  verbose = TRUE
)
}
\arguments{
\item{x}{\code{\link[=problem]{problem()}} object.}

\item{gap}{\code{numeric} gap to optimality. This gap is relative
and expresses the acceptable deviance from the optimal objective.
For example, a value of 0.01 will result in the solver stopping when
it has found a solution within 1\% of optimality.
Additionally, a value of 0 will result in the solver stopping
when it has found an optimal solution.
The default value is 0.1 (i.e., 10\% from optimality).}

\item{time_limit}{\code{numeric} time limit (seconds) for generating solutions.
The solver will return the current best solution when this time limit is
exceeded. The default value is the largest integer value
(i.e., \code{.Machine$integer.max}), effectively meaning that solver
will keep running until a solution within the optimality gap is found.}

\item{presolve}{\code{logical} attempt to simplify the
problem before solving it? Defaults to \code{TRUE}.}

\item{threads}{\code{integer} number of threads to use for the
optimization algorithm. The default value is 1.}

\item{first_feasible}{\code{logical} should the first feasible solution be
be returned? If \code{first_feasible} is set to \code{TRUE}, the solver
will return the first solution it encounters that meets all the
constraints, regardless of solution quality. Note that the first feasible
solution is not an arbitrary solution, rather it is derived from the
relaxed solution, and is therefore often reasonably close to optimality.
Defaults to \code{FALSE}.}

\item{start_solution}{\code{NULL} or object containing the starting solution
for the solver. Defaults to \code{NULL} such that no starting solution is used.
To specify a starting solution, the argument to \code{start_solution} should
be in the same format as the planning units (i.e., a \code{NULL}, \code{numeric},
\code{matrix}, \code{data.frame}, \code{\link[terra:rast]{terra::rast()}}, or \code{\link[sf:sf]{sf::sf()}} object).
See the Start solution format section for more information.}

\item{verbose}{\code{logical} should information be printed while solving
optimization problems? Defaults to \code{TRUE}.}
}
\value{
An updated \code{\link[=problem]{problem()}} object with the solver added to it.
}
\description{
Specify that the \href{https://github.com/coin-or/Cbc}{\emph{CBC}}
(COIN-OR branch and cut) software should be used to solve a conservation
planning problem (Forrest & Lougee-Heimer 2005).
This function can also be used to customize the behavior of the solver.
It requires the \pkg{rcbc} package to be installed
(only \href{https://github.com/dirkschumacher/rcbc}{available on GitHub},
see below for installation instructions).
}
\details{
\href{https://github.com/coin-or/Cbc}{\emph{CBC}} is an
open-source mixed integer programming solver that is part of the
Computational Infrastructure for Operations Research (COIN-OR) project.
This solver seems to have much better performance than the other open-source
solvers (i.e., \code{\link[=add_highs_solver]{add_highs_solver()}}, \code{\link[=add_rsymphony_solver]{add_rsymphony_solver()}},
\code{\link[=add_lpsymphony_solver]{add_lpsymphony_solver()}})
(see the \emph{Solver benchmarks} vignette for details).
As such, it is strongly recommended to use this solver if the \emph{Gurobi} and
\emph{IBM CPLEX} solvers are not available.
}
\section{Installation}{

The \pkg{rcbc} package is required to use this solver. Since the
\pkg{rcbc} package is not available on the
the Comprehensive R Archive Network (CRAN), it must be installed from
\href{https://github.com/dirkschumacher/rcbc}{its GitHub repository}. To
install the \pkg{rcbc} package, please use the following code:

\if{html}{\out{<div class="sourceCode">}}\preformatted{if (!require(remotes)) install.packages("remotes")
remotes::install_github("dirkschumacher/rcbc")
}\if{html}{\out{</div>}}

Note that you may also need to install several dependencies --
such as the
\href{https://cran.r-project.org/bin/windows/Rtools/}{Rtools software}
or system libraries -- prior to installing the \pkg{rcbc} package.
For further details on installing this package, please consult the
\href{https://dirkschumacher.github.io/rcbc/}{online package documentation}.
}

\section{Start solution format}{

Broadly speaking, the argument to \code{start_solution} must be in the same
format as the planning unit data in the argument to \code{x}.
Further details on the correct format are listed separately
for each of the different planning unit data formats:

\describe{

\item{\code{x} has \code{numeric} planning units}{The argument to \code{start_solution} must be a
\code{numeric} vector with each element corresponding to a different planning
unit. It should have the same number of planning units as those
in the argument to \code{x}. Additionally, any planning units missing
cost (\code{NA}) values should also have missing (\code{NA}) values in the
argument to \code{start_solution}.
}

\item{\code{x} has \code{matrix} planning units}{The argument to \code{start_solution} must be a
\code{matrix} vector with each row corresponding to a different planning
unit, and each column correspond to a different management zone.
It should have the same number of planning units and zones
as those in the argument to \code{x}. Additionally, any planning units
missing cost (\code{NA}) values for a particular zone should also have a
missing (\code{NA}) values in the argument to \code{start_solution}.
}

\item{\code{x} has \code{\link[terra:rast]{terra::rast()}} planning units}{The argument to \code{start_solution}
be a \code{\link[terra:rast]{terra::rast()}} object where different grid cells (pixels) correspond
to different planning units and layers correspond to
a different management zones. It should have the same dimensionality
(rows, columns, layers), resolution, extent, and coordinate reference
system as the planning units in the argument to \code{x}. Additionally,
any planning units missing cost (\code{NA}) values for a particular zone
should also have missing (\code{NA})  values in the argument to \code{start_solution}.
}

\item{\code{x} has \code{data.frame} planning units}{The argument to \code{start_solution} must
be a \code{data.frame} with each column corresponding to a different zone,
each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if a \code{data.frame}
object containing the solution also contains additional columns, then
these columns will need to be subsetted prior to using this function
(see below for example with \code{\link[sf:sf]{sf::sf()}} data).
Additionally, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{start_solution}.
}

\item{\code{x} has \code{\link[sf:sf]{sf::sf()}} planning units}{The argument to \code{start_solution} must be
a \code{\link[sf:sf]{sf::sf()}} object with each column corresponding to a different
zone, each row corresponding to a different planning unit, and cell values
corresponding to the solution value. This means that if the
\code{\link[sf:sf]{sf::sf()}} object containing the solution also contains additional
columns, then these columns will need to be subsetted prior to using this
function (see below for example).
Additionally, the argument to \code{start_solution} must also have the same
coordinate reference system as the planning unit data.
Furthermore, any planning units missing cost
(\code{NA}) values for a particular zone should also have missing (\code{NA})
values in the argument to \code{start_solution}.
}
}
}

\examples{
\dontrun{
# load data
sim_pu_raster <- get_sim_pu_raster()
sim_features <- get_sim_features()

# create problem
p1 <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_binary_decisions() \%>\%
  add_cbc_solver(gap = 0, verbose = FALSE)

# generate solution \%>\%
s1 <- solve(p1)

# plot solution
plot(s1, main = "solution", axes = FALSE)

# create a similar problem with boundary length penalties and
# specify the solution from the previous run as a starting solution
p2 <-
  problem(sim_pu_raster, sim_features) \%>\%
  add_min_set_objective() \%>\%
  add_relative_targets(0.1) \%>\%
  add_boundary_penalties(10) \%>\%
  add_binary_decisions() \%>\%
  add_cbc_solver(gap = 0, start_solution = s1, verbose = FALSE)

# generate solution
s2 <- solve(p2)

# plot solution
plot(s2, main = "solution with boundary penalties", axes = FALSE)
}
}
\references{
Forrest J and Lougee-Heimer R (2005) CBC User Guide. In Emerging theory,
Methods, and Applications (pp. 257--277). INFORMS, Catonsville, MD.
\doi{10.1287/educ.1053.0020}.
}
\seealso{
Other solvers: 
\code{\link{add_cplex_solver}()},
\code{\link{add_default_solver}()},
\code{\link{add_gurobi_solver}()},
\code{\link{add_highs_solver}()},
\code{\link{add_lsymphony_solver}},
\code{\link{add_rsymphony_solver}()}
}
\concept{solvers}
