% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/RcppExports.R, R/func_tapers.R
\name{parabolic_weights_rcpp}
\alias{parabolic_weights}
\alias{parabolic_weights.tapers}
\alias{parabolic_weights_fast}
\alias{parabolic_weights_rcpp}
\title{Calculate parabolic weighting factors.}
\usage{
parabolic_weights_rcpp(ntap = 1L)

parabolic_weights(tapvec, tap.index = 1L)

\method{parabolic_weights}{tapers}(tapvec, tap.index = 1L)

parabolic_weights_fast(ntap = 1L)
}
\arguments{
\item{ntap}{integer; the number of tapers to provide weightings for.}

\item{tapvec}{\code{'tapers'} object; the number of tapers at each frequency}

\item{tap.index}{integer; the index of \code{tapvec} from which to produce a sequence of weights for}
}
\value{
A list with the number of tapers, indices of the taper sequence, and the weights \eqn{W_N}{w}.
}
\description{
The resampled spectrum involves summing weighted tapers; this produces
the weighting factors.
\code{\link{parabolic_weights_rcpp}} is the fastest implementation, used by
\code{\link{resample_fft_rcpp}}, but it takes only a single value.
\code{\link{parabolic_weights}} calls \code{\link{parabolic_weights_fast}} for vectors.
}
\details{
If one has a \code{tapers} object, specify the \code{taper.index} to
produce a sequence of weights up to the value at that index; the user
is likely to never need to use this function though.

Weighting factors, \eqn{W}{w}, are calculated as follows:
\deqn{
 W \equiv \frac{6 (n^2 - K^2)}{n (4 * n - 1) (n + 1)}
}{
 w = 6 (k^2 - (K-1)^2) / (k (4 * k - 1) (k + 1))
}
where \eqn{n}{k} is the total number of tapers, and
\eqn{K}{K} is the integer sequence \eqn{[0,n-1]}{[0,K-1]}.

The sum of tapers should equal 1, within machine precision, when \eqn{n>0}{k>0}.
}
\examples{
\dontrun{#REX
library(psd)
library(grDevices)
library(RColorBrewer)

##
## Show parabolic weighting factors as a function of maximum tapers
##

# maximum number of tapers
maxx <- 1e3
# sequence in logspace
xseq <- seq(from=1,to=2.8,by=0.2)

# plot palette
pal <- "Spectral"
npal <- switch(pal, RdYlBu=11, Spectral=11, Blues=9)
pal.col <- RColorBrewer::brewer.pal(npal, pal)
cols <- rev(grDevices::colorRampPalette(pal.col)(maxx))

to_df <- function(W){
  # convert parabolic results to data.frame
  with(W, data.frame(taper_seq=as.vector(taper_seq), taper_weights=as.vector(taper_weights)))
}

## a roundabout way of bootstrapping y-axis limits:
#  upper
WgtsU <- parabolic_weights_fast(5)
DfU <- to_df(WgtsU)
#  lower
WgtsL <- parabolic_weights_fast(maxx)
DfL <- to_df(WgtsL)

ylims <- range(pretty(dB(c(DfL$taper_weights, DfU$taper_weights)))) + c(-2,5)

# function for plotting text
TFUN <- function(Df.){
  tx <- max(Df.$taper_seq)
  ty <- mean(Df.$taper_weights)
  text(log10(tx)+0.1, dB(ty), sprintf("\%i", tx), col=cols[tx])
}

# function for weighting factors and plotting
WFUN <- function(x){
  message(x)
  Wgts <- parabolic_weights_fast(x)
  Df <- to_df(Wgts)
  lcol <- cols[x]
  lines(dB(taper_weights) ~ log10(taper_seq), Df, type="s", lwd=2, col=lcol)
  TFUN(Df)
}

## Plot parabolic weighting, in dB, colored by maximum num tapers
plot(dB(taper_weights) ~ log10(taper_seq), DfU, type="s", 
     xlim=c(0, log10(maxx)+0.2), 
     ylim=ylims, yaxs="i",
     col=cols[5], lwd=2,  
     main="Multitaper weighting factors by maximum tapers applied",
     xlab="log10 taper sequence", 
     ylab="dB")
TFUN(DfU)
invisible(lapply(round(10**xseq), FUN=WFUN))
WFUN(maxx)

##
}#REX
}
\author{
A.J. Barbour adapted the original algorithm (R.L. Parker), and authored the optimized versions.
}
\seealso{
\code{\link{resample_fft_rcpp}}, \code{\link{psdcore}}, \code{\link{riedsid}}
}

