% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/impactspar.R
\name{impactspar}
\alias{impactspar}
\title{Compute direct, indirect and total impacts
  for continous parametric covariates.}
\usage{
impactspar(obj, ..., tr = NULL, R = 1000, listw = NULL, tol = 1e-06, Q = NULL)
}
\arguments{
\item{obj}{A `pspatreg` object created by \code{\link{pspatfit}}.}

\item{...}{Arguments passed through to methods in the \pkg{coda} package}

\item{tr}{A vector of traces of powers of the spatial weights matrix created using \code{trW}, for approximate impact measures; if not given, \code{listw} must be given for exact measures (for small to moderate spatial weights matrices); the traces must be for the same spatial weights as were used in fitting the spatial regression, and must be row-standardised}

\item{R}{If given, simulations are used to compute distributions for the impact measures, returned as \code{mcmc} objects; the objects are used for convenience but are not output by an MCMC process}

\item{listw}{If \code{tr} is not given, a spatial weights object as created by \code{nb2listw}; they must be the same spatial weights as were used in fitting the spatial regression, but do not have to be row-standardised}

\item{tol}{Argument passed to \code{mvrnorm}: tolerance (relative to largest variance) for numerical lack of positive-definiteness in the coefficient covariance matrix}

\item{Q}{default NULL, else an integer number of cumulative power series impacts to calculate if \code{tr} is given}
}
\value{
An object of class \emph{impactspar.pspatreg}. Can be printed
  with \code{summary}.
        
 If \code{type = "sar", "sdm", "sarar"}, the object returned is a list 
 with 4 objects including the type of model and three matrices including the simulated
 total, direct and indirect impacts:         
 \tabular{ll}{
   \emph{type} \tab Type of spatial econometric model. \cr
   \emph{mimpactstot} \tab Matrix including simulated total impacts 
                           for each variable in rows. \cr
   \emph{mimpactsdir} \tab Matrix including simulated direct impacts 
                           for each variable in rows. \cr
   \emph{mimpactsind} \tab Matrix including simulated indirect impacts 
                           for each variable in rows. \cr
   }
   If \code{type = "slx", "sdem"} the object returned is a list 
   with 5 objects including the type of model and four matrices including 
   the computed total, direct and indirect impacts, the standard errors,
   the z-values and p-values of each type of impact:
 \tabular{ll}{
   \emph{type} \tab Type of spatial econometric model. \cr
   \emph{mimpact} \tab Matrix including computed total, direct and indirect
                       impacts for each variable in rows. \cr
   \emph{semimpact} \tab Matrix including standard errors of total, direct 
                         and indirect impacts for each variable 
                         in rows. \cr
   \emph{zvalmimpact} \tab Matrix including z-values of total, direct 
                         and indirect impacts for each variable 
                         in rows. \cr
   \emph{pvalmimpact} \tab Matrix including p-values of total, direct 
                         and indirect impacts for each variable 
                         in rows. \cr
   }
}
\description{
Compute direct, indirect and total 
  impacts for parametric covariates included in a semiparametric spatial
  or spatio-temporal model. The models can be of type \emph{ps-sar}, 
  \emph{ps-sarar}, \emph{ps-sdm}, \emph{ps-sdem} or
  \emph{ps-slx}.
}
\details{
This function is similar to the 
  \code{\link[spatialreg]{impacts}} method used in \pkg{spatialreg}.
   package. The function
  \code{\link{impactspar}} obtains the three type of impacts (total, direct
  and indirect) together with a measure of statistical 
  significance, according to the randomization approach described in 
  LeSage and Pace (2009). Briefly, they suggest to obtain a 
  sequence of \emph{nsim} random matrices using a multivariate normal 
  distribution N(0; \emph{Sigma}), being \emph{Sigma} the estimated 
  covariance matrix of the fitted \emph{beta} for parametric covariates
  and spatial parameters of the model. 
  These random matrices, combined with the values of the fitted 
  \emph{beta} for parametric covariates and the estimated 
  values of the spatial parameters, are used to obtain simulated values. 
  The function \code{\link{impactspar}} obtains the standard 
  deviations using the \emph{nsim} simulated impacts in the randomization 
  procedure, which are used to test the significance of the estimated 
  impacts for the original data.
  Finally, if the spatial model is type = "slx" or "sdem", then there is no
  need to simulate to make inference of the impacts. The standard errors of the 
  impacts are computed directly using the \emph{Sigma} matrix of the estimated
  covariances of \emph{beta} and spatial parameters.
}
\examples{
################################################
 ###################### Examples using a panel data of rate of
 ###################### unemployment for 103 Italian provinces in period 1996-2014.
library(pspatreg)
data(unemp_it, package = "pspatreg")
## Wsp_it is a matrix. Create a neighboord list 
lwsp_it <- spdep::mat2listw(Wsp_it)
## short sample for spatial pure case (2d)
unemp_it_short <- unemp_it[unemp_it$year == 2019, ]  
########  No Spatial Trend: PSAR including a spatial 
########  lag of the dependent variable
form1 <- unrate ~ partrate + agri + cons +
                 pspl(serv, nknots = 15) +
                 pspl(empgrowth, nknots = 20)
### example with type = "sar"                   
gamsar <- pspatfit(form1, 
                   data = unemp_it_short, 
                   type = "sar", 
                   listw = lwsp_it)
summary(gamsar)
###### Parametric Total, Direct and Indirect Effects
imp_parvar <- impactspar(gamsar, listw = lwsp_it)
summary(imp_parvar)

### example with type = "slx"                   

gamslx <- pspatfit(form1, 
                   data = unemp_it_short, 
                   type = "slx", 
                   listw = lwsp_it)
                   
summary(gamslx)
###### Parametric Total, Direct and Indirect Effects
imp_parvarslx <- impactspar(gamslx, listw = lwsp_it)
summary(imp_parvarslx)

}
\references{
\itemize{ 
  \item LeSage, J. and Pace, K. (2009). \emph{Introduction to 
        Spatial Econometrics}. CRC Press, Boca Raton.
 }
}
\seealso{
\itemize{
  \item \code{\link{pspatfit}} estimate spatial or spatio-temporal 
          semiparametric ps-sar, ps-sem, ps-sarar, ps-slx or 
          ps-durbin regression models.
  \item \code{\link{impactsnopar}} compute total, direct and indirect impact
          functions for non-parametric continuous covariates.
  \item \code{\link{fit_terms}} compute smooth term functions for 
    non-parametric continuous covariates.
  \item \code{\link[spdep]{impacts}} similar function in \pkg{spdep} 
          package to compute impacts in spatial parametric econometric 
          models.
}
}
