% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ferns.R
\name{rFerns}
\alias{rFerns}
\alias{rFerns.formula}
\alias{rFerns.matrix}
\alias{rFerns.default}
\title{Classification with random ferns}
\usage{
rFerns(x, ...)

\method{rFerns}{formula}(formula, data = .GlobalEnv, ...)

\method{rFerns}{matrix}(x, y, ...)

\method{rFerns}{default}(
  x,
  y,
  depth = 5,
  ferns = 1000,
  importance = "none",
  saveForest = TRUE,
  consistentSeed = NULL,
  threads = 0,
  ...
)
}
\arguments{
\item{x}{Data frame containing attributes; must have unique names and contain only numeric, integer or (ordered) factor columns.
Factors must have less than 31 levels. No \code{NA} values are permitted.}

\item{...}{For formula and matrix methods, a place to state parameters to be passed to default method.
For the print method, arguments to be passed to \code{print}.}

\item{formula}{alternatively, formula describing model to be analysed.}

\item{data}{in which to interpret formula.}

\item{y}{A decision vector. Must a factor of the same length as \code{nrow(X)} for ordinary many-label classification, or a logical matrix with each column corresponding to a class for multi-label classification.}

\item{depth}{The depth of the ferns; must be in 1--16 range. Note that time and memory requirements scale with \code{2^depth}.}

\item{ferns}{Number of ferns to be build.}

\item{importance}{Set to calculate attribute importance measure (VIM);
\code{"simple"} will calculate the default mean decrease of true class score (MDTS, something similar to Random Forest's MDA/MeanDecreaseAccuracy),
\code{"shadow"} will calculate MDTS and additionally MDTS of this attribute shadow, an implicit feature build by shuffling values within it, thus stripping it from information (which is slightly slower).
Shadow importance is useful as a reference to judge significance of a regular importance.
\code{"none"} turns importance calculation off, for a slightly faster execution.
For compatibility with pre-1.2 rFerns, \code{TRUE} will resolve to \code{"simple"} and \code{FALSE} to \code{"none"}.
Abbreviation can be used instead of a full value.}

\item{saveForest}{Should the model be saved? It must be \code{TRUE} if you want to use the model for prediction; however, if you are interested in importance or OOB error only, setting it to \code{FALSE} significantly improves memory requirements, especially for large \code{depth} and \code{ferns}.}

\item{consistentSeed}{PRNG seed used for shadow importance \emph{only}.
Must be either a 2-element integer vector or \code{NULL}, which corresponds to seeding from the default PRNG.}

\item{threads}{Number or OpenMP threads to use. The default value of \code{0} means all available to OpenMP.
It should be set to the same value in two merged models to make shadow importance meaningful.}
}
\value{
An object of class \code{rFerns}, which is a list with the following components:
\item{model}{The built model; \code{NULL} if \code{saveForest} was \code{FALSE}.}
\item{oobErr}{OOB approximation of accuracy.
Ignores never-OOB-tested objects (see oobScores element).}
\item{importance}{The importance scores or \code{NULL} if \code{importance} was set to \code{"none"}.
In a first case it is a \code{data.frame} with two or three columns:
\code{MeanScoreLoss} which is a mean decrease of a score of a correct class when a certain attribute is permuted,
\code{Tries} which is number of ferns which utilised certain attribute, and, only when \code{importance} was set to \code{"shadow"},
\code{Shadow}, which is a mean decrease of accuracy for the correct class for a permuted copy of an attribute (useful as a baseline for normal importance).
The \code{rownames} are set and equal to the \code{names(x)}.}
\item{oobScores}{A matrix of OOB scores of each class for each object in training set.
Rows correspond to classes in the same order as in \code{levels(Y)}.
If the \code{ferns} is too small, some columns may contain \code{NA}s, what means that certain objects were never in test set.}
\item{oobPreds}{A vector of OOB predictions of class for each object in training set. Never-OOB-tested objects (see above) have predictions equal to \code{NA}.}
\item{oobConfusionMatrix}{Confusion matrix build from \code{oobPreds} and \code{y}.}
\item{timeTaken}{Time used to train the model (smaller than wall time because data preparation and model final touches are excluded; however it includes the time needed to compute importance, if it applies).
An object of \code{difftime} class.}
\item{parameters}{Numerical vector of three elements: \code{classes}, \code{depth} and \code{ferns}, containing respectively the number of classes in decision and copies of \code{depth} and \code{ferns} parameters.}
\item{classLabels}{Copy of \code{levels(Y)} after purging unused levels.}
\item{consistentSeed}{Consistent seed used; only present for \code{importance="shadow"}.
Can be used to seed a new model via \code{consistentSeed} argument.}
\item{isStruct}{Copy of the train set structure, required internally by predict method.}
}
\description{
This function builds a random ferns model on the given training data.
}
\note{
The unused levels of the decision will be removed; on the other hand unused levels of categorical attributes will be preserved, so that they could be present in the data later predicted with the model.
The levels of ordered factors in training and predicted data must be identical.

Do not use formula interface for a data with large number of attributes; the overhead from handling the formula may be significant.
}
\examples{
set.seed(77)
#Fetch Iris data
data(iris)
#Build model
rFerns(Species~.,data=iris)
##Importance
rFerns(Species~.,data=iris,importance="shadow")->model
print(model$imp)
}
\references{
Ozuysal M, Calonder M, Lepetit V & Fua P. (2009). \emph{Fast Keypoint Recognition using Random Ferns}, IEEE Transactions on Pattern Analysis and Machine Intelligence, 32(3), 448-461.

Kursa MB (2014). \emph{rFerns: An Implementation of the Random Ferns Method for General-Purpose Machine Learning}, Journal of Statistical Software, 61(10), 1-13.
}
