\name{runifInterface}
\alias{runifInterface}
\alias{set.generator}
\alias{put.description}
\alias{get.description}
\title{Functions for using runif() and rnorm() with randtoolbox generators}
\description{
These functions allow to set some of the random number generators from randtoolbox
package so that it can be used in the standard R functions, which use random numbers,
for example \code{runif()}, \code{rnorm()}, \code{sample()} and
also \code{set.seed()}.
}
\usage{
set.generator(name=c("congruRand", "WELL", "MersenneTwister", "default"), 
parameters=NULL, seed=NULL, ..., only.dsc=FALSE)
put.description(description)		
get.description()
}
\arguments{
	\item{name}{a character string for the RNG name.}
	\item{parameters}{a numeric or character vector describing a RNG from the family
           specified by the \code{name} parameter.}
	\item{seed}{a number to be used as a seed}	
	\item{...}{arguments describing the components of the vector parameters,
           if argument \code{parameters} is NULL.}	
	\item{only.dsc}{a logical. If \code{TRUE} the description of a RNG is created,
           but the generator is not initialized.}
	\item{description}{a list describing a generator as created by \code{set.generator()}}
           or \code{get.description()}
}
\details{
Random number generators provided by R extension packages are set using \code{RNGkind("user-supplied")}.
The package \pkg{randtoolbox} assumes that this function is not called by the user directly.
Instead, it is called from the functions \code{set.generator()} and \code{put.description()} used
for setting some of a larger collection of the supported generators.

Random number generators in \pkg{randtoolbox} are represented at the R level by a list
containing mandatory components \code{name, parameters, state} and possibly an
optional component \code{authors}. The function \code{set.generator()} internally
creates this list from the user supplied information and then runs \code{put.description()}
on this list in order to really initialize the generator for the functions \code{runif()}
and \code{set.seed()}. If \code{set.generator()} is called with the parameter
\code{only.dsc=TRUE}, then the generator is not initialized and only
its description is created. If the generator is initialized, then the function
\code{get.description()} may be used to get the actual
state of the generator, which may be stored and used later in \code{put.description()}
to continue the sequence of the random numbers from the point, where \code{get.description()}
was called. This may be used, for example, to alternate between the streams of random numbers
generated by different generators.

The component \code{parameters} is a character or a numeric vector, whose structure
is different for different types of the generators.

\subsection{Linear congruential generators}{
Parameters for the linear congruential generators (\code{name="congruRand"})
are integers represented either as a character or a numeric vector. The
components are
\describe{
  \item{mod}{The modulus.}
  \item{mult}{The multiplier.}
  \item{incr}{The increment.}
}
}

\subsection{WELL generators}{
Parameters for the WELL generators is a character vector with components
\describe{
  \item{order}{The number of bits in the internal state.
   Possible values are 512, 521, 607, 800, 1024, 19937, 21701, 23209, 44497.}
  \item{version}{The version letter "a" or "b" to be appended to the order.}
  \item{temper}{Indication, whether tempering is used. Either "" or "Temp".}
}
Concatenation of \code{order}, \code{version} and \code{temper} should belong
to \code{"512a", "521a", "521b", "607a", "607b", "800a", "800b", "1024a", "1024b",
"19937a", "19937aTemp", "19937b", "21701a", "23209a", "23209b", "44497a", "44497aTemp"}.
}

\subsection{Mersenne Twister 2002 generator}{
Parameters for the Mersenne Twister 2002 is a character vector with components
\describe{
  \item{initialization}{Either "init2002" or "array2002". The initialization to be used.}
  \item{resolution}{Either 53 or 32. The number of random bits in each number.}
}
}

}
\value{
\code{set.generator()} with the parameter \code{only.dsc=TRUE} and
\code{get.description()} return the list describing a generator.
\code{put.description()} with the parameter \code{only.dsc=TRUE} (the default)
and \code{put.description()} return \code{NULL}.
}
\seealso{
\code{RNGkind}
}
\author{
 Petr Savicky and Christophe Dutang
}
\examples{
RNGkind() # [1] "Mersenne-Twister" "Inversion"   

#parameters for Park Miller congruential generator
paramParkMiller <- c(mod=2^31-1, mult=16807, incr=0)
set.generator(name="congruRand", parameters=paramParkMiller, seed=1)

RNGkind() # [1] "user-supplied" "Inversion"

#description of the RNG set by set.generator(), i.e. Park Miller
print(ParkMiller <- get.description())

#generate 10 random points from the Park-Miller sequence
x1 <- runif(10)

#the seed has changed
get.description()

# the Knuth Lewis RNG
paramKnuthLewis <- c(mod="4294967296", mult="1664525", incr="1013904223")
set.generator(name="congruRand", parameters= paramKnuthLewis, seed=1)

#description of the current RNG, i.e. Knuth Lewis
KLwithseed1 <- get.description()

x2 <- runif(10)

#reinitiate the RNG setting
put.description(ParkMiller)

#the same as x1
x1 == runif(10)

#set WELL RNGs
set.generator("WELL", seed=12345, order=1024, version="a")
get.description()

# generate the same random numbers as in Matlab
set.generator("MersenneTwister", initialization="init2002", resolution=53, seed=12345)
runif(5)
# [1] 0.9296161 0.3163756 0.1839188 0.2045603 0.5677250
# Matlab commands rand('twister', 12345); rand(1, 5) generate the same numbers,
# which in short format are   0.9296    0.3164    0.1839    0.2046    0.5677

#get back to the original R setting
set.generator("default")
RNGkind()

}
\keyword{distribution}

