% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rankCorrCluster.R
\name{rankCorrCluster}
\alias{rankCorrCluster}
\title{Total, between-, and within-cluster Spearman rank correlations for clustered data}
\usage{
rankCorrCluster(
  x,
  y,
  cluster,
  link.x = c("probit", "logistic", "cauchit", "loglog", "cloglog"),
  link.y = c("probit", "logistic", "cauchit", "loglog", "cloglog"),
  weights = c("obs", "clusters"),
  methods_between_corr = c("cluster-median", "approx", "both"),
  conf.int = 0.95,
  fisher = FALSE,
  na.rm = FALSE
)
}
\arguments{
\item{x}{a numeric or factor vector.}

\item{y}{a numeric or factor vector.}

\item{cluster}{a vector of cluster index corresponding to \code{x} and \code{y}.}

\item{link.x, link.y}{the link family to be used for the ordinal models of
\code{x} and \code{y} on cluster index. Defaults to \samp{logit}. Other options are
\samp{probit}, \samp{cloglog}, \samp{cauchit}, and \samp{logistic} (equivalent with \samp{logit}).}

\item{weights}{a character string indicating which weighting method is used.
Should be one of the strings \code{"obs"} and \code{"clusters"}. Default is \code{"obs"}. See Details.}

\item{methods_between_corr}{a character string indicating which estimation method of the between-cluster correlation is used.
Should be one of the strings \code{"cluster-median"}, \code{"approx"}, and \code{"both"}. Default is \code{"cluster-median"}. See Details.}

\item{conf.int}{numeric specifying confidence interval level.}

\item{fisher}{logical, indicating whether to apply Fisher transformation to compute confidence intervals.}

\item{na.rm}{logical. Should missing values be removed?}
}
\value{
a list with following components.
\tabular{ll}{
  \code{'Total'} \tab the total Spearman rank correlation, including the estimate (\code{Estimate}), the standard error (\code{SE}), the lower and upper bounds of the confidence interval (\code{Lower, Upper}).\cr
  \tab \cr
  \code{'Within'} \tab the within-cluster Spearman rank correlation, including the estimate (\code{Estimate}), the standard error (\code{SE}), the lower and upper bounds of the confidence interval (\code{Lower, Upper}).\cr
  \tab \cr
  \code{'Between'} \tab the between-cluster Spearman rank correlation estimated, including the estimation method name, the estimate (\code{Estimate}), the standard error (\code{SE}), the lower and upper bounds of the confidence interval (\code{Lower, Upper}).\cr
  \tab \cr
  \code{'Rank ICC'} \tab the rank intraclass correlation coefficients of \code{x} and \code{y}. It would be presented in the output if \code{'approx'} is used. \cr
}
}
\description{
\code{rankCorrCluster} computes the total, between-, and within-cluster Spearman rank correlations between two variables using two-level clustered data. It can be used with any orderable variable, including continuous and discrete variables. Two weighting methods are provided, including assigning equal weights to observations or to clusters.
}
\details{
The weighting method \code{"obs"} assigns equal weights to observations; \eqn{w_{ij} = 1/N}, where \var{N} is the total number of observations. The weighting method \code{"clusters"} assigns equal weights to clusters; \eqn{w_{ij} = 1/(nk_i)}, where \var{n} is the total number of clusters and k_i is the cluster size.
The estimation method \code{"cluster-median"} estimates the between-cluster Spearman rank correlation using the coefficients from the cumulative probability models of \code{x} and \code{y} on cluster index \code{cluster}. The estimation method \code{"approx"} estimates the between-cluster Spearman rank correlation using the approximated linear relationship between the total, between-, and within-cluster Spearman rank correlations.
}
\examples{
library(mvtnorm)
k <- 50; m <- 5
sigma.u <- matrix(c(1, 0.6, 0.6, 4), ncol=2); sigma.e <- matrix(c(1, 0.6, 0.6, 1), ncol=2)
u <- rmvnorm(k, c(1, -1), sigma.u)
x1 <- matrix(NA, k, m)
y1 <- matrix(NA, k, m)
for (i in 1:k){
r <- rmvnorm(m, c(0, 0), sigma.e)
x1[i,] <- u[i, 1] + r[, 1]
y1[i,] <- u[i, 2] + r[, 2]
}
x <- as.vector(t(x1))
y <- as.vector(t(y1))
cluster <- rep(1:k, each=m)
rankCorrCluster(x, y, cluster, link.x = "probit", link.y = "probit",
methods_between_corr = "approx")
idx <- sample(1:250, 200, replace = TRUE)
rankCorrCluster(x[idx], y[idx], cluster[idx], link.x = "probit", link.y = "probit",
weights = "clusters")
}
\references{
\tabular{ll}{
Tu, S., Li, C., Zeng, D., and Shepherd, B. E. (2023). Rank intraclass correlation for clustered data. Statistics in Medicine 42, 4333-4348. \cr
Shih, J. H. and Fay, M. P. (2017). Pearson's chi-square test and rank correlation inferences for clustered data. Biometrics 73, 822-834. \cr
Tu S, Li C, Shepherd BE (2023). Between- and within-cluster Spearman rank correlations. \cr
}
}
