library(rearrr)
context("cluster_groups()")


test_that("fuzz testing cluster_groups()", {
  xpectr::set_test_seed(42)

  set.seed(2)

  # Create a data frame
  df <- data.frame(
    "x" = runif(15),
    "y" = runif(15),
    "z" = runif(15),
    "g" = rep(c(1, 2, 3, 4, 5), each = 3),
    stringsAsFactors = FALSE
  )

  # Ensure df is the same!
  ## Testing 'df'                                                           ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(df),
    "data.frame",
    fixed = TRUE)
  # Testing column values
  expect_equal(
    df[["x"]],
    c(0.18488, 0.70237, 0.57333, 0.16805, 0.94384, 0.94347, 0.12916,
      0.83345, 0.46802, 0.54998, 0.55267, 0.23889, 0.76051, 0.18082,
      0.40528),
    tolerance = 1e-4)
  expect_equal(
    df[["y"]],
    c(0.85355, 0.9764, 0.22583, 0.44481, 0.07498, 0.6619, 0.38755, 0.83689,
      0.1505, 0.34727, 0.48877, 0.14925, 0.35706, 0.96264, 0.13237),
    tolerance = 1e-4)
  expect_equal(
    df[["z"]],
    c(0.01041, 0.16464, 0.81019, 0.86886, 0.51428, 0.6272, 0.84443,
      0.28487, 0.66723, 0.15047, 0.98173, 0.29701, 0.11508, 0.1632,
      0.94404),
    tolerance = 1e-4)
  expect_equal(
    df[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(df),
    c("x", "y", "z", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(df),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(df),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(df),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(df)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'df'                                                  ####


  # Generate expectations for 'cluster_groups'
  # Tip: comment out the gxs_function() call
  # so it is easy to regenerate the tests
  xpectr::set_test_seed(42)
  # xpectr::gxs_function(
  #   fn = cluster_groups,
  #   args_values = list(
  #     "data" = list(df, 1:15, NA),
  #     "cols" = list(c("x","y","z"), c("x", "y"), "x", "hej", 3, NA),
  #     "group_cols" = list("g", NULL),
  #     "scale_min_fn" = list(function(x) {quantile(x, 0.025)}),
  #     "scale_max_fn" = list(function(x) {quantile(x, 0.975)}),
  #     "keep_centroids" = list(FALSE, TRUE),
  #     "multiplier" = list(0.05, -0.05, 0.2, "hej", NA),
  #     "suffix" = list("_clustered", "", NA),
  #     "keep_original" = list(FALSE, TRUE),
  #     "overwrite" = list(TRUE, FALSE)
  #   ),
  #   extra_combinations = list(
  #     list("data" = dplyr::group_by(df, g), "group_cols" = NULL),
  #     list("overwrite" = FALSE, "suffix" = "")
  #   ),
  #   indentation = 2
  # )


  ## Testing 'cluster_groups'                                                 ####
  ## Initially generated by xpectr
  # Testing different combinations of argument values

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19148[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["y_clustered"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["z_clustered"]],
    c(0.04705, 0.06638, 0.14731, 0.96854, 0.92409, 0.93824, 0.79572,
      0.72557, 0.7735, 0.41709, 0.5213, 0.43546, 0.2484, 0.25443,
      0.35232),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_19148),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = 1:15, cols = c("x", "y...
  # Changed from baseline: data = 1:15
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19370 <- xpectr::capture_side_effects(cluster_groups(data = 1:15, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'integer'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19370[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = NA, cols = c("x", "y",...
  # Changed from baseline: data = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12861 <- xpectr::capture_side_effects(cluster_groups(data = NA, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12861[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'logical'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12861[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = NULL, cols = c("x", "y...
  # Changed from baseline: data = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18304 <- xpectr::capture_side_effects(cluster_groups(data = NULL, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'data': Must be of type 'data.frame', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = dplyr::group_by(df, g)...
  # Changed from baseline: data, group_cols
  xpectr::set_test_seed(42)
  # Assigning output
  output_16417 <- cluster_groups(data = dplyr::group_by(df, g), cols = c("x", "y", "z"), group_cols = NULL, scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_16417),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_16417[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_16417[["y_clustered"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_16417[["z_clustered"]],
    c(0.04705, 0.06638, 0.14731, 0.96854, 0.92409, 0.93824, 0.79572,
      0.72557, 0.7735, 0.41709, 0.5213, 0.43546, 0.2484, 0.25443,
      0.35232),
    tolerance = 1e-4)
  expect_equal(
    output_16417[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_16417),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16417),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16417),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16417),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16417)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y")...
  # Changed from baseline: cols = c("x", "y")
  xpectr::set_test_seed(42)
  # Assigning output
  output_15190 <- cluster_groups(data = df, cols = c("x", "y"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_15190),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_15190[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_15190[["y_clustered"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_15190[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_15190),
    c("x_clustered", "y_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_15190),
    c("numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_15190),
    c("double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_15190),
    c(15L, 3L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_15190)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = "x", group_...
  # Changed from baseline: cols = "x"
  xpectr::set_test_seed(42)
  # Assigning output
  output_17365 <- cluster_groups(data = df, cols = "x", group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_17365),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_17365[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_17365[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_17365),
    c("x_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17365),
    c("numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17365),
    c("double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17365),
    c(15L, 2L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17365)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = "hej", grou...
  # Changed from baseline: cols = "hej"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11346 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = "hej", group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_11346[['error']], lowercase = TRUE),
    xpectr::strip("Must include the elements {hej,g}.", lowercase = TRUE), # colnames(data)
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11346[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = 3, group_co...
  # Changed from baseline: cols = 3
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16569 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = 3, group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'cols': Must be of type 'character', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16569[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = NA, group_c...
  # Changed from baseline: cols = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17050 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = NA, group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error']]),
    xpectr::strip("Assertion on 'must.include' failed. May not contain missing values, first at position 1."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17050[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = NULL, group...
  # Changed from baseline: cols = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14577 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = NULL, group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14577[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'cols': Must be of type 'character', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14577[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: group_cols = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17191 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = NULL, scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error']]),
    xpectr::strip("1 assertions failed:\n * when 'group_cols' is 'NULL', 'data' should be grouped."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: scale_min_fn = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19346 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = NULL, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'scale_min_fn': Must be a function, not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: scale_max_fn = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12554 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = NULL, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12554[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'scale_max_fn': Must be a function, not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12554[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: keep_centroids = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_14622 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = TRUE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_14622),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_14622[["x_clustered"]],
    c(0.44081, 0.51973, 0.50005, 0.60626, 0.72458, 0.72452, 0.42385,
      0.53126, 0.47552, 0.46286, 0.46327, 0.41542, 0.4964, 0.40799,
      0.44222),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["y_clustered"]],
    c(0.70465, 0.7188, 0.63232, 0.39976, 0.35715, 0.42478, 0.45016,
      0.50194, 0.42285, 0.3306, 0.34691, 0.30778, 0.4694, 0.53918,
      0.44351),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["z_clustered"]],
    c(0.28855, 0.30789, 0.38881, 0.69503, 0.65058, 0.66473, 0.62963,
      0.55948, 0.60741, 0.43554, 0.53975, 0.45391, 0.37079, 0.37682,
      0.47471),
    tolerance = 1e-4)
  expect_equal(
    output_14622[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_14622),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_14622),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_14622),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_14622),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_14622)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: keep_centroids = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19400 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = NULL, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'keep_centroids': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: multiplier = -0.05
  xpectr::set_test_seed(42)
  # Assigning output
  output_19782 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = -0.05, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_19782),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19782[["x_clustered"]],
    c(0.34084, 0.26621, 0.28482, 0.94371, 0.83183, 0.83188, 0.31863,
      0.21706, 0.26976, 0.16802, 0.16763, 0.21289, 0.14277, 0.22637,
      0.194),
    tolerance = 1e-4)
  expect_equal(
    output_19782[["y_clustered"]],
    c(0.90065, 0.88676, 0.97158, 0.25539, 0.29719, 0.23086, 0.41474,
      0.36396, 0.44152, 0.11106, 0.09507, 0.13344, 0.4792, 0.41077,
      0.50459),
    tolerance = 1e-4)
  expect_equal(
    output_19782[["z_clustered"]],
    c(0.14569, 0.12667, 0.04705, 0.92481, 0.96854, 0.95461, 0.74323,
      0.81224, 0.76508, 0.5117, 0.40918, 0.49363, 0.33746, 0.33152,
      0.23522),
    tolerance = 1e-4)
  expect_equal(
    output_19782[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_19782),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19782),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19782),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19782),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19782)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: multiplier = 0.2
  xpectr::set_test_seed(42)
  # Assigning output
  output_11174 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.2, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_11174),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_11174[["x_clustered"]],
    c(0.21593, 0.4586, 0.39809, 0.57992, 0.94371, 0.94354, 0.17107,
      0.50133, 0.32997, 0.31272, 0.31398, 0.16684, 0.41461, 0.14277,
      0.24803),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["y_clustered"]],
    c(0.92382, 0.97158, 0.67977, 0.31179, 0.168, 0.39619, 0.38971, 0.56441,
      0.29754, 0.17206, 0.22707, 0.09507, 0.41784, 0.65329, 0.33048),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["z_clustered"]],
    c(0.04705, 0.11092, 0.37824, 0.96854, 0.8217, 0.86846, 0.84036,
      0.60865, 0.76698, 0.35018, 0.69441, 0.41086, 0.22129, 0.24122,
      0.56457),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_11174),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11174),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11174),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11174),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11174)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: multiplier = "hej"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14749 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = "hej", suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'multiplier': Must be of type 'number', not 'character'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: multiplier = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15603 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = NA, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15603[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'multiplier': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15603[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: multiplier = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19040 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = NULL, suffix = "_clustered", keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'multiplier': Must be of type 'number', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: suffix = ""
  xpectr::set_test_seed(42)
  # Assigning output
  output_11387 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "", keep_original = FALSE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_11387),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_11387[["x"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_11387[["y"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_11387[["z"]],
    c(0.04705, 0.06638, 0.14731, 0.96854, 0.92409, 0.93824, 0.79572,
      0.72557, 0.7735, 0.41709, 0.5213, 0.43546, 0.2484, 0.25443,
      0.35232),
    tolerance = 1e-4)
  expect_equal(
    output_11387[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_11387),
    c("x", "y", "z", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11387),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11387),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11387),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11387)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: suffix = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19888 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = NA, keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19888[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'suffix': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19888[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: suffix = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19466 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = NULL, keep_original = FALSE, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19466[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'suffix': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19466[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: suffix, overwrite
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10824 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "", keep_original = FALSE, overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10824[['error']]),
    xpectr::strip("1 assertions failed:\n * The column 'x' already exists and 'overwrite' is disabled."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10824[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: keep_original = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_15142 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = TRUE, overwrite = TRUE)
  # Testing class
  expect_equal(
    class(output_15142),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_15142[["x"]],
    c(0.18488, 0.70237, 0.57333, 0.16805, 0.94384, 0.94347, 0.12916,
      0.83345, 0.46802, 0.54998, 0.55267, 0.23889, 0.76051, 0.18082,
      0.40528),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["y"]],
    c(0.85355, 0.9764, 0.22583, 0.44481, 0.07498, 0.6619, 0.38755, 0.83689,
      0.1505, 0.34727, 0.48877, 0.14925, 0.35706, 0.96264, 0.13237),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["z"]],
    c(0.01041, 0.16464, 0.81019, 0.86886, 0.51428, 0.6272, 0.84443,
      0.28487, 0.66723, 0.15047, 0.98173, 0.29701, 0.11508, 0.1632,
      0.94404),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["y_clustered"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["z_clustered"]],
    c(0.04705, 0.06638, 0.14731, 0.96854, 0.92409, 0.93824, 0.79572,
      0.72557, 0.7735, 0.41709, 0.5213, 0.43546, 0.2484, 0.25443,
      0.35232),
    tolerance = 1e-4)
  expect_equal(
    output_15142[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_15142),
    c("x", "y", "z", "x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_15142),
    c("numeric", "numeric", "numeric", "numeric", "numeric", "numeric",
      "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_15142),
    c("double", "double", "double", "double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_15142),
    c(15L, 7L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_15142)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: keep_original = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13902 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = NULL, overwrite = TRUE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13902[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'keep_original': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13902[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: overwrite = FALSE
  xpectr::set_test_seed(42)
  # Assigning output
  output_19057 <- cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19057),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19057[["x_clustered"]],
    c(0.25347, 0.33239, 0.31271, 0.8254, 0.94371, 0.94366, 0.21604,
      0.32345, 0.26772, 0.19418, 0.19459, 0.14674, 0.23118, 0.14277,
      0.177),
    tolerance = 1e-4)
  expect_equal(
    output_19057[["y_clustered"]],
    c(0.95743, 0.97158, 0.8851, 0.27245, 0.22983, 0.29746, 0.40688,
      0.45866, 0.37957, 0.11788, 0.13419, 0.09507, 0.45966, 0.52944,
      0.43377),
    tolerance = 1e-4)
  expect_equal(
    output_19057[["z_clustered"]],
    c(0.04705, 0.06638, 0.14731, 0.96854, 0.92409, 0.93824, 0.79572,
      0.72557, 0.7735, 0.41709, 0.5213, 0.43546, 0.2484, 0.25443,
      0.35232),
    tolerance = 1e-4)
  expect_equal(
    output_19057[["g"]],
    c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4, 5, 5, 5),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_19057),
    c("x_clustered", "y_clustered", "z_clustered", "g"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19057),
    c("numeric", "numeric", "numeric", "numeric"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19057),
    c("double", "double", "double", "double"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19057),
    c(15L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19057)),
    character(0),
    fixed = TRUE)

  # Testing cluster_groups(data = df, cols = c("x", "y",...
  # Changed from baseline: overwrite = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14469 <- xpectr::capture_side_effects(cluster_groups(data = df, cols = c("x", "y", "z"), group_cols = "g", scale_min_fn = function(x) {
      quantile(x, 0.025)
  }, scale_max_fn = function(x) {
      quantile(x, 0.975)
  }, keep_centroids = FALSE, multiplier = 0.05, suffix = "_clustered", keep_original = FALSE, overwrite = NULL), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14469[['error']]),
    xpectr::strip("Assertion on 'overwrite' failed: Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14469[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  ## Finished testing 'cluster_groups'                                        ####
  #

})
