#' @name deleteArms
#' @title Delete Arms From a Project
#' 
#' @description Delete arms from a project.  This is a destructive action that
#'   will result in deletion of any events and data associated with the arm.
#'   Due to its destructive nature, it may only be performed on databases 
#'   in development status.  Data loss is non-reversible.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param arms \code{integerish}, a vector of arm numbers that will be 
#'   deleted.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param ... Additional arguments to pass to other methods.
#' 
#' @section REDCap API Documentation:
#' This method allows you to delete Arms from a project. 
#' Notice: Because of this method's destructive nature, it is only 
#' available for use for projects in Development status. Additionally, 
#' please be aware that deleting an arm also automatically deletes all 
#' events that belong to that arm, and will also automatically delete 
#' any records/data that have been collected under that arm (this is 
#' non-reversible data loss).
#' 
#' NOTE: This only works for longitudinal projects.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' None.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 

deleteArms <- function(rcon, arms, ...){
  UseMethod("deleteArms")
}

#' @rdname deleteArms

deleteArms.redcapDbConnection <- function(rcon, arms, ...){
  message("Please accept my apologies.  The deleteArms method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname deleteArms

deleteArms.redcapApiConnection <- function(rcon, arms, ...,
                                           error_handling = getOption("redcap_error_handling")){
  checkmate::assert_integerish(arms)
  
  body <- list(token = rcon$token,
               content = "arm",
               action = "delete",
               arms = paste0(arms, collapse = ","))
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) return(redcap_error(x, error_handling))
}
