#' @name deleteRecords
#' @title Delete Records From a Project
#' 
#' @description Delete records from a project.  This is a destructive action that
#'   will result in deletion of any events and data associated with the arm.
#'   Data loss is non-reversible.  The user must have 'Delete Record' privileges
#'   in the database.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param records a vector of record names to be deleted.
#' @param arm \code{integerish}, the arm number of the arm in which the 
#'   record(s) should be deleted. (This can only be used if the project is 
#'   longitudinal with more than one arm.) NOTE: If the arm parameter is not 
#'   provided, the specified records will be deleted from all arms in which 
#'   they exist. Whereas, if arm is provided, they will only be deleted from 
#'   the specified arm.
#' @param ... Additional arguments to pass to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @section REDCap API Documentation:
#' This method allows you to delete one or more records from a project in a 
#' single API request.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' The number of deleted records.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' @export

deleteRecords <- function(rcon, 
                          records, 
                          arm      = NULL,
                          ...){
  UseMethod("deleteRecords")
}

#' @rdname deleteRecords
#' @export

deleteRecords.redcapApiConnection <- function(rcon, 
                                              records, 
                                              arm            = NULL, 
                                              ...,
                                              error_handling = getOption("redcap_error_handling"), 
                                              config         = list(), 
                                              api_param      = list()){
  
  if (is.numeric(records)) records <- as.character(records)
  if (is.character(arm)) arm <- as.numeric(arm)
  
   ##################################################################
  # Argument Validation
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_character(x = records,
                              any.missing = FALSE,
                              min.len = 1,
                              add = coll)
  
  checkmate::assert_integerish(arm,
                               len = 1, 
                               any.missing = FALSE,
                               null.ok = TRUE,
                               add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  Arms <- rcon$arms()
  
  checkmate::assert_subset(x = arm,
                           choices = Arms$arm_num, 
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Build the Body List
  
  body <- list(token = rcon$token,
               content = "record",
               action = "delete", 
               arm = arm)
  
  body <- c(body,
            vectorToApiBodyList(vector = records,
                                parameter_name = "records"))

  body <- body[lengths(body) > 0]
  
   ##################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200){
    return(redcap_error(response, error_handling))
  } 
  
  as.character(response)
}
