#' @name redcapDataStructures
#' @title REDCap Data Structures
#'
#' @description Utilities for recognizing and validating data structures 
#'   for use with the REDCap API
#' 
#' @param data `data.frame` User provided data to be compared against
#'   the established REDCap data structure.
#' @param redcap_data `data.frame` A data set from the `redcapAPI`
#'   package to use a reference for comparing to expected data structure. 

validateRedcapData <- function(data, redcap_data){
  if (is.null(data)) return(NULL)
  
  checkmate::assert_data_frame(x = data, 
                               col.names = "named")
  
  in_data_and_not_structure <- setdiff(names(data), 
                                       names(redcap_data))
  
  # names in data not in the redcap structure
  if (length(in_data_and_not_structure) > 0){
    warning("The following names in 'data' are not recognized in the REDCap data structure.\n", 
            "Downstream functions may not operate as expected. {", 
            paste0(in_data_and_not_structure, collapse = ", "), "}\n")
  }
  
  # names in the redcap structure not in the data
  
  in_structure_and_not_data <- setdiff(names(redcap_data), 
                                       names(data))
  
  if (length(in_structure_and_not_data) > 0){
    warning("The following names in the REDCap data structure are not in 'data'. \n", 
            "Downstream functions may not operate as expected. {", 
            paste0(in_structure_and_not_data, collapse = ", "), "}\n")
  }
  return(data)
}

#' @rdname redcapDataStructures
#' @export

# The field names listed here are those generated by REDCap records export. 
# A listing of these fields was needed to support missingSummary.
# It would not hurt to contemplate how these could be obtained 
# dynamically rather than relying up updating constants.

REDCAP_SYSTEM_FIELDS <- c("redcap_event_name", 
                          "redcap_data_access_group", 
                          "redcap_repeat_instrument", 
                          "redcap_repeat_instance")

# Arms --------------------------------------------------------------
# Arms - Data Frame Structure

REDCAP_ARMS_STRUCTURE <- 
  data.frame(arm_num = numeric(0), 
             name = character(0), 
             stringsAsFactors = FALSE)

# Data Access Groups ------------------------------------------------
# DAG Structure

REDCAP_DAG_STRUCTURE <- 
  data.frame(data_access_group_name = character(0), 
             unique_group_name = character(0), 
             data_access_group_id = numeric(0), 
             stringsAsFactors = FALSE)

# DAG Assignment Structure

REDCAP_DAG_ASSIGNMENT_STRUCTURE <- 
  data.frame(username = character(0), 
             redcap_data_access_group = character(0), 
             stringsAsFactors = FALSE)

# Events ------------------------------------------------------------
# Event Structure

REDCAP_EVENT_STRUCTURE <- 
  data.frame(event_name = character(0), 
             arm_num = numeric(0), 
             unique_event_name = character(0), 
             custom_event_label = character(0), 
             event_id = numeric(0),
             days_offset = numeric(0), 
             offset_min = numeric(0), 
             offset_max = numeric(0),
             stringsAsFactors = FALSE)

# Field Names -------------------------------------------------------
# Field Name Structure 

REDCAP_FIELDNAME_STRUCTURE <- 
  data.frame(original_field_name = character(0), 
             choice_value = character(0), 
             export_field_name = character(0), 
             stringsAsFactors = FALSE)

# Instruments -------------------------------------------------------
# Instrument Structure

REDCAP_INSTRUMENT_STRUCTURE <- 
  data.frame(instrument_name = character(0), 
             instrument_label = character(0), 
             stringsAsFactors = FALSE)

# Instrument Mapping Structure 

REDCAP_INSTRUMENT_MAPPING_STRUCTURE <- 
  data.frame(arm_num = numeric(0), 
             unique_event_name = character(0), 
             form = character(0), 
             stringsAsFactors = FALSE)

# Logging -----------------------------------------------------------
# Logging Structure

REDCAP_LOGGING_STRUCTURE <- 
  data.frame(timestamp = as.POSIXct(character(0)), 
             username = character(0), 
             action = character(0), 
             details = character(0), 
             record = character(0), 
             stringsAsFactors = FALSE)

# Meta Data ---------------------------------------------------------
# Meta Data - Data Frame Structure 

REDCAP_METADATA_STRUCTURE <- 
  data.frame(field_name = character(0),
             form_name = character(0),
             section_header = character(0),
             field_type = character(0),
             field_label = character(0),
             select_choices_or_calculations = character(0),
             field_note = character(0),
             text_validation_type_or_show_slider_number = character(0),
             text_validation_min = character(0),
             text_validation_max = character(0),
             identifier = character(0),
             branching_logic = character(0),
             required_field = character(0),
             custom_alignment = character(0),
             question_number = character(0),
             matrix_group_name = character(0),
             matrix_ranking = character(0),
             field_annotation = character(0), 
             stringsAsFactors = FALSE)

REDCAP_METADATA_API_UI_MAPPING <- 
  c("Variable...Field.Name" = "field_name", 
    "Form.Name" = "form_name", 
    "Section.Header" = "section_header", 
    "Field.Type" = "field_type", 
    "Field.Label" = "field_label", 
    "Choices..Calculations..OR.Slider.Labels" = "select_choices_or_calculations", 
    "Field.Note" = "field_note", 
    "Text.Validation.Type.OR.Show.Slider.Number" = "text_validation_type_or_show_slider_number", 
    "Text.Validation.Min" = "text_validation_min", 
    "Text.Validation.Max" = "text_validation_max", 
    "Identifier." = "identifier", 
    "Branching.Logic..Show.field.only.if...." = "branching_logic", 
    "Required.Field." = "required_field", 
    "Custom.Alignment" = "custom_alignment", 
    "Question.Number..surveys.only." = "question_number", 
    "Matrix.Group.Name" = "matrix_group_name", 
    "Matrix.Ranking." = "matrix_ranking", 
    "Field.Annotation" = "field_annotation")

# Meta Data - Known Field Types 

#' @rdname redcapDataStructures
#' @export

REDCAP_METADATA_FIELDTYPE <- c("calc", 
                               "checkbox", 
                               "descriptive", 
                               "dropdown", 
                               "file", 
                               "notes", 
                               "radio", 
                               "slider", 
                               "text", 
                               "truefalse", 
                               "yesno")

# Meta Data - Known Validation Types 

#' @rdname redcapDataStructures
#' @export
REDCAP_METADATA_VALIDATION_TYPE <- c("alpha_only",
                                     "date_dmy", 
                                     "date_mdy", 
                                     "date_ymd", 
                                     "datetime_dmy", 
                                     "datetime_mdy", 
                                     "datetime_seconds_dmy", 
                                     "datetime_seconds_mdy", 
                                     "datetime_seconds_ymd", 
                                     "datetime_ymd", 
                                     "email", 
                                     "integer", 
                                     "number", 
                                     "number_1dp", 
                                     "number_1dp_comma_decimal", 
                                     "number_2dp", 
                                     "number_2dp_comma_decimal",
                                     "phone", 
                                     "signature",
                                     "time", 
                                     "time_hh_mm_ss",
                                     "time_mm_ss",
                                     "zipcode")

# Project Information -----------------------------------------------
# Project Information Structure 

REDCAP_PROJECT_INFORMATION_STRUCTURE <- 
  data.frame(project_id = character(0),
             project_title = character(0),
             creation_time = character(0),
             production_time = character(0),
             in_production = character(0),
             project_language = character(0),
             purpose = character(0),
             purpose_other = character(0),
             project_notes = character(0),
             custom_record_label = character(0),
             secondary_unique_field = character(0),
             is_longitudinal = character(0),
             has_repeating_instruments_or_events = character(0),
             surveys_enabled = character(0),
             scheduling_enabled = character(0),
             record_autonumbering_enabled = character(0),
             randomization_enabled = character(0),
             ddp_enabled = character(0),
             project_irb_number = character(0),
             project_grant_number = character(0),
             project_pi_firstname = character(0),
             project_pi_lastname = character(0),
             display_today_now_button = character(0),
             missing_data_codes = character(0),
             external_modules = character(0),
             bypass_branching_erase_field_prompt = character(0), 
             stringsAsFactors = FALSE)

# This is the list of fields recognized for updates in importProjectInformation

PROJECT_INFO_FIELDS_EDITABLE <- 
  c("project_title", 
    "project_language", 
    "purpose", 
    "purpose_other", 
    "project_notes", 
    "custom_record_label",
    "secondary_unique_field", 
    "is_longitudinal", 
    "surveys_enabled", 
    "scheduling_enabled", 
    "record_autonumbering_enabled", 
    "randomization_enabled", 
    "project_irb_number", 
    "project_grant_number", 
    "project_pi_firstname", 
    "project_pi_lastname", 
    "display_today_now_button", 
    "bypass_branching_erase_field_prompt")

PROJECT_INFO_FIELDS_FIXED <- # These cannot be updated via the API
  c("project_id",
    "creation_time",
    "production_time",
    "in_production",
    "has_repeating_instruments_or_events",
    "ddp_enabled",
    "missing_data_codes",
    "external_modules")

# Repeating Instruments and Events ----------------------------------

#' @rdname redcapDataStructures
#' @export

REDCAP_REPEAT_INSTRUMENT_STRUCTURE <- 
  data.frame(event_name = character(0), 
             form_name = character(0), 
             custom_form_label = character(0))

# Users -------------------------------------------------------------
# Users Structure

REDCAP_USER_STRUCTURE <- 
  data.frame(username = character(0),
             email = character(0),
             firstname = character(0),
             lastname = character(0),
             expiration = as.POSIXct(character(0)),
             data_access_group = character(0),
             data_access_group_id = character(0),
             design = character(0),
             alerts = character(0),
             user_rights = character(0),
             data_access_groups = character(0),
             reports = character(0),
             stats_and_charts = character(0),
             manage_survey_participants = character(0),
             calendar = character(0),
             data_import_tool = character(0),
             data_comparison_tool = character(0),
             logging = character(0),
             file_repository = character(0),
             data_quality_create = character(0),
             data_quality_execute = character(0),
             api_export = character(0),
             api_import = character(0),
             mobile_app = character(0),
             mobile_app_download_data = character(0),
             record_create = character(0),
             record_rename = character(0),
             record_delete = character(0),
             lock_records_all_forms = character(0),
             lock_records = character(0),
             lock_records_customization = character(0),
             mycap_participants = character(0),
             random_setup = character(0),
             random_dashboard = character(0),
             random_perform = character(0),
             forms = character(0),
             forms_export = character(0), 
             stringsAsFactors = FALSE)

# These are variables in the Users table coded as 0 = No Access, 1 = Access
REDCAP_USER_TABLE_ACCESS_VARIABLES <- 
  c("design", 
    "alerts",
    "user_rights", 
    "data_access_group", 
    "reports", 
    "stats_and_charts", 
    "manage_survey_participants", 
    "calendar", 
    "data_import_tool", 
    "data_comparison_tool", 
    "logging", 
    "file_repository", 
    "data_quality_create", 
    "data_quality_execute", 
    "api_export", 
    "api_import", 
    "mobile_app", 
    "mobile_app_download_data", 
    "record_create", 
    "record_rename", 
    "record_delete", 
    "lock_records_all_forms", 
    "lock_records", 
    "lock_records_customization", 
    "mycap_participants", 
    "random_setup", 
    "random_dashboard", 
    "random_perform")

# User Roles --------------------------------------------------------
# User Role Structure

REDCAP_USER_ROLE_STRUCTURE <- 
  data.frame(unique_role_name = character(0),	
             role_label = character(0), 
             design = character(0),	
             user_rights = character(0),	
             data_access_groups = character(0),
             # data_export = character(0),
             reports = character(0),
             stats_and_charts = character(0),
             manage_survey_participants = character(0),
             calendar = character(0),
             data_import_tool = character(0),
             data_comparison_tool = character(0),
             logging = character(0),
             file_repository = character(0),
             data_quality_create = character(0),
             data_quality_execute = character(0),
             api_export = character(0),
             api_import = character(0),	
             mobile_app = character(0),
             mobile_app_download_data = character(0),
             record_create = character(0),
             record_rename = character(0),
             record_delete = character(0),
             lock_records_customization = character(0),
             lock_records = character(0),
             lock_records_all_forms = character(0),
             mycap_participants = character(0),
             forms = character(0),
             forms_export = character(0),
             random_setup = character(0),
             random_dashboard = character(0),
             random_perform = character(0),
             stringsAsFactors = FALSE)

# User Role Table Access Variables

REDCAP_USER_ROLE_TABLE_ACCESS_VARIABLES <- 
  c("design", 
    "alerts", 
    "user_rights", 
    "data_access_groups", 
    # "data_export",
    "reports", 
    "stats_and_charts", 
    "manage_survey_participants", 
    "calendar", 
    "data_import_tool", 
    "data_comparison_tool", 
    "logging", 
    "file_repository", 
    "data_quality_create", 
    "data_quality_execute", 
    "api_export", 
    "api_import", 
    "mobile_app", 
    "mobile_app_download_data", 
    "record_create", 
    "record_rename", 
    "record_delete", 
    "lock_records_customization", 
    "lock_records", 
    "lock_records_all_forms")

# User-Role Assignment Structure 

REDCAP_USER_ROLE_ASSIGNMENT_STRUCTURE <- 
  data.frame(username = character(0), 
             unique_role_name = character(0), 
             stringsAsFactors = FALSE)
