if (getRversion() >= "3.1.0") {
  utils::globalVariables(c(".", "artifact", "createdDate", "deeperThan3", "differs",
                           "fun", "hash", "i.hash", "iden", "N", "tag",
                           "tagKey", "tagValue"))
}

.reproEnv <- new.env(parent = asNamespace("reproducible"))

#' Cache method that accommodates environments, S4 methods, Rasters, & nested caching
#'
#' @details
#' Caching R objects using \code{\link[archivist]{cache}} has five important limitations:
#' \enumerate{
#'   \item the \code{archivist} package detects different environments as different;
#'   \item it also does not detect S4 methods correctly due to method inheritance;
#'   \item it does not detect objects that have file-base storage of information
#'         (specifically \code{\link[raster]{RasterLayer-class}} objects);
#'   \item the default hashing algorithm is relatively slow.
#'   \item heavily nested function calls may want Cache arguments to propagate through
#' }
#' This version of the \code{Cache} function accommodates those four special,
#' though quite common, cases by:
#' \enumerate{
#'   \item converting any environments into list equivalents;
#'   \item identifying the dispatched S4 method (including those made through
#'         inheritance) before hashing so the correct method is being cached;
#'   \item by hashing the linked file, rather than the Raster object.
#'         Currently, only file-backed \code{Raster*} objects are digested
#'         (e.g., not \code{ff} objects, or any other R object where the data
#'         are on disk instead of in RAM);
#'   \item Uses \code{\link[digest]{digest}} (formerly fastdigest, which does
#'         not translate between operating systems).
#'         This is used for file-backed objects as well.
#'   \item Cache will save arguments passed by user in a hidden environment. Any
#'         nested Cache functions will use arguments in this order 1) actual arguments
#'         passed at each Cache call, 2) any inherited arguments from an outer Cache
#'         call, 3) the default values of the Cache function. See section on \emph{Nested
#'         Caching}.
#' }
#'
#' If \code{Cache} is called within a SpaDES module, then the cached entry will automatically
#' get 3 extra \code{userTags}: \code{eventTime}, \code{eventType}, and \code{moduleName}.
#' These can then be used in \code{clearCache} to selectively remove cached objects
#' by \code{eventTime}, \code{eventType} or \code{moduleName}.
#'
#' \code{Cache} will add a tag to the artifact in the database called \code{accessed},
#' which will assign the time that it was accessed, either read or write.
#' That way, artifacts can be shown (using \code{showCache}) or removed (using
#' \code{clearCache}) selectively, based on their access dates, rather than only
#' by their creation dates. See example in \code{\link{clearCache}}.
#' \code{Cache} (uppercase C) is used here so that it is not confused with, and does
#' not mask, the \code{archivist::cache} function.
#'
#' @section Nested Caching:
#' Commonly, Caching is nested, i.e., an outer function is wrapped in a \code{Cache}
#' function call, and one or more inner functions are also wrapped in a \code{Cache}
#' function call. A user \emph{can} always specify arguments in every Cache function
#' call, but this can get tedious and can be prone to errors. The normal way that
#' \emph{R} handles arguments is it takes the user passed arguments if any, and
#' default arguments for all those that have no user passed arguments. We have inserted
#' a middle step. The order or precedence for any given \code{Cache} function call is
#' 1. user arguments, 2. inherited arguments, 3. default arguments. At this time,
#' the top level \code{Cache} arguments will propagate to all inner functions unless
#' each individual \code{Cache} call has other arguments specified, i.e., "middle"
#' nested \code{Cache} function calls don't propagate their arguments to further "inner"
#' \code{Cache} function calls.  See example.
#'
#' \code{userTags} is unique of all arguments: its values will be appended to the
#' inherited \code{userTags}.
#'
#' @section Caching Speed:
#' Caching speed may become a critical aspect of a final product. For example,
#' if the final product is a shiny app, rerunning the entire project may need
#' to take less then a few seconds at most. There are 3 arguments that affect
#' Cache speed: \code{quick}, \code{length}, and
#' \code{algo}. \code{quick} is passed to \code{.robustDigest}, which currently
#' only affects \code{Path} and \code{Raster*} class objects. In both cases, \code{quick}
#' means that little or no disk-based information will be assessed.
#'
#'
#' @section Filepaths:
#' If a function has a path argument, there is some ambiguity about what should be
#' done. Possibilities include:
#' \enumerate{
#'   \item hash the string as is (this will be very system specific, meaning a
#'         \code{Cache} call will not work if copied between systems or directories);
#'   \item hash the \code{basename(path)};
#'   \item hash the contents of the file.
#' }
#' If paths are passed in as is (i.e,. character string), the result will not be predictable.
#' Instead, one should use the wrapper function \code{asPath(path)}, which sets the
#' class of the string to a \code{Path}, and one should decide whether one wants
#' to digest the content of the file (using \code{quick = FALSE}),
#' or just the filename (\code{(quick = TRUE)}). See examples.
#'
#' @section Stochasticity:
#' In general, it is expected that caching will only be used when stochasticity
#' is not relevant, or if a user has achieved sufficient stochasticity (e.g., via
#' sufficient number of calls to \code{experiment}) such that no new explorations
#' of stochastic outcomes are required. It will also be very useful in a
#' reproducible workflow.
#'
#' @section useCache:
#' Logical or numeric. If \code{FALSE} or \code{0}, then the entire Caching
#' mechanism is bypassed and the
#' function is evaluated as if it was not being Cached. Default is
#' \code{getOption("reproducible.useCache")}), which is \code{TRUE} by default,
#' meaning use the Cache mechanism. This may be useful to turn all Caching on or
#' off in very complex scripts and nested functions. Increasing levels of numeric
#' values will cause deeper levels of Caching to occur. Currently, only implemented
#' in \code{postProcess}: to do both caching of inner \code{cropInputs}, \code{projectInputs}
#' and \code{maskInputs}, and caching of outer \code{postProcess}, use
#' \code{useCache = 2}; to skip the inner sequence of 3 functions, use \code{useCache = 1}.
#' For large objects, this may prevent many duplicated save to disk events.
#'
#' If \code{"overwrite"}
#' (which can be set with \code{options("reproducible.useCache" =
#' "overwrite")}), then the function invoke the caching mechanism but will purge
#' any entry that is matched, and it will be replaced with the results of the
#' current call.
#'
#' If \code{"devMode"}: The point of this mode is to facilitate using the Cache when
#' functions and datasets are continually in flux, and old Cache entries are
#' likely stale very often. In `devMode`, the cache mechanism will work as
#' normal if the Cache call is the first time for a function OR if it
#' successfully finds a copy in the cache based on the normal Cache mechanism.
#' It *differs* from the normal Cache if the Cache call does *not* find a copy
#' in the `cacheRepo`, but it does find an entry that matches based on
#' `userTags`. In this case, it will delete the old entry in the `cacheRepo`
#' (identified based on matching `userTags`), then continue with normal `Cache`.
#' For this to work correctly, `userTags` must be unique for each function call.
#' This should be used with caution as it is still experimental. Currently, if
#' \code{userTags} are not unique to a single entry in the cacheRepo, it will
#' default to the behaviour of \code{useCache = TRUE} with a message. This means
#' that \code{"devMode"} is most useful if used from the start of a project.
#'
#' @section \code{sideEffect}:
#' If \code{sideEffect} is not \code{FALSE}, then metadata about any files that
#' added to \code{sideEffect} will be added as an attribute to the cached copy.
#' Subsequent calls to this function
#'        will assess for the presence of the new files in the \code{sideEffect} location.
#'        If the files are identical (\code{quick = FALSE}) or their file size is
#'        identical (\code{quick = TRUE}), then the cached copy of the function will
#'        be returned (and no files changed). If there are missing or incorrect files,
#'        then the function will re-run. This will accommodate the situation where the
#'        function call is identical, but somehow the side effect files were modified.
#'        If \code{sideEffect} is logical, then the function will check the
#'        \code{cacheRepo}; if it is a path, then it will check the path. The function will
#'        assess whether the files to be downloaded are found locally
#'        prior to download. If it fails the local test, then it will try to recover from a
#'        local copy if (\code{makeCopy} had been set to \code{TRUE} the first time
#'        the function was run. Currently, local recovery will only work if\code{makeCOpy} was
#'        set to \code{TRUE} the first time \code{Cache}
#'        was run). Default is \code{FALSE}.
#'
#' @note As indicated above, several objects require pre-treatment before
#' caching will work as expected. The function \code{.robustDigest} accommodates this.
#' It is an S4 generic, meaning that developers can produce their own methods for
#' different classes of objects. Currently, there are methods for several types
#' of classes. See \code{\link{.robustDigest}}.
#'
#' See \code{\link{.robustDigest}} for other specifics for other classes.
#'
#' @inheritParams archivist::cache
#' @inheritParams archivist::saveToLocalRepo
#' @include cache-helpers.R
#' @include robustDigest.R
#'
#' @param FUN Either a function or an unevaluated function call (e.g., using
#'            \code{quote}.
#'
#' @param .objects Character vector of objects to be digested. This is only applicable
#'                if there is a list, environment (or similar) named objects
#'                within it. Only this/these objects will be considered for caching,
#'                i.e., only use a subset of
#'                the list, environment or similar objects.
#' @param outputObjects Optional character vector indicating which objects to
#'                      return. This is only relevant for list, environment (or similar) objects
#'
#' @param cacheRepo A repository used for storing cached objects.
#'                  This is optional if \code{Cache} is used inside a SpaDES module.
#'
#' @param length Numeric. If the element passed to Cache is a \code{Path} class
#'        object (from e.g., \code{asPath(filename)}) or it is a \code{Raster} with
#'        file-backing, then this will be
#'        passed to \code{digest::digest}, essentially limiting the number of bytes
#'        to digest (for speed). This will only be used if \code{quick = FALSE}.
#'        Default is \code{getOption("reproducible.length")},
#'        which is set to \code{Inf}.
#'
#' @param compareRasterFileLength Being deprecated; use \code{length}.
#'
#' @param omitArgs Optional character string of arguments in the FUN to omit from the digest.
#'
#' @param classOptions Optional list. This will pass into \code{.robustDigest} for
#'        specific classes. Should be options that the \code{.robustDigest} knows what
#'        to do with.
#'
#' @param debugCache Character or Logical. Either \code{"complete"} or \code{"quick"} (uses
#'        partial matching, so "c" or "q" work). \code{TRUE} is
#'        equivalent to \code{"complete"}.
#'        If \code{"complete"}, then the returned object from the Cache
#'        function will have two attributes, \code{debugCache1} and \code{debugCache2},
#'        which are the entire \code{list(...)} and that same object, but after all
#'        \code{.robustDigest} calls, at the moment that it is digested using
#'        \code{digest}, respectively. This \code{attr(mySimOut, "debugCache2")}
#'        can then be compared to a subsequent call and individual items within
#'        the object \code{attr(mySimOut, "debugCache1")} can be compared.
#'        If \code{"quick"}, then it will return the same two objects directly,
#'        without evalutating the \code{FUN(...)}.
#'
#' @param sideEffect Logical or path. Determines where the function will look for
#'        new files following function completion. See Details.
#'        \emph{NOTE: this argument is experimental and may change in future releases.}
#'
#' @param makeCopy Logical. If \code{sideEffect = TRUE}, and \code{makeCopy = TRUE},
#'        a copy of the downloaded files will be made and stored in the \code{cacheRepo}
#'        to speed up subsequent file recovery in the case where the original copy
#'        of the downloaded files are corrupted or missing. Currently only works when
#'        set to \code{TRUE} during the first run of \code{Cache}. Default is \code{FALSE}.
#'        \emph{NOTE: this argument is experimental and may change in future releases.}
#'
#' @param quick Logical. If \code{TRUE},
#'        little or no disk-based information will be assessed, i.e., mostly its
#'        memory content. This is relevant for objects of class \code{character},
#'        \code{Path} and \code{Raster} currently. For class \code{character}, it is ambiguous
#'        whether this represents a character string or a vector of file paths. The function
#'        will assess if it is a path to a file or directory first. If not, it will treat
#'        the object as a character string. If it is known that character strings should
#'        not be treated as paths, then \code{quick = TRUE} will be much faster, with no loss
#'        of information. If it is file or directory, then it will digest the file content,
#'        or \code{basename(object)}. For class \code{Path} objects, the file's metadata
#'        (i.e., filename and file size)
#'        will be hashed instead of the file contents if \code{quick = TRUE}.
#'        If set to \code{FALSE} (default),
#'        the contents of the file(s) are hashed.
#'        If \code{quick = TRUE}, \code{length} is ignored. \code{Raster} objects are treated
#'        as paths, if they are file-backed.
#'
#' @param verbose Numeric, with 0 being off, 1 being a little, 2 being more verbose etc.
#'        Above 1 will output much more information about the internals of
#'        Caching, which may help diagnose Caching challenges.
#'
#' @param cacheId Character string. If passed, this will override the calculated hash
#'        of the inputs, and return the result from this cacheId in the cacheRepo.
#'        Setting this is equivalent to manually saving the output of this function, i.e.,
#'        the object will be on disk, and will be recovered in subsequent
#'        This may help in some particularly finicky situations
#'        where Cache is not correctly detecting unchanged inputs. This will guarantee
#'        the object will be identical each time; this may be useful in operational code.
#'
#' @param useCache Logical, numeric or \code{"overwrite"} or \code{"devMode"}. See details.
#'
#' @param useCloud Logical. If \code{TRUE}, then this Cache call will download
#'   (if local copy doesn't exist,
#'   but cloud copy does exist), upload (local copy does or doesn't exist and
#'   cloud copy doesn't exist), or
#'   will not download nor upload if object exists in both. If \code{TRUE} will be at
#'   least 1 second slower than setting this to \code{FALSE}, and likely even slower as the
#'   cloud folder gets large. If a user wishes to keep "high-level" control, set this to
#'   \code{getOption("reproducible.useCloud", FALSE)} or
#'   \code{getOption("reproducible.useCloud", TRUE)} (if the default behaviour should
#'   be \code{FALSE} or \code{TRUE}, respectively) so it can be turned on and off with
#'   this option. NOTE: \emph{This argument will not be passed into inner/nested Cache calls.})
#'
#' @param cloudFolderID A googledrive id of a folder, e.g., using \code{drive_mkdir()}.
#'   If left as \code{NULL}, the function will create a cloud folder with a warning.
#'   The warning will have the \code{cloudFolderID} that should be used in subsequent calls.
#'   It will also be added to \code{options("reproducible.cloudFolderID")},
#'   but this will not persist across sessions.
#'
#' @param showSimilar A logical or numeric. Useful for debugging.
#'        If \code{TRUE} or \code{1}, then if the Cache
#'        does not find an identical archive in the cacheRepo, it will report (via message)
#'        the next most similar archive, and indicate which argument(s) is/are different.
#'        If a number larger than \code{1}, then it will report the N most similar archived
#'        objects.
#'
#' @inheritParams digest::digest
#'
#' @param digestPathContent Being deprecated. Use \code{quick}.
#'
#' @return As with \code{\link[archivist]{cache}}, returns the value of the
#' function call or the cached version (i.e., the result from a previous call
#' to this same cached function with identical arguments).
#'
#' @seealso \code{\link[archivist]{cache}}, \code{\link{.robustDigest}}
#'
#' @author Eliot McIntire
#' @export
#' @importClassesFrom raster RasterBrick
#' @importClassesFrom raster RasterLayer
#' @importClassesFrom raster RasterLayerSparse
#' @importClassesFrom raster RasterStack
#' @importClassesFrom sp Spatial
#' @importClassesFrom sp SpatialLines
#' @importClassesFrom sp SpatialLinesDataFrame
#' @importClassesFrom sp SpatialPixels
#' @importClassesFrom sp SpatialPixelsDataFrame
#' @importClassesFrom sp SpatialPoints
#' @importClassesFrom sp SpatialPointsDataFrame
#' @importClassesFrom sp SpatialPolygons
#' @importClassesFrom sp SpatialPolygonsDataFrame
#' @importFrom archivist cache loadFromLocalRepo saveToLocalRepo showLocalRepo
#' @importFrom archivist createLocalRepo addTagsRepo
#' @importFrom digest digest
#' @importFrom data.table setDT := setkeyv .N .SD setattr
#' @importFrom magrittr %>%
#' @importFrom stats na.omit
#' @importFrom utils object.size tail methods
#' @importFrom methods formalArgs
#' @importFrom tools file_path_sans_ext
#' @importFrom googledrive drive_mkdir drive_ls drive_upload drive_download
#' @rdname cache
#'
#' @example inst/examples/example_Cache.R
#'
setGeneric(
  "Cache", signature = "...",
  function(FUN, ..., notOlderThan = NULL,
           .objects = NULL, #objects = NULL,
           outputObjects = NULL, # nolint
           algo = "xxhash64", cacheRepo = NULL,
           length = getOption("reproducible.length", Inf),
           compareRasterFileLength, userTags = c(),
           digestPathContent, omitArgs = NULL,
           classOptions = list(), debugCache = character(),
           sideEffect = FALSE, makeCopy = FALSE,
           quick = getOption("reproducible.quick", FALSE),
           verbose = getOption("reproducible.verbose", 0), cacheId = NULL,
           useCache = getOption("reproducible.useCache", TRUE),
           useCloud = FALSE,
           cloudFolderID = getOption("reproducible.cloudFolderID", NULL),
           showSimilar = getOption("reproducible.showSimilar", FALSE)) {
    archivist::cache(cacheRepo, FUN, ..., notOlderThan, algo, userTags = userTags)
  })

#' @export
#' @rdname cache
setMethod(
  "Cache",
  definition = function(FUN, ..., notOlderThan, .objects,
                        #objects,
                        outputObjects,  # nolint
                        algo, cacheRepo, length, compareRasterFileLength, userTags,
                        digestPathContent, omitArgs, classOptions,
                        debugCache, sideEffect, makeCopy, quick, verbose,
                        cacheId, useCache,
                        showSimilar) {

    if (!is.null(list(...)$objects)) {
      message("Please use .objects (if trying to pass to Cache) instead of objects which is being deprecated")
    }

    if (missing(FUN)) stop("Cache requires the FUN argument")

    # returns "modifiedDots", "originalDots", "FUN", "funName", which will
    #  have modifications under many circumstances, e.g., do.call, specific methods etc.
    fnDetails <- .fnCleanup(FUN = FUN, callingFun = "Cache", ...)

    FUN <- fnDetails$FUN
    modifiedDots <- fnDetails$modifiedDots
    originalDots <- fnDetails$originalDots

    if (isFALSE(useCache) || isTRUE(0 == useCache)) {
      message(crayon::green("useCache is FALSE, skipping Cache.",
                            "To turn Caching on, use options(reproducible.useCache = TRUE)"))
      if (fnDetails$isDoCall) {
        do.call(modifiedDots$what, args = modifiedDots$args)
      } else {
        do.call(FUN, args = modifiedDots)
      }
    } else {

      startCacheTime <- verboseTime(verbose)

      if (!missing(compareRasterFileLength)) {
        message("compareRasterFileLength argument being deprecated. Use 'length'")
        length <- compareRasterFileLength
      }
      if (!missing(digestPathContent)) {
        message("digestPathContent argument being deprecated. Use 'quick'.")
        quick <- !digestPathContent
      }

      nestedTags <- determineNestedTags(envir = environment(),
                                                  mc = match.call(expand.dots = TRUE),
                                                  userTags = userTags)
      userTags <- unique(c(userTags, .reproEnv$userTags))
      if (any(!nestedTags$objOverride)) {
        on.exit({
          if (any(!nestedTags$prevVals)) {
            # THe suppressWarnings is about objects that aren't there -- so far only happens
            #  when interrupting a process, which means it is spurious
            suppressWarnings(rm(list = nestedTags$namesUserCacheArgs,
                                envir = .reproEnv))
            if (nestedTags$prevUserTags)
              .reproEnv$userTags <- nestedTags$oldUserTags
          }
          if (nestedTags$prevUserTags) {
            .reproEnv$userTags <- nestedTags$oldUserTags
          }
        }, add = TRUE)
      }

      # get cacheRepo if not supplied
      cacheRepos <- getCacheRepos(cacheRepo, modifiedDots)
      cacheRepo <- cacheRepos[[1]]

      if (fnDetails$isPipe) {
        pipeRes <- .CachePipeFn1(modifiedDots, fnDetails, FUN)
        modifiedDots <- pipeRes$modifiedDots
        fnDetails <- pipeRes$fnDetails
      }

      modifiedDots$.FUN <- fnDetails$.FUN # put in modifiedDots for digesting  # nolint

      # This is for Pipe operator -- needs special consideration
      scalls <- if (!is(FUN, "function")) .CacheFn1(FUN, sys.calls()) else NULL

      # extract other function names that are not the ones the focus of the Cache call
      otherFns <- .getOtherFnNamesAndTags(scalls = scalls)

      if (missing(notOlderThan)) notOlderThan <- NULL

      # if a simList is in ...
      # userTags added based on object class
      userTags <- c(userTags, unlist(lapply(modifiedDots, .tagsByClass)))

      if (sideEffect != FALSE) if (isTRUE(sideEffect)) sideEffect <- cacheRepo

      isIntactRepo <- unlist(lapply(cacheRepos, function(cacheRepo) {
        all(file.exists(file.path(cacheRepo,  c("gallery", "backpack.db"))))
      }))
      if (any(!isIntactRepo))
        ret <- lapply(seq(cacheRepos)[!isIntactRepo], function(cacheRepoInd) {
          archivist::createLocalRepo(cacheRepos[[cacheRepoInd]],
                                     force = isIntactRepo[cacheRepoInd])
        })


      # List file prior to cache
      if (sideEffect != FALSE) {
        priorRepo <- list.files(sideEffect, full.names = TRUE)
      }

      # remove things in the Cache call that are not relevant to Caching
      if (!is.null(modifiedDots$progress)) if (!is.na(modifiedDots$progress)) modifiedDots$progress <- NULL

      # Do the digesting
      if (!is.null(omitArgs)) {
        modifiedDots[omitArgs] <- NULL
      }

      # don't digest the dotPipe elements as they are already
      # extracted individually into modifiedDots list elements
      dotPipe <- startsWith(names(modifiedDots), "._")
      preDigestByClass <- lapply(seq_along(modifiedDots[!dotPipe]), function(x) {
        .preDigestByClass(modifiedDots[!dotPipe][[x]])
      })

      startHashTime <- verboseTime(verbose)

      # remove some of the arguments passed to Cache, which are irrelevant for digest
      argsToOmitForDigest <- dotPipe |
        (names(modifiedDots) %in%
           .defaultCacheOmitArgs)

      cacheDigest <- CacheDigest(modifiedDots[!argsToOmitForDigest], .objects = .objects,
                                 length = length, algo = algo, quick = quick,
                                 classOptions = classOptions)
      preDigest <- cacheDigest$preDigest
      outputHash <- cacheDigest$outputHash

      # This does to depth 3
      preDigestUnlistTrunc <- unlist(
        .unlistToCharacter(preDigest, getOption("reproducible.showSimilarDepth", 3)))

      if (verbose > 1) {
        a <- .CacheVerboseFn1(preDigest, fnDetails,
                              startHashTime, modifiedDots, dotPipe, quick = quick)
        on.exit({
          assign("cacheTimings", .reproEnv$verboseTiming, envir = .reproEnv)
          print(.reproEnv$verboseTiming)
          message("This object is also available from .reproEnv$cacheTimings")
          if (exists("verboseTiming", envir = .reproEnv))
            rm("verboseTiming", envir = .reproEnv)
        },
        add = TRUE)
      }

      if (length(debugCache)) {
        if (!is.na(pmatch(debugCache, "quick")))
          return(list(hash = preDigest, content = list(...)))
      }

      if (!is.null(cacheId)) {
        outputHashManual <- cacheId
        if (identical(outputHashManual, outputHash)) {
          message("cacheId is same as calculated hash")
        } else {
          message("cacheId is not same as calculated hash. Manually searching for cacheId:", cacheId)
        }
        outputHash <- outputHashManual
      }

      # compare outputHash to existing Cache record
      tries <- 1
      if (useCloud) {
        # Here, test that cloudFolderID exists and get obj details that matches outputHash, if present
        #  returns NROW 0 gdriveLs if not present
        cloudFolderID <- checkAndMakeCloudFolderID(cloudFolderID)
        message("Retrieving file list in cloud folder")
        gdriveLs <- retry(drive_ls(path = as_id(cloudFolderID), pattern = outputHash))
      }
      while (tries <= length(cacheRepos)) {
        repo <- cacheRepos[[tries]]
        tries <- tries + 1
        localTags <- getLocalTags(repo)
        isInRepo <- localTags[localTags$tag == paste0("cacheId:", outputHash), , drop = FALSE]
        if (NROW(isInRepo) > 1) isInRepo <- isInRepo[NROW(isInRepo),]
        if (NROW(isInRepo) > 0) {
          cacheRepo <- repo
          break
        }
      }

      userTags <- c(userTags, if (!is.na(fnDetails$functionName))
        paste0("function:", fnDetails$functionName)
      )

      outputHashNew <- outputHash # Keep a copy of this because it may be replaced next, but we need to know old one

      # First, if this is not matched by outputHash, test that it is matched by
      #   userTags and in devMode
      needFindByTags <- identical("devMode", useCache) &&
        NROW(isInRepo) == 0
      if (identical("devMode", useCache) && NROW(isInRepo) == 0) {
        devModeOut <- devModeFn1(localTags, userTags, scalls, preDigestUnlistTrunc, useCache, verbose, isInRepo, outputHash)
        outputHash <- devModeOut$outputHash
        isInRepo <- devModeOut$isInRepo
        needFindByTags <- devModeOut$needFindByTags
      }

      if (identical("overwrite", useCache)  && NROW(isInRepo) > 0 || needFindByTags) {
        suppressMessages(clearCache(x = cacheRepo, userTags = outputHash, ask = FALSE))
        if (identical("devMode", useCache)) {
          isInRepo <- isInRepo[!isInRepo$tag %in% userTags, , drop = FALSE]
          outputHash <- outputHashNew
          message("Overwriting Cache entry with userTags: '",paste(userTags, collapse = ", ") ,"'")
        } else {
          isInRepo <- isInRepo[isInRepo$tag != paste0("cacheId:", outputHash), , drop = FALSE]
          message("Overwriting Cache entry with function '",fnDetails$functionName ,"'")
        }
      }

      # If it is in the existing record:
      if (NROW(isInRepo) > 0) {
        # make sure the notOlderThan is valid, if not, exit this loop
        lastEntry <- max(isInRepo$createdDate)
        lastOne <- order(isInRepo$createdDate, decreasing = TRUE)[1]
        if (is.null(notOlderThan) || (notOlderThan < lastEntry)) {
          objSize <- file.size(file.path(cacheRepo, "gallery", paste0(isInRepo$artifact, ".rda")))
          class(objSize) <- "object_size"
          if (objSize > 1e6)
            message(crayon::blue(paste0("  ...(Object to retrieve is large: ", format(objSize, units = "auto"), ")")))
          output <- try(.getFromRepo(FUN, isInRepo = isInRepo, notOlderThan = notOlderThan,
                                 lastOne = lastOne, cacheRepo = cacheRepo,
                                 fnDetails = fnDetails,
                                 modifiedDots = modifiedDots, debugCache = debugCache,
                                 verbose = verbose, sideEffect = sideEffect,
                                 quick = quick, algo = algo,
                                 preDigest = preDigest, startCacheTime = startCacheTime,
                                 ...))
          if (is(output, "try-error")) {
            cID <- gsub("cacheId:", "", isInRepo$tag)
            stop("Error in trying to recover cacheID: ", cID,
                 "\nYou will likely need to remove that item from Cache, e.g., ",
                 "\nclearCache(userTags = '", cID, "')")
          }

          if (useCloud) {
            # Here, upload local copy to cloud folder
            isInCloud <- cloudUpload(isInRepo, outputHash, gdriveLs, cacheRepo, cloudFolderID, output)
          }

          return(output)
        }
      } else {
        # find similar -- in progress
        if (!is.null(showSimilar)) { # TODO: Needs testing
          if (!isFALSE(showSimilar)) {
            .findSimilar(localTags, showSimilar, scalls, preDigestUnlistTrunc,
                         userTags, useCache = useCache)
          }
        }
      }

      startRunTime <- verboseTime(verbose)

      .CacheIsNew <- TRUE
      if (useCloud) {
        # Here, download cloud copy to local folder, skip the running of FUN
        newFileName <- paste0(outputHash,".rda")
        isInCloud <- gsub(gdriveLs$name, pattern = "\\.rda", replacement = "") %in% outputHash
        if (any(isInCloud)) {
          output <- cloudDownload(outputHash, newFileName, gdriveLs, cacheRepo, cloudFolderID)
          if (is.null(output)) {
            retry(drive_rm(as_id(gdriveLs$id[isInCloud])))
            isInCloud[isInCloud] <- FALSE
          } else {
            .CacheIsNew <- FALSE
          }

        }
      }

      # check that it didn't come from cloud or failed to find complete cloud (i.e., output is NULL)
      if (!exists("output", inherits = FALSE) || is.null(output)) {
        # Run the FUN
        if (fnDetails$isPipe) {
          output <- eval(modifiedDots$._pipe, envir = modifiedDots$._envir)
        } else {
          output <- FUN(...)
        }
      }

      output <- .addChangedAttr(output, preDigest, origArguments = modifiedDots[!dotPipe],
                                .objects = outputObjects, length = length,
                                algo = algo, quick = quick, classOptions = classOptions, ...)
      verboseDF1(verbose, fnDetails$functionName, startRunTime)

      # Delete previous version if notOlderThan violated --
      #   but do this AFTER new run on previous line, in case function call
      #   makes it crash, or user interrupts long function call and wants
      #   a previous version
      if (nrow(isInRepo) > 0) {
        # flush it if notOlderThan is violated
        if (notOlderThan >= lastEntry) {
          suppressMessages(clearCache(userTags = isInRepo$artifact[lastOne], x = cacheRepo,
                                      ask = FALSE))
        }
      }

      # need something to attach tags to if it is actually NULL
      isNullOutput <- if (is.null(output)) TRUE else FALSE
      if (isNullOutput) output <- "NULL"

      .setSubAttrInList(output, ".Cache", "newCache", .CacheIsNew)
      setattr(output, "tags", paste0("cacheId:", outputHash))
      setattr(output, "call", "")
      # attr(output, "tags") <- paste0("cacheId:", outputHash)
      # attr(output, ".Cache")$newCache <- TRUE
      # attr(output, "call") <- ""
      if (!identical(attr(output, ".Cache")$newCache, .CacheIsNew)) stop("attributes are not correct 3")
      if (!identical(attr(output, "call"), "")) stop("attributes are not correct 4")
      if (!identical(attr(output, "tags"), paste0("cacheId:", outputHash))) stop("attributes are not correct 5")

      if (sideEffect != FALSE) {
        output <- .CacheSideEffectFn2(sideEffect, cacheRepo, priorRepo, algo, output,
                                      makeCopy, quick)
      }

      if (isS4(FUN)) {
        setattr(output, "function", FUN@generic)
        #attr(output, "function") <- FUN@generic
        if (!identical(attr(output, "function"), FUN@generic))
          stop("There is an unknown error 03")
      }
      # Can make new methods by class to add tags to outputs
      outputToSave <- .addTagsToOutput(output, outputObjects, FUN,
                                       preDigestByClass)

      # Remove from otherFunctions if it is "function"
      alreadyIn <- gsub(otherFns, pattern = "otherFunctions:", replacement = "") %in%
        as.character(attr(output, "function"))
      if (isTRUE(any(alreadyIn)))
        otherFns <- otherFns[!alreadyIn]

      outputToSaveIsList <- is(outputToSave, "list") # is.list is TRUE for anything, e.g., data.frame. We only want "list"
      if (outputToSaveIsList) {
        rasters <- unlist(lapply(outputToSave, is, "Raster"))
      } else {
        rasters <- is(outputToSave, "Raster")
      }
      if (any(rasters)) {
        if (outputToSaveIsList) {
          outputToSave[rasters] <- lapply(outputToSave[rasters], function(x)
            .prepareFileBackedRaster(x, repoDir = cacheRepo, overwrite = FALSE))
        } else {
          outputToSave <- .prepareFileBackedRaster(outputToSave, repoDir = cacheRepo,
                                                   overwrite = FALSE)
        }

        setattr(outputToSave, "tags", attr(output, "tags"))
        .setSubAttrInList(outputToSave, ".Cache", "newCache", attr(output, ".Cache")$newCache)
        setattr(outputToSave, "call", attr(output, "call"))

        # attr(outputToSave, "tags") <- attr(output, "tags")
        # attr(outputToSave, "call") <- attr(output, "call")
        # attr(outputToSave, ".Cache")$newCache <- attr(output, ".Cache")$newCache
        if (!identical(attr(outputToSave, ".Cache")$newCache, attr(output, ".Cache")$newCache))
          stop("attributes are not correct 6")
        if (!identical(attr(outputToSave, "call"), attr(output, "call")))
          stop("attributes are not correct 7")
        if (!identical(attr(outputToSave, "tags"), attr(output, "tags")))
          stop("attributes are not correct 8")

        if (isS4(FUN)) {
          setattr(outputToSave, "function", attr(output, "function"))
          if (!identical(attr(outputToSave, "function"), attr(output, "function")))
            stop("There is an unknown error 04")
        }
        # attr(outputToSave, "function") <- attr(output, "function")

        output <- outputToSave
      }
      if (length(debugCache)) {
        if (!is.na(pmatch(debugCache, "complete"))) {
          output <- .debugCache(output, preDigest, ...)
          outputToSave <- .debugCache(outputToSave, preDigest, ...)
        }
      }

      startSaveTime <- verboseTime(verbose)
      # This is for write conflicts to the SQLite database
      #   (i.e., keep trying until it is written)

      objSize <- .objSizeInclEnviros(outputToSave)
      userTags <- c(userTags,
                    paste0("object.size:", objSize),
                    paste0("accessed:", Sys.time()),
                    paste0(otherFns),
                    paste("preDigest", names(preDigestUnlistTrunc),
                          preDigestUnlistTrunc, sep = ":")
                    )

      written <- 0

      useFuture <- FALSE
      .onLinux <- .Platform$OS.type == "unix" && unname(Sys.info()["sysname"]) == "Linux"
      if (.onLinux) {
        if (!isFALSE(getOption("reproducible.futurePlan")) &&
            requireNamespace("future", quietly = TRUE)) {
          useFuture <- TRUE
        }
      }
      if (useFuture) {
        if (exists("futureEnv", envir = .reproEnv))
          .reproEnv$futureEnv <- new.env()

        if (isTRUE(getOption("reproducible.futurePlan"))) {
          message('options("reproducible.futurePlan") is TRUE. Setting it to "multiprocess"\n',
                  'Please specify a plan by name, e.g., options("reproducible.futurePlan" = "multiprocess")')
          future::plan("multiprocess")
        } else {
          if (!is(future::plan(), getOption("reproducible.futurePlan"))) {
            thePlan <- getOption("reproducible.futurePlan")
            future::plan(thePlan)
          }
        }
        .reproEnv$futureEnv[[paste0("future_", rndstr(1,10))]] <-
          #saved <-
          future::futureCall(
            FUN = writeFuture,
            args = list(written, outputToSave, cacheRepo, userTags),
            globals = list(written = written,
                           saveToLocalRepo = archivist::saveToLocalRepo,
                           outputToSave = outputToSave,
                           cacheRepo = cacheRepo,
                           userTags = userTags)
          )
        if (is.null(.reproEnv$alreadyMsgFuture)) {
          message("  Cache saved in a separate 'future' process. ",
                  "Set options('reproducible.futurePlan' = FALSE), if there is strange behaviour.",
                  "This message will not be shown again until next reload of reproducible")
          .reproEnv$alreadyMsgFuture <- TRUE
        }
      } else {
        while (written >= 0) {
          otsObjSize <- gsub(grep("object.size", userTags, value = TRUE),
                             pattern = "object.size:", replacement = "")
          otsObjSize <- as.numeric(otsObjSize)
          class(otsObjSize) <- "object_size"

          if (otsObjSize > 1e7)
            message("Saving large object to Cache: ", format(otsObjSize, units = "auto"))
          saved <- suppressWarnings(try(silent = TRUE,
                                        saveToLocalRepo(
                                          outputToSave,
                                          repoDir = cacheRepo,
                                          artifactName = NULL,
                                          archiveData = FALSE,
                                          archiveSessionInfo = FALSE,
                                          archiveMiniature = FALSE,
                                          rememberName = FALSE,
                                          silent = TRUE,
                                          userTags = userTags
                                        )
          ))

          # This is for simultaneous write conflicts. SQLite on Windows can't handle them.
          written <- if (is(saved, "try-error")) {
            Sys.sleep(sum(runif(written + 1, 0.05, 0.1)))
            written + 1
          } else {
            -1
          }
        }

      }

      if (useCloud) {
        # Here, upload local copy to cloud folder if it isn't already there
        cloudUploadFromCache(isInCloud, outputHash, saved, cacheRepo, cloudFolderID, outputToSave, rasters)
      }

      verboseDF2(verbose, fnDetails$functionName, startSaveTime)

      verboseDF3(verbose, fnDetails$functionName, startCacheTime)

      if (isNullOutput) return(NULL) else return(output)
    }
})

#' @keywords internal
.formalsCache <- formals(Cache)[-(1:2)]

#' @keywords internal
.formalsCache[c("compareRasterFileLength", "digestPathContent")] <- NULL

#' @keywords internal
.namesCacheFormals <- names(.formalsCache)[]

#' @keywords internal
.loadFromLocalRepoMem2 <- function(md5hash, ...) {
  out <- loadFromLocalRepo(md5hash, ...)
  out <- makeMemoisable(out)
  return(out)
}

#' @keywords internal
.loadFromLocalRepoMem <- memoise::memoise(.loadFromLocalRepoMem2)

#' @keywords internal
.unlistToCharacter <- function(l, max.level = 1) {
  if (max.level > 0) {
    lapply(l, function(l1) {
      if (is.character(l1)) {
        l1
      } else {
        if (is.list(l1)) {
          .unlistToCharacter(l1, max.level = max.level - 1)
        } else {
          "not list"
        }
      }
    })
  } else {
    "other"
  }
}

#' Generic method to make or unmake objects memoisable
#'
#' This is just a pass through for all classes in \pkg{reproducible}.
#' This generic is here so that downstream methods can be created.
#'
#' @param x  An object to make memoisable.
#'           See individual methods in other packages.
#' @return The same object, but with any modifications, especially
#' dealing with saving of environments, which memoising doesn't handle
#' correctly in some cases.
#'
#' @export
#' @rdname makeMemoisable
makeMemoisable <- function(x) {
  UseMethod("makeMemoisable")
}

#' @export
#' @rdname makeMemoisable
makeMemoisable.default <- function(x) {
  x
}

#' @export
#' @rdname makeMemoisable
unmakeMemoisable <- function(x) {
  UseMethod("unmakeMemoisable")
}

#' @export
#' @rdname makeMemoisable
unmakeMemoisable.default <- function(x) {
  x
}

#' @inheritParams archivist showLocalRepo
#' @importFrom fastdigest fastdigest
showLocalRepo2 <- function(repoDir, algo = "xxhash64") {
  #if (requireNamespace("future"))
  #  checkFutures() # will pause until all futures are done
  aa <- showLocalRepo(repoDir) # much faster than showLocalRepo(repoDir, "tags")
  dig <- fastdigest(aa$md5hash) # this is only on local computer, don't use digest::digest; too slow
  bb <- showLocalRepo3Mem(repoDir, dig)
  return(bb)
}

showLocalRepo3 <- function(repoDir, dig) {
  showLocalRepo(repoDir, "tags")
}

showLocalRepo3Mem <- memoise::memoise(showLocalRepo3)

#' Write to archivist repository, using \code{future::future}
#'
#' This will be used internally if \code{options("reproducible.futurePlan" = TRUE)}.
#' This is still experimental.
#'
#' @param written Integer. If zero or positive then it needs to be written still.
#'                Should be 0 to start.
#' @param outputToSave The R object to save to repository
#' @param cacheRepo The file path of the repository
#' @param userTags Character string of tags to attach to this \code{outputToSave} in
#'                 the \code{CacheRepo}
#'
#' @export
#' @importFrom archivist saveToLocalRepo
#' @importFrom stats runif
writeFuture <- function(written, outputToSave, cacheRepo, userTags) {
  counter <- 0
  if (!file.exists(file.path(cacheRepo, "backpack.db"))) {
    stop("That cacheRepo does not exist")
  }
  while (written >= 0) {
    #future::plan(multiprocess)
    saved <- #suppressWarnings(
      try(#silent = TRUE,
        saveToLocalRepo(
          outputToSave,
          repoDir = cacheRepo,
          artifactName = NULL,
          archiveData = FALSE,
          archiveSessionInfo = FALSE,
          archiveMiniature = FALSE,
          rememberName = FALSE,
          silent = TRUE,
          userTags = userTags
        )
        #)
      )

    # This is for simultaneous write conflicts. SQLite on Windows can't handle them.
    written <- if (is(saved, "try-error")) {
      Sys.sleep(sum(runif(written + 1, 0.05, 0.1)))
      written + 1
    } else {
      -1
    }
    counter <- counter + 1
    if (isTRUE(startsWith(saved[1], "Error in checkDirectory")))
      stop(saved)
    if (counter > 10) stop("Can't write to cacheRepo")
  }
  return(saved)
}

.fnCleanup <- function(FUN, ..., callingFun) {
  modifiedDots <- list(...)
  isPipe <- isTRUE(!is.null(modifiedDots$._pipe))
  originalDots <- modifiedDots

  # If passed with 'quote'
  if (!is.function(FUN)) {
    parsedFun <- parse(text = FUN)
    evaledParsedFun <- eval(parsedFun[[1]])
    if (is.function(evaledParsedFun)) {
      tmpFUN <- evaledParsedFun
      mc <- match.call(tmpFUN, FUN)
      FUN <- tmpFUN # nolint
      originalDots <- append(originalDots, as.list(mc[-1]))
      modifiedDots <- append(modifiedDots, as.list(mc[-1]))
    }
    fnDetails <- list(functionName = as.character(parsedFun[[1]]))
  } else {
    if (!isPipe) {
      fnDetails <- getFunctionName(FUN, #...,
                                   originalDots = originalDots,
                                   isPipe = isPipe)

      # i.e., if it did extract the name
      if (!is.na(fnDetails$functionName)) {
        if (is.primitive(FUN)) {
          modifiedDots <- list(...)
        } else {
          modifiedDots <- as.list(
            match.call(FUN, as.call(list(FUN, ...))))[-1]
        }
      }
    } else {
      fnDetails <- list()
    }
  }

  isDoCall <- FALSE
  forms <- suppressWarnings(names(formals(FUN)))
  if (!is.null(fnDetails$functionName)) {
    if (!is.na(fnDetails$functionName)) {
      if (fnDetails$functionName == "do.call") {
        isDoCall <- TRUE
        possFunNames <- lapply(substitute(placeholderFunction(...))[-1],
                               deparse, backtick = TRUE)
        #doCallMatched <- as.list(match.call(modifiedDots$what, call = as.call(append(list(modifiedDots$what), modifiedDots$args))))
        doCallMatched <- as.list(match.call(do.call, as.call(append(list(do.call), possFunNames))))
        whatArg <- doCallMatched$what

        whArgs <- which(names(modifiedDots) %in% "args")
        doCallFUN <- modifiedDots$what

        forms <- names(formals(doCallFUN))
        if (isS4(doCallFUN)) {
          fnName <- doCallFUN@generic

          # Not easy to selectMethod -- can't have trailing "ANY" -- see ?selectMethod last
          #  paragraph of "Using findMethod()" which says:
          # "Notice also that the length of the signature must be what the corresponding
          #  package used. If thisPkg had only methods for one argument, only length-1
          # signatures will match (no trailing "ANY"), even if another currently loaded
          # package had signatures with more arguments.
          numArgsInSig <- try({
            suppressWarnings(info <- attr(utils::methods(fnName), "info")) # from hadley/sloop package s3_method_generic
            max(unlist(lapply(strsplit(rownames(info), split = ","), length) ) - 1)
          }, silent = TRUE)
          matchOn <- doCallFUN@signature[seq(numArgsInSig)]

          mc <- as.list(match.call(doCallFUN, as.call(append(fnName, modifiedDots[[whArgs]])))[-1])
          mc <- mc[!unlist(lapply(mc, is.null))]
          argsClasses <- unlist(lapply(mc, function(x) class(x)[1]))
          argsClasses <- argsClasses[names(argsClasses) %in% matchOn]
          missingArgs <- matchOn[!(matchOn %in% names(argsClasses))]

          missings <- rep("missing", length(missingArgs))
          names(missings) <- missingArgs
          argsClasses <- c(argsClasses, missings)

          forms <- names(formals(selectMethod(fnName, signature = argsClasses)))
          fnDetails$functionName <- fnName
        } else {
          classes <- try({
            suppressWarnings(info <- attr(utils::methods(whatArg), "info")) # from hadley/sloop package s3_method_generic
            classes <- unlist(lapply(strsplit(rownames(info), split = "\\."), function(x) x[[2]]))
            gsub("-method$", "", classes)
          }, silent = TRUE)
          if (is(classes, "try-error")) classes <- NA_character_
          mc <- as.list(match.call(doCallFUN, as.call(append(whatArg, modifiedDots[[whArgs]])))[-1])
          theClass <- classes[unlist(lapply(classes, function(x) inherits(mc[[1]], x)))]
          forms <- if (length(theClass)) {
            aa <- try(names(formals(paste0(whatArg, ".", theClass))))
            aa
          } else {
            if (all(is.na(classes))) {
              names(formals(doCallFUN))
            } else {
              names(formals(whatArg))
            }

          }
        }
      }
    }
  }
  # Determine if some of the Cache arguments are also arguments to FUN
  callingFunFormals <- if (callingFun == "Cache") .namesCacheFormals else names(formals(callingFun))
  if (isDoCall) {
    argNamesOfAllClasses <- forms
    fnDetails$.FUN <- format(doCallFUN) # nolint
    formalsInCallingAndFUN <- argNamesOfAllClasses[argNamesOfAllClasses %in% callingFunFormals]
  } else {
    fnDetails$.FUN <- format(FUN) # nolint
    formalsInCallingAndFUN <- forms[forms %in% callingFunFormals]
  }

  # If arguments to FUN and Cache are identical, pass them through to FUN
  if (length(formalsInCallingAndFUN)) {
    formalsInCallingAndFUN <- grep("\\.\\.\\.", formalsInCallingAndFUN, value = TRUE, invert = TRUE)
    commonArguments <- try(mget(formalsInCallingAndFUN, inherits = FALSE, envir = parent.frame()),
                           silent = TRUE)
    if (!is(commonArguments, "try-error")) {
      if (isDoCall) {
        modifiedDots$args[formalsInCallingAndFUN] <- commonArguments
      } else {
        modifiedDots[formalsInCallingAndFUN] <- commonArguments
      }
    }
  }
  return(append(fnDetails, list(originalDots = originalDots, FUN = FUN, isPipe = isPipe,
                                modifiedDots = modifiedDots, isDoCall = isDoCall,
                                formalArgs = forms)))
}

#' Set subattributes within a list by reference
#'
#' This uses \code{data.table::setattr}, but in the case where there is
#' only a single element within a list attribute.
#' @param object An arbitrary object
#' @param attr The attribute name (that is a list object) to change
#' @param subAttr The list element name to change
#' @param value The new value
#'
#' @export
#' @importFrom data.table setattr
#' @rdname setSubAttrInList
.setSubAttrInList <- function(object, attr, subAttr, value) {
  .CacheAttr <- attr(object, attr)
  if (is.null(.CacheAttr)) .CacheAttr <- list()
  .CacheAttr[[subAttr]] <- value
  setattr(object, attr, .CacheAttr)
}

#' The exact digest function that \code{Cache} uses
#'
#' This can be used by a user to pre-test their arguments before running
#' \code{Cache}, for example to determine whether there is a cached copy.
#'
#'
#' @param ... passed to \code{.robustDigest}; this is generally empty except
#'    for advanced use.
#' @param objsToDigest A list of all the objects (e.g., arguments) to be digested
#' @param calledFrom a Character string, length 1, with the function to
#'    compare with. Default is "Cache". All other values may not produce
#'    robust CacheDigest results.
#'
#' @inheritParams Cache
#' @importFrom fastdigest fastdigest
#'
#' @return
#' A list of length 2 with the \code{outputHash}, which is the digest
#' that Cache uses for \code{cacheId} and also \code{preDigest}, which is
#' the digest of each sub-element in \code{objsToDigest}.
#'
#' @export
#'
#' @examples
#' \dontrun{
#'   a <- Cache(rnorm, 1)
#'   CacheDigest(list(rnorm, 1))
#'
#' }
CacheDigest <- function(objsToDigest, algo = "xxhash64", calledFrom = "Cache", ...) {
  if (identical("Cache", calledFrom)) {
    namesOTD <- names(objsToDigest)
    lengthChars <- nchar(namesOTD)
    if (!any(namesOTD == "FUN")) {
      zeroLength <- which(lengthChars == 0)
      if (sum(zeroLength ) > 0) {
        names(objsToDigest)[zeroLength[1]] <- ".FUN"
      }
    }
  }

  # need to omit arguments that are in Cache function call
  objsToDigest[names(objsToDigest) %in% .defaultCacheOmitArgs] <- NULL

  preDigest <- lapply(objsToDigest, function(x) {
    # remove the "newCache" attribute, which is irrelevant for digest
    if (!is.null(attr(x, ".Cache")$newCache)) {
      .setSubAttrInList(x, ".Cache", "newCache", NULL)
      # attr(x, ".Cache")$newCache <- NULL
      if (!identical(attr(x, ".Cache")$newCache, NULL)) stop("attributes are not correct 1")
    }
    .robustDigest(x, algo = algo, ...)
  })

  res <- if (isTRUE(getOption("reproducible.useNewDigestAlgorithm"))) {
    .robustDigest(unname(sort(unlist(preDigest))), algo = algo, ...)
  } else {
    fastdigest(preDigest)
  }
  list(outputHash = res, preDigest = preDigest)
}

#' @importFrom data.table setDT setkeyv melt
#' @keywords internal
.findSimilar <- function(localTags, showSimilar, scalls, preDigestUnlistTrunc, userTags,
                         useCache = getOption("reproducible.useCache", TRUE)) {
  setDT(localTags)
  isDevMode <- identical("devMode", useCache)
  if (isDevMode) {
    showSimilar <- 1
  }
  userTags2 <- .getOtherFnNamesAndTags(scalls = scalls)
  userTags2 <- c(userTags2, paste("preDigest", names(preDigestUnlistTrunc), preDigestUnlistTrunc, sep = ":")) #nolint
  userTags3 <- c(userTags, userTags2)
  aa <- localTags[tag %in% userTags3][,.N, keyby = artifact]
  setkeyv(aa, "N")
  similar <- if (NROW(localTags) > 0) {
    localTags[tail(aa, as.numeric(showSimilar)), on = "artifact"][N == max(N)]
  } else {
    localTags
  }
  if (NROW(similar)) {
    similar2 <- similar[grepl("preDigest", tag)]
    cacheIdOfSimilar <- similar[grepl("cacheId", tag)]$tag
    cacheIdOfSimilar <- unlist(strsplit(cacheIdOfSimilar, split = ":"))[2]

    similar2[, `:=`(fun = unlist(lapply(strsplit(tag, split = ":"), function(xx) xx[[2]])),
                    hash = unlist(lapply(strsplit(tag, split = ":"), function(xx) xx[[3]])))]

    a <- setDT(as.list(preDigestUnlistTrunc))
    a <- melt(a, measure.vars = seq_along(names(a)), variable.name = "fun", value.name = "hash")


    similar2 <- similar2[a, on = "fun", nomatch = NA]
    similar2[, differs := (i.hash != hash)]

    similar2[!(fun %in% names(preDigestUnlistTrunc)), differs := NA]
    similar2[(hash %in% "other"), deeperThan3 := TRUE]
    similar2[(hash %in% "other"), differs := NA]
    differed <- FALSE
    if (isDevMode) {
      message(crayon::cyan(" ------ devMode -------"))
      message(crayon::cyan("This call to cache will replace"))
    } else {
      message(crayon::cyan(" ------ showSimilar -------"))
      message(crayon::cyan("This call to cache differs from the next closest:"))
    }
    message(crayon::cyan(paste0("... artifact with cacheId ", cacheIdOfSimilar)))

    if (sum(similar2[differs %in% TRUE]$differs, na.rm = TRUE)) {
      differed <- TRUE
      message(crayon::cyan("... different", paste(similar2[differs %in% TRUE]$fun, collapse = ", ")))
    }

    if (length(similar2[is.na(differs) & deeperThan3 == TRUE]$differs)) {
      differed <- TRUE
      message(crayon::cyan("... possible, unknown, differences in a nested list",
                           "that is deeper than",getOption("reproducible.showSimilarDepth",3),"in ",
              paste(collapse = ", ", as.character(similar2[deeperThan3 == TRUE]$fun))))
    }
    missingArgs <- similar2[is.na(deeperThan3) & is.na(differs)]$fun
    if (length(missingArgs)) {
      differed <- TRUE
      message(crayon::cyan("... because of (a) new argument(s): ",
              #"argument currently specified that was not in similar cache: ",
              paste(as.character(missingArgs), collapse = ", ")))

    }
    # message(crayon::cyan("Only the following elements differ (dots are replacements for $ or @)"))
    # oldColsToKeep <- c("fun", "differs")
    # cleanDT <- similar2[, ..oldColsToKeep]
    # data.table::setnames(cleanDT, old = oldColsToKeep, new = c("element", "different"))
    # message(crayon::cyan(
    #   paste0(capture.output(
    #     cleanDT)
    # , collapse = "\n")))
    if (isDevMode) {
      message(crayon::cyan(" ------ end devMode -------"))
    } else {
      message(crayon::cyan(" ------ end showSimilar -------"))
    }

  } else {
    if (!identical("devMode", useCache))
      message("There is no similar item in the cacheRepo")
  }
}

#' @keywords internal
getLocalTags <- function(cacheRepo) {
  written <- 0
  while (written >= 0) {
    localTags <- suppressWarnings(try(showLocalRepo2(cacheRepo), silent = TRUE))
    #localTags <- suppressWarnings(try(showLocalRepo(cacheRepo, "tags"), silent = TRUE))
    written <- if (is(localTags, "try-error")) {
      if (grepl("Error in checkDirectory", localTags)) {
        stop(localTags, "\nThis likely means the cacheRepo must be deleted")
      }
      Sys.sleep(sum(runif(written + 1,0.05, 0.2)))
      written + 1
    } else {
      -1
    }
  }
  localTags
}

#' @keywords internal
.defaultCacheOmitArgs <- c("useCloud", "checksumsFileID", "cloudFolderID",
                           "notOlderThan", ".objects", "outputObjects", "algo", "cacheRepo",
                           "length", "compareRasterFileLength", "userTags", "digestPathContent",
                           "omitArgs", "classOptions", "debugCache", "sideEffect", "makeCopy",
                           "quick", "verbose", "cacheId", "useCache", "showSimilar")

#' @keywords internal
verboseTime <- function(verbose) {
  if (verbose > 1) {
    return(Sys.time())
  }
}

#' @keywords internal
verboseMessage1 <- function(verbose, userTags) {
  if (verbose > 0)
    message("Using devMode; overwriting previous Cache entry with tags: ",
            paste(userTags, collapse = ", "))
  invisible(NULL)
}

#' @keywords internal
verboseMessage2 <- function(verbose) {
  if (verbose > 0)
    message("Using devMode; Found entry with identical userTags, ",
            "but since it is very different, adding new entry")
  invisible(NULL)
}

#' @keywords internal
verboseMessage3 <- function(verbose, artifact) {
  if (length(unique(artifact)) > 1) {
    if (verbose > 0)
      message("Using devMode, but userTags are not unique; defaulting to normal useCache = TRUE")
  }
}

#' @keywords internal
verboseDF1 <- function(verbose, functionName, startRunTime) {
  if (verbose > 1) {
    endRunTime <- Sys.time()
    verboseDF <- data.frame(
      functionName = functionName,
      component = paste("Running", functionName),
      elapsedTime = as.numeric(difftime(endRunTime, startRunTime, units = "secs")),
      units = "secs",
      stringsAsFactors = FALSE
    )

    if (exists("verboseTiming", envir = .reproEnv)) {
      .reproEnv$verboseTiming <- rbind(.reproEnv$verboseTiming, verboseDF)
    }
  }
}

#' @keywords internal
verboseDF2 <- function(verbose, functionName, startSaveTime) {
  if (verbose > 1) {
    endSaveTime <- Sys.time()
    verboseDF <-
      data.frame(
        functionName = functionName,
        component = "Saving to repo",
        elapsedTime = as.numeric(difftime(endSaveTime, startSaveTime, units = "secs")),
        units = "secs",
        stringsAsFactors = FALSE
      )

    if (exists("verboseTiming", envir = .reproEnv)) {
      .reproEnv$verboseTiming <- rbind(.reproEnv$verboseTiming, verboseDF)
    }
  }
}

#' @keywords internal
verboseDF3 <- function(verbose, functionName, startCacheTime) {
  if (verbose > 1) {
    endCacheTime <- Sys.time()
    verboseDF <- data.frame(functionName = functionName,
                            component = "Whole Cache call",
                            elapsedTime = as.numeric(difftime(endCacheTime, startCacheTime,
                                                              units = "secs")),
                            units = "secs",
                            stringsAsFactors = FALSE)

    if (exists("verboseTiming", envir = .reproEnv)) {
      .reproEnv$verboseTiming <- rbind(.reproEnv$verboseTiming, verboseDF)
    }
  }
}

#' @keywords internal
determineNestedTags <- function(envir, mc, userTags) {
  argsNoNesting <- "useCloud"
  # if (R.version[['minor']] <= "4.0") {
  #   # match.call changed how it worked between 3.3.2 and 3.4.x MUCH SLOWER
  #   lsCurEnv <- ls(all.names = TRUE, envir = envir)
  #   objs <- lsCurEnv[lsCurEnv %in% .namesCacheFormals]
  #   objs <- objs[match(.namesCacheFormals, objs)]# sort so same order as R > 3.4
  #   args <- mget(objs, envir = envir)
  #   forms <- lapply(.formalsCache, function(x) eval(x))
  #   objOverride <- unlist(lapply(objs, function(obj) identical(args[[obj]], forms[[obj]])))
  #   userCacheArgs <- objs[!objOverride]
  #   namesUserCacheArgs <- userCacheArgs
  # } else {
    mc <- as.list(mc[-1])
    namesMatchCall <- names(mc)
    namesMatchCall <- namesMatchCall[!namesMatchCall %in% argsNoNesting]
    userCacheArgs <- match(.namesCacheFormals, namesMatchCall)
    namesUserCacheArgs <- namesMatchCall[na.omit(userCacheArgs)]
    objOverride <- is.na(userCacheArgs)
  #}

  oldUserTags <- NULL
  prevUserTags <- FALSE
  prevValsInitial <- NULL
  if (any(!objOverride)) { # put into .reproEnv
    lsDotReproEnv <- ls(.reproEnv)
    namesMatchCallUserCacheArgs <- namesUserCacheArgs
    prevVals <- namesMatchCallUserCacheArgs %in% lsDotReproEnv

    # userTags is special because it gets appended
    prevUserTags <- if ("userTags" %in% namesMatchCallUserCacheArgs &&
                        "userTags" %in% lsDotReproEnv) {
      TRUE
    } else {
      FALSE
    }

    if (prevUserTags) {
      oldUserTags <- .reproEnv$userTags
      userTags <- c(userTags, .reproEnv$userTags)
      list2env(list(userTags = userTags), .reproEnv)
      # on.exit({
      #   .reproEnv$userTags <- oldUserTags
      # }, add = TRUE)
    }

    if (any(!prevVals)) {
      # don't override previous values -- except for userTags
      list2env(mget(namesUserCacheArgs[!prevVals], envir = envir), .reproEnv)
    }
    prevValsInitial <- prevVals
  }


  if (any(objOverride)) {
    # get from .reproEnv
    lsDotReproEnv <- ls(.reproEnv)
    prevVals <- .namesCacheFormals[objOverride] %in% lsDotReproEnv
    if (any(prevVals)) {
      list2env(mget(.namesCacheFormals[objOverride][prevVals], .reproEnv), envir = envir)
    }
  }

  return(list(oldUserTags = oldUserTags, namesUserCacheArgs = namesUserCacheArgs,
              prevVals = prevValsInitial, prevUserTags = prevUserTags,
              objOverride = objOverride))
}

getCacheRepos <- function(cacheRepo, modifiedDots) {
  if (is.null(cacheRepo)) {
    cacheRepos <- .checkCacheRepo(modifiedDots, create = TRUE)
  } else {
    cacheRepos <- lapply(cacheRepo, function(repo) {
      repo <- checkPath(repo, create = TRUE)
    })
  }
  return(cacheRepos)
}

devModeFn1 <- function(localTags, userTags, scalls, preDigestUnlistTrunc, useCache, verbose,
                       isInRepo, outputHash) {
  isInRepoAlt <- localTags[localTags$tag %in% userTags, , drop = FALSE]
  data.table::setDT(isInRepoAlt)
  isInRepoAlt <- isInRepoAlt[, iden := identical(sum(tag %in% userTags), length(userTags)),
                             by = "artifact"][iden == TRUE]
  if (NROW(isInRepoAlt) > 0 && length(unique(isInRepoAlt$artifact)) == 1) {
    newLocalTags <- localTags[localTags$artifact %in% isInRepoAlt$artifact,]
    tags1 <- grepl(paste0("(",
                          paste("accessed", "cacheId", "class", "date", "format", "function",
                                "name", "object.size", "otherFunctions", "preDigest",
                                sep = "|"),
                          ")"),
                   newLocalTags$tag)
    localTagsAlt <- newLocalTags[!tags1,]
    if (all(localTagsAlt$tag %in% userTags)) {
      mess <- capture.output(type = "output", {
        similars <- .findSimilar(newLocalTags, scalls = scalls,
                                 preDigestUnlistTrunc = preDigestUnlistTrunc,
                                 userTags = userTags,
                                 useCache = useCache)
      })
      similarsHaveNA <- sum(is.na(similars$differs))
      #similarsAreDifferent <- sum(similars$differs == TRUE, na.rm = TRUE)
      #likelyNotSame <- sum(similarsHaveNA, similarsAreDifferent)/NROW(similars)

      if (similarsHaveNA < 2) {
        verboseMessage1(verbose, userTags)
        outputHash <- gsub("cacheId:", "", newLocalTags[newLocalTags$artifact %in% isInRepoAlt$artifact & #nolint
                                                          startsWith(newLocalTags$tag, "cacheId"), ]$tag) #nolint
        isInRepo <- isInRepoAlt
      } else {
        verboseMessage2(verbose)
      }
    }
    needFindByTags <- TRUE # it isn't there
  } else {
    verboseMessage3(verbose, isInRepoAlt$artifact)
    needFindByTags <- FALSE # it isn't there
  }
  return(list(isInRepo = isInRepo, outputHash = outputHash, needFindByTags = needFindByTags))
}
