% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rposterior_rcpp.R
\name{rpost_rcpp}
\alias{rpost_rcpp}
\title{Random sampling from extreme value posterior distributions}
\usage{
rpost_rcpp(
  n,
  model = c("gev", "gp", "bingp", "pp", "os"),
  data,
  prior,
  ...,
  nrep = NULL,
  thresh = NULL,
  noy = NULL,
  use_noy = TRUE,
  npy = NULL,
  ros = NULL,
  bin_prior = structure(list(prior = "bin_beta", ab = c(1/2, 1/2), class = "binprior")),
  init_ests = NULL,
  mult = 2,
  use_phi_map = FALSE
)
}
\arguments{
\item{n}{A numeric scalar. The size of posterior sample required.}

\item{model}{A character string.  Specifies the extreme value model.}

\item{data}{Sample data, of a format appropriate to the value of
\code{model}.
\itemize{
  \item \code{"gp"}. A numeric vector of threshold excesses or raw data.
  \item \code{"bingp"}. A numeric vector of raw data.
  \item \code{"gev"}. A numeric vector of block maxima.
  \item \code{"pp"}. A numeric vector of raw data.
  \item \code{"os"}. A numeric matrix or data frame. Each row should contain
    the largest order statistics for a block of data.  These need not
    be ordered: they are sorted inside \code{rpost}. If a block
    contains fewer than \code{dim(as.matrix(data))[2]} order statistics
    then the corresponding row should be padded by \code{NA}s. If
    \code{ros} is supplied then only the largest \code{ros} values in
    each row are used.  If a vector is supplied then this is converted
    to a matrix with one column.  This is equivalent to using
    \code{model = "gev"}.
}}

\item{prior}{A list specifying the prior for the parameters of the extreme
value model, created by \code{\link{set_prior}}.}

\item{...}{Further arguments to be passed to \code{\link[rust]{ru_rcpp}}.
Most importantly \code{trans} and \code{rotate} (see \strong{Details}),
and perhaps \code{r}, \code{ep}, \code{a_algor}, \code{b_algor},
\code{a_method}, \code{b_method}, \code{a_control}, \code{b_control}.
May also be used to pass the arguments \code{n_grid} and/or \code{ep_bc}
to \code{\link[rust]{find_lambda}}.}

\item{nrep}{A numeric scalar.  If \code{nrep} is not \code{NULL} then
\code{nrep} gives the number of replications of the original dataset
simulated from the posterior predictive distribution.
Each replication is based on one of the samples from the posterior
distribution.  Therefore, \code{nrep} must not be greater than \code{n}.
In that event \code{nrep} is set equal to \code{n}.
Currently only implemented if \code{model = "gev"} or \code{"gp"} or
\code{"bingp"} or \code{"pp"}, i.e. \emph{not} implemented if
\code{model = "os"}.}

\item{thresh}{A numeric scalar.  Extreme value threshold applied to data.
Only relevant when \code{model = "gp"}, \code{"pp"} or \code{"bingp"}.
Must be supplied when \code{model = "pp"} or \code{"bingp"}.
If \code{model = "gp"} and \code{thresh} is not supplied then
\code{thresh = 0} is used and \code{data} should contain threshold
excesses.}

\item{noy}{A numeric scalar. The number of blocks of observations,
excluding any missing values.  A block is often a year.
Only relevant, and must be supplied, if \code{model = "pp"}.}

\item{use_noy}{A logical scalar.  Only relevant if model is "pp".
If \code{use_noy = FALSE} then sampling is based on a likelihood in
which the number of blocks (years) is set equal to the number of threshold
excesses, to reduce posterior dependence between the parameters
(Wadsworth \emph{et al}., 2010).
The sampled values are transformed back to the required parameterisation
before returning them to the user.  If \code{use_noy = TRUE} then the
user's value of \code{noy} is used in the likelihood.}

\item{npy}{A numeric scalar. The mean number of observations per year
  of data, after excluding any missing values, i.e. the number of
  non-missing observations divided by total number of years' worth of
  non-missing data.

  The value of \code{npy} does not affect any calculation in
  \code{rpost}, it only affects subsequent extreme value inferences using
  \code{predict.evpost}.  However, setting \code{npy} in the call to
  \code{rpost} avoids the need to supply \code{npy} when calling
  \code{predict.evpost}.  This is likely to be useful only when
  \code{model = bingp}. See the documentation of
  \code{\link{predict.evpost}} for further details.}

\item{ros}{A numeric scalar.  Only relevant when \code{model = "os"}. The
largest \code{ros} values in each row of the matrix \code{data} are used
in the analysis.}

\item{bin_prior}{A list specifying the prior for a binomial probability
\eqn{p}, created by \code{\link{set_bin_prior}}.  Only relevant if
\code{model = "bingp"}.  If this is not supplied then the Jeffreys
beta(1/2, 1/2) prior is used.}

\item{init_ests}{A numeric vector.  Initial parameter estimates for search
for the mode of the posterior distribution.}

\item{mult}{A numeric scalar.  The grid of values used to choose the Box-Cox
transformation parameter lambda is based on the maximum a posteriori (MAP)
estimate +/- mult x estimated posterior standard deviation.}

\item{use_phi_map}{A logical scalar. If trans = "BC" then \code{use_phi_map}
determines whether the grid of values for phi used to set lambda is
centred on the maximum a posterior (MAP) estimate of phi
(\code{use_phi_map = TRUE}), or on the initial estimate of phi
(\code{use_phi_map = FALSE}).}
}
\value{
An object (list) of class \code{"evpost"}, which has the same
  structure as an object of class \code{"ru"} returned from
  \code{\link[rust]{ru_rcpp}}.  In addition this list contains

    \item{\code{model}:}{The argument \code{model} to \code{rpost}
      detailed above.}
    \item{\code{data}:}{The content depends on \code{model}:
      if \code{model = "gev"} then this is the argument \code{data} to
      \code{rpost} detailed above, with missing values removed;
      if \code{model = "gp"} then only the values that lie above the
      threshold are included; if \code{model = "bingp"} or
      \code{model = "pp"} then the input data are returned
      but any value lying below the threshold is set to \code{thresh};
      if \code{model = "os"} then the order statistics used are returned
      as a single vector.}
    \item{\code{prior}:}{The argument \code{prior} to \code{rpost}
      detailed above.}
    \item{\code{logf_rho_args}:}{A list of arguments to the (transformed)
      target log-density.}

  If \code{nrep} is not \code{NULL} then this list also contains
  \code{data_rep}, a numerical matrix with \code{nrep} rows.  Each
  row contains a replication of the original data \code{data}
  simulated from the posterior predictive distribution.
  If \code{model = "bingp"} or \code{"pp"} then the rate of threshold
  exceedance is part of the inference.  Therefore, the number of values in
  \code{data_rep} that lie above the threshold varies between
  predictive replications (different rows of \code{data_rep}).
  Values below the threshold are left-censored at the threshold, i.e. they
  are set at the threshold.

  If \code{model == "pp"} then this list also contains the argument
    \code{noy} to \code{rpost} detailed above.
  If the argument \code{npy} was supplied then this list also contains
  \code{npy}.

  If \code{model == "gp"} or \code{model == "bingp"} then this list also
    contains the argument \code{thresh} to \code{rpost} detailed above.

  If \code{model == "bingp"} then this list also contains

    \item{\code{bin_sim_vals}:}{An \code{n} by 1 numeric matrix of values
      simulated from the posterior for the binomial probability \eqn{p}}
    \item{\code{bin_logf}:}{A function returning the log-posterior for
      \eqn{p}.}
    \item{\code{bin_logf_args}:}{A list of arguments to \code{bin_logf}.}
}
\description{
Uses the \code{\link[rust]{ru_rcpp}} function in the
\code{\link[rust]{rust}} package to simulate from the posterior distribution
of an extreme value model.
}
\details{
\emph{Generalised Pareto (GP)}: \code{model = "gp"}.  A model for threshold
  excesses.  Required arguments: \code{n}, \code{data} and \code{prior}.
  If \code{thresh} is supplied then only the values in \code{data} that
  exceed \code{thresh} are used and the GP distribution is fitted to the
  amounts by which those values exceed \code{thresh}.
  If \code{thresh} is not supplied then the GP distribution is fitted to
  all values in \code{data}, in effect \code{thresh = 0}.
  See also \code{\link{gp}}.

\emph{Binomial-GP}: \code{model = "bingp"}.  The GP model for threshold
  excesses supplemented by a binomial(\code{length(data)}, \eqn{p})
  model for the number of threshold excesses.  See Northrop et al. (2017)
  for details.  Currently, the GP and binomial parameters are assumed to
  be independent \emph{a priori}.

\emph{Generalised extreme value (GEV) model}: \code{model = "gev"}.  A
  model for block maxima.  Required arguments: \code{n}, \code{data},
  \code{prior}.  See also \code{\link{gev}}.

\emph{Point process (PP) model}: \code{model = "pp"}. A model for
  occurrences of threshold exceedances and threshold excesses.  Required
  arguments: \code{n}, \code{data}, \code{prior}, \code{thresh} and
  \code{noy}.

\emph{r-largest order statistics (OS) model}: \code{model = "os"}.
  A model for the largest order statistics within blocks of data.
  Required arguments: \code{n}, \code{data}, \code{prior}.  All the values
  in \code{data} are used unless \code{ros} is supplied.

\emph{Parameter transformation}.  The scalar logical arguments (to the
  function \code{\link{ru}}) \code{trans} and \code{rotate} determine,
  respectively, whether or not Box-Cox transformation is used to reduce
  asymmetry in the posterior distribution and rotation of parameter
  axes is used to reduce posterior parameter dependence.  The default
  is \code{trans = "none"} and \code{rotate = TRUE}.

  See the \href{https://CRAN.R-project.org/package=revdbayes}{Introducing revdbayes vignette}
  for further details and examples.
}
\examples{
# GP model
u <- quantile(gom, probs = 0.65)
fp <- set_prior(prior = "flat", model = "gp", min_xi = -1)
gpg <- rpost_rcpp(n = 1000, model = "gp", prior = fp, thresh = u,
                  data = gom)
plot(gpg)

# GP model, user-defined prior (same prior as the previous example)
ptr_gp_flat <- create_prior_xptr("gp_flat")
p_user <- set_prior(prior = ptr_gp_flat, model = "gp", min_xi = -1)
gpg <- rpost_rcpp(n = 1000, model = "gp", prior = p_user, thresh = u,
                  data = gom)
plot(gpg)

# Binomial-GP model
u <- quantile(gom, probs = 0.65)
fp <- set_prior(prior = "flat", model = "gp", min_xi = -1)
bp <- set_bin_prior(prior = "jeffreys")
bgpg <- rpost_rcpp(n = 1000, model = "bingp", prior = fp, thresh = u,
                   data = gom, bin_prior = bp)
plot(bgpg, pu_only = TRUE)
plot(bgpg, add_pu = TRUE)

# Setting the same binomial (Jeffreys) prior by hand
beta_prior_fn <- function(p, ab) {
  return(stats::dbeta(p, shape1 = ab[1], shape2 = ab[2], log = TRUE))
}
jeffreys <- set_bin_prior(beta_prior_fn, ab = c(1 / 2, 1 / 2))
bgpg <- rpost_rcpp(n = 1000, model = "bingp", prior = fp, thresh = u,
                   data = gom, bin_prior = jeffreys)
plot(bgpg, pu_only = TRUE)
plot(bgpg, add_pu = TRUE)

# GEV model
mat <- diag(c(10000, 10000, 100))
pn <- set_prior(prior = "norm", model = "gev", mean = c(0, 0, 0), cov = mat)
gevp  <- rpost_rcpp(n = 1000, model = "gev", prior = pn, data = portpirie)
plot(gevp)

# GEV model, user-defined prior (same prior as the previous example)
mat <- diag(c(10000, 10000, 100))
ptr_gev_norm <- create_prior_xptr("gev_norm")
pn_u <- set_prior(prior = ptr_gev_norm, model = "gev", mean = c(0, 0, 0),
                  icov = solve(mat))
gevu <- rpost_rcpp(n = 1000, model = "gev", prior = pn_u, data = portpirie)
plot(gevu)

# GEV model, informative prior constructed on the probability scale
pip  <- set_prior(quant = c(85, 88, 95), alpha = c(4, 2.5, 2.25, 0.25),
  model = "gev", prior = "prob")
ox_post <- rpost_rcpp(n = 1000, model = "gev", prior = pip, data = oxford)
plot(ox_post)

# PP model
pf <- set_prior(prior = "flat", model = "gev", min_xi = -1)
ppr <- rpost_rcpp(n = 1000, model = "pp", prior = pf, data = rainfall,
                  thresh = 40, noy = 54)
plot(ppr)

# PP model, user-defined prior (same prior as the previous example)
ptr_gev_flat <- create_prior_xptr("gev_flat")
pf_u <- set_prior(prior = ptr_gev_flat, model = "gev", min_xi = -1,
                  max_xi = Inf)
ppru <- rpost_rcpp(n = 1000, model = "pp", prior = pf_u, data = rainfall,
                   thresh = 40, noy = 54)
plot(ppru)

# PP model, informative prior constructed on the quantile scale
piq <- set_prior(prob = 10^-(1:3), shape = c(38.9, 7.1, 47),
                 scale = c(1.5, 6.3, 2.6), model = "gev", prior = "quant")
rn_post <- rpost_rcpp(n = 1000, model = "pp", prior = piq, data = rainfall,
                      thresh = 40, noy = 54)
plot(rn_post)

# OS model
mat <- diag(c(10000, 10000, 100))
pv <- set_prior(prior = "norm", model = "gev", mean = c(0, 0, 0), cov = mat)
osv <- rpost_rcpp(n = 1000, model = "os", prior = pv, data = venice)
plot(osv)
}
\references{
Coles, S. G. and Powell, E. A. (1996) Bayesian methods in
  extreme value modelling: a review and new developments.
  \emph{Int. Statist. Rev.}, \strong{64}, 119-136.

Northrop, P. J., Attalides, N. and Jonathan, P. (2017)
  Cross-validatory extreme value threshold selection and uncertainty
  with application to ocean storm severity.
  \emph{Journal of the Royal Statistical Society Series C: Applied
  Statistics}, \strong{66}(1), 93-120.
  \doi{10.1111/rssc.12159}

Stephenson, A. (2016) Bayesian Inference for Extreme Value
  Modelling. In \emph{Extreme Value Modeling and Risk Analysis: Methods and
  Applications}, edited by D. K. Dey and J. Yan, 257-80. London:
  Chapman and Hall. \doi{10.1201/b19721}
  value posterior using the evdbayes package.

Wadsworth, J. L., Tawn, J. A. and Jonathan, P. (2010)
  Accounting for choice of measurement scale in extreme value modeling.
 \emph{The Annals of Applied Statistics}, \strong{4}(3), 1558-1578.
  \doi{10.1214/10-AOAS333}
}
\seealso{
\code{\link{set_prior}} for setting a prior distribution.

\code{\link{rpost}} for posterior simulation without using
  the Rcpp package.

\code{\link{plot.evpost}}, \code{\link{summary.evpost}} and
  \code{\link{predict.evpost}} for the S3 \code{plot}, \code{summary}
  and \code{predict} methods for objects of class \code{evpost}.

\code{\link[rust]{ru_rcpp}} in the \code{\link[rust]{rust}}
  package for details of the arguments that can be passed to
  \code{ru_rcpp} and the form of the object returned by \code{rpost_rcpp}.

\code{\link[rust]{find_lambda}} in the
  \code{\link[rust]{rust}} package is used inside \code{rpost} to set the
  Box-Cox transformation parameter lambda when the \code{trans = "BC"}
  argument is given.
}
