% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/occ_data.R
\name{occ_data}
\alias{occ_data}
\title{Search for GBIF occurrences - simplified for speed}
\usage{
occ_data(taxonKey = NULL, scientificName = NULL, country = NULL,
  publishingCountry = NULL, hasCoordinate = NULL, typeStatus = NULL,
  recordNumber = NULL, lastInterpreted = NULL, continent = NULL,
  geometry = NULL, geom_big = "asis", geom_size = 40, geom_n = 10,
  recordedBy = NULL, basisOfRecord = NULL, datasetKey = NULL,
  eventDate = NULL, catalogNumber = NULL, year = NULL, month = NULL,
  decimalLatitude = NULL, decimalLongitude = NULL, elevation = NULL,
  depth = NULL, institutionCode = NULL, collectionCode = NULL,
  hasGeospatialIssue = NULL, issue = NULL, search = NULL,
  mediaType = NULL, limit = 500, start = 0, ...)
}
\arguments{
\item{taxonKey}{A taxon key from the GBIF backbone. All included and synonym taxa
are included in the search, so a search for aves with taxononKey=212
(i.e. /occurrence/search?taxonKey=212) will match all birds, no matter which
species. You can pass many keys by passing occ_search in a call to an
lapply-family function (see last example below).}

\item{scientificName}{A scientific name from the GBIF backbone. All included and synonym
taxa are included in the search.}

\item{country}{The 2-letter country code (as per ISO-3166-1) of the country in
which the occurrence was recorded. See here
\url{http://en.wikipedia.org/wiki/ISO_3166-1_alpha-2}}

\item{publishingCountry}{The 2-letter country code (as per ISO-3166-1) of the
country in which the occurrence was recorded.}

\item{hasCoordinate}{(logical) Return only occurence records with lat/long data (TRUE) or
all records (FALSE, default).}

\item{typeStatus}{Type status of the specimen. One of many options. See ?typestatus}

\item{recordNumber}{Number recorded by collector of the data, different from GBIF record
number. See \url{http://rs.tdwg.org/dwc/terms/#recordNumber} for more info}

\item{lastInterpreted}{Date the record was last modified in GBIF, in ISO 8601 format:
yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd.  Supports range queries, smaller,larger (e.g.,
'1990,1991', whereas '1991,1990' wouldn't work)}

\item{continent}{Continent. One of africa, antarctica, asia, europe, north_america
(North America includes the Caribbean and reachies down and includes Panama), oceania,
or south_america}

\item{geometry}{Searches for occurrences inside a polygon described in Well Known
Text (WKT) format. A WKT shape written as either POINT, LINESTRING, LINEARRING
or POLYGON. Example of a polygon: ((30.1 10.1, 20, 20 40, 40 40, 30.1 10.1))
 would be queried as \url{http://bit.ly/1BzNwDq}. See also the section \strong{WKT} below.}

\item{geom_big}{(character) One of "axe", "bbox", or "asis" (default). See Details.}

\item{geom_size}{(integer) An integer indicating size of the cell. Default: 40. See Details.}

\item{geom_n}{(integer) An integer indicating number of cells in each dimension. Default: 10.
See Details.}

\item{recordedBy}{The person who recorded the occurrence.}

\item{basisOfRecord}{Basis of record, as defined in our BasisOfRecord enum here
\url{http://gbif.github.io/gbif-api/apidocs/org/gbif/api/vocabulary/BasisOfRecord.html}
Acceptable values are:
\itemize{
  \item FOSSIL_SPECIMEN An occurrence record describing a fossilized specimen.
  \item HUMAN_OBSERVATION An occurrence record describing an observation made by
  one or more people.
  \item LITERATURE An occurrence record based on literature alone.
  \item LIVING_SPECIMEN An occurrence record describing a living specimen, e.g.
  \item MACHINE_OBSERVATION An occurrence record describing an observation made
  by a machine.
  \item OBSERVATION An occurrence record describing an observation.
  \item PRESERVED_SPECIMEN An occurrence record describing a preserved specimen.
  \item UNKNOWN Unknown basis for the record.
}}

\item{datasetKey}{The occurrence dataset key (a uuid)}

\item{eventDate}{Occurrence date in ISO 8601 format: yyyy, yyyy-MM, yyyy-MM-dd, or
MM-dd. Supports range queries, smaller,larger (e.g., '1990,1991', whereas '1991,1990'
wouldn't work)}

\item{catalogNumber}{An identifier of any form assigned by the source within a
physical collection or digital dataset for the record which may not unique,
but should be fairly unique in combination with the institution and collection code.}

\item{year}{The 4 digit year. A year of 98 will be interpreted as AD 98. Supports range queries,
smaller,larger (e.g., '1990,1991', whereas '1991,1990' wouldn't work)}

\item{month}{The month of the year, starting with 1 for January. Supports range queries,
smaller,larger (e.g., '1,2', whereas '2,1' wouldn't work)}

\item{decimalLatitude}{Latitude in decimals between -90 and 90 based on WGS 84.
Supports range queries, smaller,larger (e.g., '25,30', whereas '30,25' wouldn't work)}

\item{decimalLongitude}{Longitude in decimals between -180 and 180 based on WGS 84.
Supports range queries (e.g., '-0.4,-0.2', whereas '-0.2,-0.4' wouldn't work).}

\item{elevation}{Elevation in meters above sea level. Supports range queries, smaller,larger
(e.g., '5,30', whereas '30,5' wouldn't work)}

\item{depth}{Depth in meters relative to elevation. For example 10 meters below a
lake surface with given elevation. Supports range queries, smaller,larger (e.g., '5,30',
whereas '30,5' wouldn't work)}

\item{institutionCode}{An identifier of any form assigned by the source to identify
the institution the record belongs to. Not guaranteed to be que.}

\item{collectionCode}{An identifier of any form assigned by the source to identify
the physical collection or digital dataset uniquely within the text of an institution.}

\item{hasGeospatialIssue}{(logical) Includes/excludes occurrence records which contain spatial
issues (as determined in our record interpretation), i.e. \code{hasGeospatialIssue=TRUE}
returns only those records with spatial issues while \code{hasGeospatialIssue=FALSE} includes
only records without spatial issues. The absence of this parameter returns any
record with or without spatial issues.}

\item{issue}{(character) One or more of many possible issues with each occurrence record. See
Details. Issues passed to this parameter filter results by the issue.}

\item{search}{Query terms. The value for this parameter can be a simple word or a phrase.}

\item{mediaType}{Media type. Default is NULL, so no filtering on mediatype. Options:
NULL, 'MovingImage', 'Sound', and 'StillImage'.}

\item{limit}{Number of records to return. Default: 500. Note that the per request maximum is
300, but since we set it at 500 for the function, we do two requests to get you the 500
records (if there are that many). Note that there is a hard maximum of 200,000, which is
calculated as the \code{limit+start}, so \code{start=199,000} and \code{limit=2000} won't work}

\item{start}{Record number to start at. Use in combination with limit to page through results.
Note that we do the paging internally for you, but you can manually set the \code{start}
parameter}

\item{...}{Further named parameters, such as \code{query}, \code{path}, etc, passed on to
\code{\link[httr]{modify_url}} within \code{\link[httr]{GET}} call. Unnamed parameters will
be combined with \code{\link[httr]{config}}.}
}
\value{
A data.frame or list
}
\description{
Search for GBIF occurrences - simplified for speed
}
\details{
\bold{Multiple parmeters}: Note that you can pass in a vector to one of taxonKey,
scientificName, datasetKey, catalogNumber, recordedBy, geometry, country, publishingCountry,
recordNumber, search, institutionCode, collectionCode, decimalLatitude, decimalLongitude,
depth, year, typeStatus, lastInterpreted, continent, or mediatype parameters in a
function call, but not a vector >1 of these parameters at the same time

\bold{Hierarchies:} hierarchies are returned wih each occurrence object. There is no
option no to return them from the API. However, within the \code{occ_search}
function you can select whether to return just hierarchies, just data, all of
data and hiearchies and metadata, or just metadata. If all hierarchies are the
same we just return one for you.

\bold{Data:} By default only three data fields are returned: name (the species name),
decimallatitude, and decimallongitude. Set parameter minimal=FALSE if you want more data.

\bold{Nerds:} You can pass parameters not defined in this function into the call to
the GBIF API to control things about the call itself using \code{...}. See an example below
that passes in the \code{verbose} function to get details on the http call.

\bold{Scientific names vs. taxon keys:} In the previous GBIF API and the version of rgbif that
wrapped that API, you could search the equivalent of this function with a species name, which
was convenient. However, names are messy right. So it sorta makes sense to sort out the species
key numbers you want exactly, and then get your occurrence data with this function. GBIF has
added a parameter scientificName to allow searches by scientific names in this function - which
includes synonym taxa. \emph{Note:} that if you do use the scientificName parameter, we will
check internally that it's not a synonym of an accepted name, and if it is, we'll search on the
accepted name. If you want to force searching by a synonym do so by finding the GBIF identifier
first with any \code{name_*} functions, then pass that ID to the \code{taxonKey} parameter.

\bold{WKT:} Examples of valid WKT objects:
\itemize{
 \item 'POLYGON((30.1 10.1, 10 20, 20 60, 60 60, 30.1 10.1))'
 \item 'POINT(-120 40)'
 \item 'LINESTRING(3 4,10 50,20 25)'
 \item 'LINEARRING' ???' - Not sure how to specify this. Anyone?
}

\bold{Long WKT:} Options for handling long WKT strings:
Note that long WKT strings are specially handled when using \code{occ_search} or
\code{occ_data}. Here are the three options for long WKT strings (> 1500 characters),
set one of these three via the parameter \code{geom_big}:
\itemize{
 \item asis - the default setting. This means we don't do anything internally. That is,
 we just pass on your WKT string just as we've done before in this package.
 \item axe - this option uses the \pkg{geoaxe} package to chop up your WKT string in
 to many polygons, which then leads to a separate data request for each polygon piece,
 then we combine all dat back together to give to you. Note that if your WKT string
 is not of type polygon, we drop back to \code{asis} as there's no way to chop up
 linestrings, etc. This option will in most cases be slower than the other two options.
 However, this polygon splitting approach won't have the problem of
 the disconnect between how many records you want and what you actually get back as
 with the bbox option.

 This method uses \code{\link[geoaxe]{chop}}, which uses \code{GridTopology} from
 the \code{sp} package, which has two parameters \code{cellsize} and \code{cells.dim}
 that we use to chop up polygons. You can tweak those parameters here by tweaking
 \code{geom_size} and \code{geom_n}. \code{geom_size} seems to be more useful in
 toggling the number of WKT strings you get back.

 See \code{\link{wkt_parse}} to manually break make WKT bounding box from a larger WKT
 string, or break a larger WKT string into many smaller ones.

 \item bbox - this option checks whether your WKT string is longer than 1500 characters,
 and if it is we create a bounding box from the WKT, do the GBIF search with that
 bounding box, then prune the resulting data to only those occurrences in your original
 WKT string. There is a big caveat however. Because we create a bounding box from the WKT,
 and the \code{limit} parameter determines some subset of records to get, then when we
 prune the resulting data to the WKT, the number of records you get could be less than
 what you set with your \code{limit} parameter. However, you could set the limit to be
 high enough so that you get all records back found in that bounding box, then you'll
 get all the records available within the WKT.
}

\bold{Range queries:} A range query is as it sounds - you query on a range of values defined by
a lower and upper limit. Do a range query by specifying the lower and upper limit in a vector
like \code{depth='50,100'}. It would be more R like to specify the range in a vector like
\code{c(50,100)}, but that sort of syntax allows you to do many searches, one for each element in
the vector - thus range queries have to differ. The following parameters support range queries.
\itemize{
 \item decimalLatitude
 \item decimalLongitude
 \item depth
 \item elevation
 \item eventDate
 \item lastInterpreted
 \item month
 \item year
}

\bold{Issue:} The options for the issue parameter (from
http://gbif.github.io/gbif-api/apidocs/org/gbif/api/vocabulary/OccurrenceIssue.html):
\itemize{
 \item BASIS_OF_RECORD_INVALID The given basis of record is impossible to interpret or seriously
 different from the recommended vocabulary.
 \item CONTINENT_COUNTRY_MISMATCH The interpreted continent and country do not match up.
 \item CONTINENT_DERIVED_FROM_COORDINATES The interpreted continent is based on the coordinates,
 not the verbatim string information.
 \item CONTINENT_INVALID Uninterpretable continent values found.
 \item COORDINATE_INVALID Coordinate value given in some form but GBIF is unable to interpret it.
 \item COORDINATE_OUT_OF_RANGE Coordinate has invalid lat/lon values out of their decimal max
 range.
 \item COORDINATE_REPROJECTED The original coordinate was successfully reprojected from a
 different geodetic datum to WGS84.
 \item COORDINATE_REPROJECTION_FAILED The given decimal latitude and longitude could not be
 reprojected to WGS84 based on the provided datum.
 \item COORDINATE_REPROJECTION_SUSPICIOUS Indicates successful coordinate reprojection according
 to provided datum, but which results in a datum shift larger than 0.1 decimal degrees.
 \item COORDINATE_ROUNDED Original coordinate modified by rounding to 5 decimals.
 \item COUNTRY_COORDINATE_MISMATCH The interpreted occurrence coordinates fall outside of the
 indicated country.
 \item COUNTRY_DERIVED_FROM_COORDINATES The interpreted country is based on the coordinates, not
 the verbatim string information.
 \item COUNTRY_INVALID Uninterpretable country values found.
 \item COUNTRY_MISMATCH Interpreted country for dwc:country and dwc:countryCode contradict each
 other.
 \item DEPTH_MIN_MAX_SWAPPED Set if supplied min>max
 \item DEPTH_NON_NUMERIC Set if depth is a non numeric value
 \item DEPTH_NOT_METRIC Set if supplied depth is not given in the metric system, for example
 using feet instead of meters
 \item DEPTH_UNLIKELY Set if depth is larger than 11.000m or negative.
 \item ELEVATION_MIN_MAX_SWAPPED Set if supplied min > max elevation
 \item ELEVATION_NON_NUMERIC Set if elevation is a non numeric value
 \item ELEVATION_NOT_METRIC Set if supplied elevation is not given in the metric system, for
 example using feet instead of meters
 \item ELEVATION_UNLIKELY Set if elevation is above the troposphere (17km) or below 11km
 (Mariana Trench).
 \item GEODETIC_DATUM_ASSUMED_WGS84 Indicating that the interpreted coordinates assume they are
 based on WGS84 datum as the datum was either not indicated or interpretable.
 \item GEODETIC_DATUM_INVALID The geodetic datum given could not be interpreted.
 \item IDENTIFIED_DATE_INVALID The date given for dwc:dateIdentified is invalid and cant be
 interpreted at all.
 \item IDENTIFIED_DATE_UNLIKELY The date given for dwc:dateIdentified is in the future or before
 Linnean times (1700).
 \item MODIFIED_DATE_INVALID A (partial) invalid date is given for dc:modified, such as a non
 existing date, invalid zero month, etc.
 \item MODIFIED_DATE_UNLIKELY The date given for dc:modified is in the future or predates unix
 time (1970).
 \item MULTIMEDIA_DATE_INVALID An invalid date is given for dc:created of a multimedia object.
 \item MULTIMEDIA_URI_INVALID An invalid uri is given for a multimedia object.
 \item PRESUMED_NEGATED_LATITUDE Latitude appears to be negated, e.g. 32.3 instead of -32.3
 \item PRESUMED_NEGATED_LONGITUDE Longitude appears to be negated, e.g. 32.3 instead of -32.3
 \item PRESUMED_SWAPPED_COORDINATE Latitude and longitude appear to be swapped.
 \item RECORDED_DATE_INVALID A (partial) invalid date is given, such as a non existing date,
 invalid zero month, etc.
 \item RECORDED_DATE_MISMATCH The recording date specified as the eventDate string and the
 individual year, month, day are contradicting.
 \item RECORDED_DATE_UNLIKELY The recording date is highly unlikely, falling either into the
 future or represents a very old date before 1600 that predates modern taxonomy.
 \item REFERENCES_URI_INVALID An invalid uri is given for dc:references.
 \item TAXON_MATCH_FUZZY Matching to the taxonomic backbone can only be done using a fuzzy, non
 exact match.
 \item TAXON_MATCH_HIGHERRANK Matching to the taxonomic backbone can only be done on a higher
 rank and not the scientific name.
 \item TAXON_MATCH_NONE Matching to the taxonomic backbone cannot be done cause there was no
 match at all or several matches with too little information to keep them apart (homonyms).
 \item TYPE_STATUS_INVALID The given type status is impossible to interpret or seriously
 different from the recommended vocabulary.
 \item ZERO_COORDINATE Coordinate is the exact 0/0 coordinate, often indicating a bad null
 coordinate.
}

\bold{Counts:} There is a slight difference in the way records are counted here vs.
results from \code{\link{occ_count}}. For equivalent outcomes, in this function
use \code{hasCoordinate=TRUE}, and \code{hasGeospatialIssue=FALSE} to have the
same outcome using \code{\link{occ_count}} with \code{isGeoreferenced=TRUE}.

This does nearly the same thing as \code{\link{occ_search}}, but is a bit simplified
for speed, and is for the most common use case where user just wants the data, and not
other information like taxon hierarchies and media (e.g., images) information. Alot of time
in \code{\link{occ_search}} is used parsing data to be more useable downstream. We do
less of that in this function.
}
\examples{
\dontrun{
(key <- name_backbone(name='Encelia californica')$speciesKey)
occ_data(taxonKey = key, limit = 4)
(res <- occ_data(taxonKey = key, limit = 400))

# Return 20 results, this is the default by the way
(key <- name_suggest(q='Helianthus annuus', rank='species')$key[1])
occ_data(taxonKey=key, limit=20)

# Instead of getting a taxon key first, you can search for a name directly
## However, note that using this approach (with \\code{scientificName="..."})
## you are getting synonyms too. The results for using \\code{scientifcName} and
## \\code{taxonKey} parameters are the same in this case, but I wouldn't be surprised if for some
## names they return different results
library("httr")
occ_data(scientificName = 'Ursus americanus', config=verbose())
key <- name_backbone(name = 'Ursus americanus', rank='species')$usageKey
occ_data(taxonKey = key)

# Search by dataset key
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a', limit=10)

# Search by catalog number
occ_data(catalogNumber="49366", limit=10)
occ_data(catalogNumber=c("49366","Bird.27847588"), limit=10)

# Use paging parameters (limit and start) to page. Note the different results
# for the two queries below.
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=10,limit=5)
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=20,limit=5)

# Many dataset keys
occ_data(datasetKey=c("50c9509d-22c7-4a22-a47d-8c48425ef4a7",
   "7b5d6a48-f762-11e1-a439-00145eb45e9a"), limit=20)

# Search by recorder
occ_data(recordedBy="smith", limit=20)

# Many collector names
occ_data(recordedBy=c("smith","BJ Stacey"), limit=20)

# Pass in curl options for extra fun
library('httr')
occ_data(taxonKey=key, limit=20, config=verbose())
x <- occ_data(taxonKey=key, limit=50, config=progress())
# occ_data(taxonKey=key, limit=20, config=timeout(0.01))

# Search for many species
splist <- c('Cyanocitta stelleri', 'Junco hyemalis', 'Aix sponsa')
keys <- sapply(splist, function(x) name_suggest(x)$key[1], USE.NAMES=FALSE)
occ_data(taxonKey=keys, limit=5)

# Search using a synonym name
#  Note that you'll see a message printing out that the accepted name will be used
occ_data(scientificName = 'Pulsatilla patens', limit=5)

# Search on latitidue and longitude
occ_data(decimalLatitude=40, decimalLongitude=-120, limit = 10)

# Search on a bounding box
## in well known text format
occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))', limit=20)
key <- name_suggest(q='Aesculus hippocastanum')$key[1]
occ_data(taxonKey=key, geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))',
   limit=20)
## or using bounding box, converted to WKT internally
occ_data(geometry=c(-125.0,38.4,-121.8,40.9), limit=20)

## you can seaerch on many geometry objects
wkts <- c('POLYGON((-102.2 46.0,-93.9 46.0,-93.9 43.7,-102.2 43.7,-102.2 46.0))',
'POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))')
occ_data(geometry = wkts, limit=20)

# Search on a long WKT string - too long for a GBIF search API request
## By default, a very long WKT string will likely cause a request failure as
## GBIF only handles strings up to about 1500 characters long. You can leave as is, or
##  - Alternatively, you can choose to break up your polygon into many, and do a
##      data request on each piece, and the output is put back together (see below)
##  - Or, 2nd alternatively, you could use the GBIF download API
wkt <- "POLYGON((13.26349675655365 52.53991761181831,18.36115300655365 54.11445544219924,
21.87677800655365 53.80418956368524,24.68927800655365 54.217364774722455,28.20490300655365
54.320018299365124,30.49005925655365 52.85948216284084,34.70880925655365 52.753220564427814,
35.93927800655365 50.46131871049754,39.63068425655365 49.55761261299145,40.86115300655365
46.381388009130845,34.00568425655365 45.279102926537,33.30255925655365 48.636868465271846,
30.13849675655365 49.78513301801265,28.38068425655365 47.2236377039631,29.78693425655365
44.6572866068524,27.67755925655365 42.62220075124676,23.10724675655365 43.77542058000212,
24.51349675655365 47.10412345120368,26.79865300655365 49.55761261299145,23.98615300655365
52.00209943876426,23.63459050655365 49.44345313705238,19.41584050655365 47.580567827212114,
19.59162175655365 44.90682206053508,20.11896550655365 42.36297154876359,22.93146550655365
40.651849782081555,25.56818425655365 39.98171166226459,29.61115300655365 40.78507856230178,
32.95099675655365 40.38459278067577,32.95099675655365 37.37491910393631,26.27130925655365
33.65619609886799,22.05255925655365 36.814081996401605,18.71271550655365 36.1072176729021,
18.53693425655365 39.16878677351903,15.37287175655365 38.346355762190846,15.19709050655365
41.578843777436326,12.56037175655365 41.050735748143424,12.56037175655365 44.02872991212046,
15.19709050655365 45.52594200494078,16.42755925655365 48.05271546733352,17.48224675655365
48.86865641518059,10.62677800655365 47.817178329053135,9.57209050655365 44.154980365192,
8.16584050655365 40.51835445724746,6.05646550655365 36.53210972067291,0.9588092565536499
31.583640057148145,-5.54509699344635 35.68001485298146,-6.77556574344635 40.51835445724746,
-9.41228449344635 38.346355762190846,-12.40056574344635 35.10683619158607,-15.74040949344635
38.07010978950028,-14.68572199344635 41.31532459432774,-11.69744074344635 43.64836179231387,
-8.88494074344635 42.88035509418534,-4.31462824344635 43.52103366008421,-8.35759699344635
47.2236377039631,-8.18181574344635 50.12441989397795,-5.01775324344635 49.55761261299145,
-2.73259699344635 46.25998980446569,-1.67790949344635 44.154980365192,-1.32634699344635
39.30493590580802,2.18927800655365 41.44721797271696,4.47443425655365 43.26556960420879,
2.18927800655365 46.7439668697322,1.83771550655365 50.3492841273576,6.93537175655365
49.671505849335254,5.00177800655365 52.32557322466785,7.81427800655365 51.67627099802223,
7.81427800655365 54.5245591562317,10.97834050655365 51.89375191441792,10.97834050655365
55.43241335888528,13.26349675655365 52.53991761181831))"
wkt <- gsub("\\n", " ", wkt)

#### Default option with large WKT string fails
# res <- occ_data(geometry = wkt)

#### if WKT too long, with 'geom_big=bbox': makes into bounding box
res <- occ_data(geometry = wkt, geom_big = "bbox")
library("rgeos")
library("sp")
wktsp <- readWKT(wkt)
plot(wktsp)
coordinates(res$data) <- ~decimalLongitude+decimalLatitude
points(res$data)

#### Or, use 'geom_big=axe'
(res <- occ_data(geometry = wkt, geom_big = "axe"))
##### manipulate essentially number of polygons that result, so number of requests
###### default geom_size is 40
###### fewer calls
(res <- occ_data(geometry = wkt, geom_big = "axe", geom_size=50))
###### more calls
(res <- occ_data(geometry = wkt, geom_big = "axe", geom_size=30))

# Search on country
occ_data(country='US', limit=20)
isocodes[grep("France", isocodes$name),"code"]
occ_data(country='FR', limit=20)
occ_data(country='DE', limit=20)
occ_data(country=c('US','DE'), limit=20)

# Get only occurrences with lat/long data
occ_data(taxonKey=key, hasCoordinate=TRUE, limit=20)

# Get only occurrences that were recorded as living specimens
occ_data(basisOfRecord="LIVING_SPECIMEN", hasCoordinate=TRUE, limit=20)

# Get occurrences for a particular eventDate
occ_data(taxonKey=key, eventDate="2013", limit=20)
occ_data(taxonKey=key, year="2013", limit=20)
occ_data(taxonKey=key, month="6", limit=20)

# Get occurrences based on depth
key <- name_backbone(name='Salmo salar', kingdom='animals')$speciesKey
occ_data(taxonKey=key, depth=1, limit=20)

# Get occurrences based on elevation
key <- name_backbone(name='Puma concolor', kingdom='animals')$speciesKey
occ_data(taxonKey=key, elevation=50, hasCoordinate=TRUE, limit=20)

# Get occurrences based on institutionCode
occ_data(institutionCode="TLMF", limit=20)
occ_data(institutionCode=c("TLMF","ArtDatabanken"), limit=20)

# Get occurrences based on collectionCode
occ_data(collectionCode="Floristic Databases MV - Higher Plants", limit=20)
occ_data(collectionCode=c("Floristic Databases MV - Higher Plants","Artport"))

# Get only those occurrences with spatial issues
occ_data(taxonKey=key, hasGeospatialIssue=TRUE, limit=20)

# Search using a query string
occ_data(search="kingfisher", limit=20)

# Range queries
## See Detail for parameters that support range queries
### this is a range depth, with lower/upper limits in character string
occ_data(depth='50,100', limit = 20)
### this is not a range search, but does two searches for each depth
occ_data(depth=c(50,100), limit = 20)

## Range search with year
occ_data(year='1999,2000', limit=20)

## Range search with latitude
occ_data(decimalLatitude='29.59,29.6', limit = 20)

# Search by specimen type status
## Look for possible values of the typeStatus parameter looking at the typestatus dataset
occ_data(typeStatus = 'allotype', limit = 20)$data[,c('name','typeStatus')]

# Search by specimen record number
## This is the record number of the person/group that submitted the data, not GBIF's numbers
## You can see that many different groups have record number 1, so not super helpful
occ_data(recordNumber = 1, limit = 20)$data[,c('name','recordNumber','recordedBy')]

# Search by last time interpreted: Date the record was last modified in GBIF
## The lastInterpreted parameter accepts ISO 8601 format dates, including
## yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd. Range queries are accepted for lastInterpreted
occ_data(lastInterpreted = '2016-04-02', limit = 20)

# Search for occurrences with images
occ_data(mediaType = 'StillImage', limit = 20)
occ_data(mediaType = 'MovingImage', limit = 20)
occ_data(mediaType = 'Sound', limit = 20)

# Search by continent
## One of africa, antarctica, asia, europe, north_america, oceania, or south_america
occ_data(continent = 'south_america', limit = 20)$meta
occ_data(continent = 'africa', limit = 20)$meta
occ_data(continent = 'oceania', limit = 20)$meta
occ_data(continent = 'antarctica', limit = 20)$meta

# Query based on issues - see Details for options
## one issue
x <- occ_data(taxonKey=1, issue='DEPTH_UNLIKELY', limit = 20)
x$data[,c('name','key','decimalLatitude','decimalLongitude','depth')]
## two issues
occ_data(taxonKey=1, issue=c('DEPTH_UNLIKELY','COORDINATE_ROUNDED'), limit = 20)
# Show all records in the Arizona State Lichen Collection that cant be matched to the GBIF
# backbone properly:
occ_data(datasetKey='84c0e1a0-f762-11e1-a439-00145eb45e9a',
   issue=c('TAXON_MATCH_NONE','TAXON_MATCH_HIGHERRANK'), limit = 20)

# Parsing output by issue
(res <- occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))', limit = 50))
## what do issues mean, can print whole table, or search for matches
head(gbif_issues())
gbif_issues()[ gbif_issues()$code \%in\% c('cdround','cudc','gass84','txmathi'), ]
## or parse issues in various ways
### remove data rows with certain issue classes
library('magrittr')
res \%>\% occ_issues(gass84)
### split issues into separate columns
res \%>\% occ_issues(mutate = "split")
### expand issues to more descriptive names
res \%>\% occ_issues(mutate = "expand")
### split and expand
res \%>\% occ_issues(mutate = "split_expand")
### split, expand, and remove an issue class
res \%>\% occ_issues(-cudc, mutate = "split_expand")
}
}
\references{
\url{http://www.gbif.org/developer/occurrence#search}
}
\seealso{
\code{\link{downloads}}, \code{\link{occ_search}}
}

