% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/occ_data.R
\name{occ_data}
\alias{occ_data}
\title{Search for GBIF occurrences - simplified for speed}
\usage{
occ_data(taxonKey = NULL, scientificName = NULL, country = NULL,
  publishingCountry = NULL, hasCoordinate = NULL, typeStatus = NULL,
  recordNumber = NULL, lastInterpreted = NULL, continent = NULL,
  geometry = NULL, geom_big = "asis", geom_size = 40, geom_n = 10,
  recordedBy = NULL, basisOfRecord = NULL, datasetKey = NULL,
  eventDate = NULL, catalogNumber = NULL, year = NULL, month = NULL,
  decimalLatitude = NULL, decimalLongitude = NULL, elevation = NULL,
  depth = NULL, institutionCode = NULL, collectionCode = NULL,
  hasGeospatialIssue = NULL, issue = NULL, search = NULL,
  mediaType = NULL, subgenusKey = NULL, repatriated = NULL,
  phylumKey = NULL, kingdomKey = NULL, classKey = NULL, orderKey = NULL,
  familyKey = NULL, genusKey = NULL, establishmentMeans = NULL,
  protocol = NULL, license = NULL, organismId = NULL,
  publishingOrg = NULL, stateProvince = NULL, waterBody = NULL,
  locality = NULL, limit = 500, start = 0, spellCheck = NULL,
  curlopts = list())
}
\arguments{
\item{taxonKey}{(numeric) A taxon key from the GBIF backbone. All included and synonym taxa
are included in the search, so a search for aves with taxononKey=212
(i.e. /occurrence/search?taxonKey=212) will match all birds, no matter which
species. You can pass many keys by passing occ_search in a call to an
lapply-family function (see last example below).}

\item{scientificName}{A scientific name from the GBIF backbone. All included and synonym
taxa are included in the search.}

\item{country}{The 2-letter country code (as per ISO-3166-1) of the country in
which the occurrence was recorded. See here
<http://en.wikipedia.org/wiki/ISO_3166-1_alpha-2>}

\item{publishingCountry}{The 2-letter country code (as per ISO-3166-1) of the
country in which the occurrence was recorded.}

\item{hasCoordinate}{(logical) Return only occurence records with lat/long data (\code{TRUE}) or
all records (\code{FALSE}, default).}

\item{typeStatus}{Type status of the specimen. One of many options. See \code{?typestatus}}

\item{recordNumber}{Number recorded by collector of the data, different from GBIF record
number. See <http://rs.tdwg.org/dwc/terms/#recordNumber> for more info}

\item{lastInterpreted}{Date the record was last modified in GBIF, in ISO 8601 format:
yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd.  Supports range queries, smaller,larger (e.g.,
'1990,1991', whereas '1991,1990' wouldn't work)}

\item{continent}{Continent. One of africa, antarctica, asia, europe, north_america
(North America includes the Caribbean and reachies down and includes Panama), oceania,
or south_america}

\item{geometry}{Searches for occurrences inside a polygon described in Well Known
Text (WKT) format. A WKT shape written as either POINT, LINESTRING, LINEARRING
POLYGON, or MULTIPOLYGON. Example of a polygon: POLYGON((30.1 10.1, 20, 20 40, 40 40, 30.1 10.1))
would be queried as <http://bit.ly/1BzNwDq>. See also the section **WKT** below.}

\item{geom_big}{(character) One of "axe", "bbox", or "asis" (default). See Details.}

\item{geom_size}{(integer) An integer indicating size of the cell. Default: 40. See Details.}

\item{geom_n}{(integer) An integer indicating number of cells in each dimension. Default: 10.
See Details.}

\item{recordedBy}{The person who recorded the occurrence.}

\item{basisOfRecord}{Basis of record, as defined in our BasisOfRecord enum here
<http://gbif.github.io/gbif-api/apidocs/org/gbif/api/vocabulary/BasisOfRecord.html>
Acceptable values are:
\itemize{
  \item FOSSIL_SPECIMEN An occurrence record describing a fossilized specimen.
  \item HUMAN_OBSERVATION An occurrence record describing an observation made by
  one or more people.
  \item LITERATURE An occurrence record based on literature alone.
  \item LIVING_SPECIMEN An occurrence record describing a living specimen, e.g.
  \item MACHINE_OBSERVATION An occurrence record describing an observation made
  by a machine.
  \item OBSERVATION An occurrence record describing an observation.
  \item PRESERVED_SPECIMEN An occurrence record describing a preserved specimen.
  \item UNKNOWN Unknown basis for the record.
}}

\item{datasetKey}{The occurrence dataset key (a uuid)}

\item{eventDate}{Occurrence date in ISO 8601 format: yyyy, yyyy-MM, yyyy-MM-dd, or
MM-dd. Supports range queries, smaller,larger (e.g., '1990,1991', whereas '1991,1990'
wouldn't work)}

\item{catalogNumber}{An identifier of any form assigned by the source within a
physical collection or digital dataset for the record which may not unique,
but should be fairly unique in combination with the institution and collection code.}

\item{year}{The 4 digit year. A year of 98 will be interpreted as AD 98. Supports range queries,
smaller,larger (e.g., '1990,1991', whereas '1991,1990' wouldn't work)}

\item{month}{The month of the year, starting with 1 for January. Supports range queries,
smaller,larger (e.g., '1,2', whereas '2,1' wouldn't work)}

\item{decimalLatitude}{Latitude in decimals between -90 and 90 based on WGS 84.
Supports range queries, smaller,larger (e.g., '25,30', whereas '30,25' wouldn't work)}

\item{decimalLongitude}{Longitude in decimals between -180 and 180 based on WGS 84.
Supports range queries (e.g., '-0.4,-0.2', whereas '-0.2,-0.4' wouldn't work).}

\item{elevation}{Elevation in meters above sea level. Supports range queries, smaller,larger
(e.g., '5,30', whereas '30,5' wouldn't work)}

\item{depth}{Depth in meters relative to elevation. For example 10 meters below a
lake surface with given elevation. Supports range queries, smaller,larger (e.g., '5,30',
whereas '30,5' wouldn't work)}

\item{institutionCode}{An identifier of any form assigned by the source to identify
the institution the record belongs to. Not guaranteed to be que.}

\item{collectionCode}{An identifier of any form assigned by the source to identify
the physical collection or digital dataset uniquely within the text of an institution.}

\item{hasGeospatialIssue}{(logical) Includes/excludes occurrence records which contain spatial
issues (as determined in our record interpretation), i.e. \code{hasGeospatialIssue=TRUE}
returns only those records with spatial issues while \code{hasGeospatialIssue=FALSE} includes
only records without spatial issues. The absence of this parameter returns any
record with or without spatial issues.}

\item{issue}{(character) One or more of many possible issues with each occurrence record. See
Details. Issues passed to this parameter filter results by the issue.}

\item{search}{Query terms. The value for this parameter can be a simple word or a phrase.}

\item{mediaType}{Media type. Default is NULL, so no filtering on mediatype. Options:
NULL, 'MovingImage', 'Sound', and 'StillImage'.}

\item{subgenusKey}{(numeric) Subgenus classification key.}

\item{repatriated}{(character) Searches for records whose publishing country
is different to the country where the record was recorded in.}

\item{phylumKey}{(numeric) Phylum classification key.}

\item{kingdomKey}{(numeric) Kingdom classification key.}

\item{classKey}{(numeric) Class classification key.}

\item{orderKey}{(numeric) Order classification key.}

\item{familyKey}{(numeric) Family classification key.}

\item{genusKey}{(numeric) Genus classification key.}

\item{establishmentMeans}{(character) EstablishmentMeans, possible values
include: INTRODUCED, INVASIVE, MANAGED, NATIVE, NATURALISED, UNCERTAIN}

\item{protocol}{(character) Protocol or mechanism used to provide the
occurrence record. See Details for possible values}

\item{license}{(character) The type license applied to the dataset or record.
Possible values: CC0_1_0, CC_BY_4_0, CC_BY_NC_4_0, UNSPECIFIED, and
UNSUPPORTED}

\item{organismId}{(numeric) An identifier for the Organism instance (as
opposed to a particular digital record of the Organism). May be a globally
unique identifier or an identifier specific to the data set.}

\item{publishingOrg}{(character) The publishing organization key (a UUID).}

\item{stateProvince}{(character) The name of the next smaller administrative
region than country (state, province, canton, department, region, etc.) in
which the Location occurs.}

\item{waterBody}{(character) The name of the water body in which the
locations occur}

\item{locality}{(character) The specific description of the place.}

\item{limit}{Number of records to return. Default: 500. Note that the per
request maximum is 300, but since we set it at 500 for the function, we
do two requests to get you the 500 records (if there are that many).
Note that there is a hard maximum of 200,000, which is calculated as the
\code{limit+start}, so \code{start=199,000} and \code{limit=2000} won't work}

\item{start}{Record number to start at. Use in combination with limit to
page through results. Note that we do the paging internally for you, but
you can manually set the \code{start} parameter}

\item{spellCheck}{(logical) If \code{TRUE} ask GBIF to check your spelling of
the value passed to the \code{search} parameter. IMPORTANT: This only checks
the input to the \code{search} parameter, and no others. Default: \code{FALSE}}

\item{curlopts}{list of named curl options passed on to
\code{\link[crul]{HttpClient}}. see \code{\link[curl]{curl_options}}
for curl options}

\item{...}{additional facet parameters}
}
\value{
An object of class \code{gbif_data}, which is a S3 class list, with
slots for metadata (\code{meta}) and the occurrence data itself
(\code{data}),
and with attributes listing the user supplied arguments and whether it was a
"single" or "many" search; that is, if you supply two values of the
\code{datasetKey} parameter to searches are done, and it's a "many".
\code{meta} is a list of length four with offset, limit, endOfRecords and
count fields. \code{data} is a tibble (aka data.frame)
}
\description{
Search for GBIF occurrences - simplified for speed
}
\details{
This does nearly the same thing as \code{\link[=occ_search]{occ_search()}}, but
is a bit simplified for speed, and is for the most common use case where
user just wants the data, and not other information like taxon hierarchies
and media (e.g., images) information. Alot of time in
\code{\link[=occ_search]{occ_search()}} is used parsing data to be more useable
downstream. We do less of that in this function.
}
\examples{
\dontrun{
(key <- name_backbone(name='Encelia californica')$speciesKey)
occ_data(taxonKey = key, limit = 4)
(res <- occ_data(taxonKey = key, limit = 400))

# Return 20 results, this is the default by the way
(key <- name_suggest(q='Helianthus annuus', rank='species')$key[1])
occ_data(taxonKey=key, limit=20)

# Instead of getting a taxon key first, you can search for a name directly
## However, note that using this approach (with \\code{scientificName="..."})
## you are getting synonyms too. The results for using \\code{scientifcName}
## and \\code{taxonKey} parameters are the same in this case, but I wouldn't
## be surprised if for some names they return different results
occ_data(scientificName = 'Ursus americanus', curlopts=list(verbose=TRUE))
key <- name_backbone(name = 'Ursus americanus', rank='species')$usageKey
occ_data(taxonKey = key)

# Search by dataset key
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a', limit=10)

# Search by catalog number
occ_data(catalogNumber="49366", limit=10)
## separate requests: use a vector of strings
occ_data(catalogNumber=c("49366","Bird.27847588"), limit=10)
## one request, many instances of same parameter: use semi-colon sep. string
occ_data(catalogNumber="49366;Bird.27847588", limit=10)

# Use paging parameters (limit and start) to page. Note the different results
# for the two queries below.
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=10,limit=5)
occ_data(datasetKey='7b5d6a48-f762-11e1-a439-00145eb45e9a',start=20,limit=5)

# Many dataset keys
## separate requests: use a vector of strings
occ_data(datasetKey=c("50c9509d-22c7-4a22-a47d-8c48425ef4a7",
   "7b5d6a48-f762-11e1-a439-00145eb45e9a"), limit=20)
## one request, many instances of same parameter: use semi-colon sep. string
v="50c9509d-22c7-4a22-a47d-8c48425ef4a7;7b5d6a48-f762-11e1-a439-00145eb45e9a"
occ_data(datasetKey = v, limit=20)

# Search by recorder
occ_data(recordedBy="smith", limit=20)

# Many collector names
## separate requests: use a vector of strings
occ_data(recordedBy=c("smith","BJ Stacey"), limit=10)
## one request, many instances of same parameter: use semi-colon sep. string
occ_data(recordedBy="smith;BJ Stacey", limit=10)

# Pass in curl options for extra fun
occ_data(taxonKey=2433407, limit=20, curlopts=list(verbose=TRUE))
occ_data(taxonKey=2433407, limit=20,
  curlopts = list(
    noprogress = FALSE,
    progressfunction = function(down, up) {
      cat(sprintf("up: \%d | down \%d\\n", up, down))
      return(TRUE)
    }
  )
)
# occ_data(taxonKey=2433407, limit=20, curlopts=list(timeout_ms=1))

# Search for many species
splist <- c('Cyanocitta stelleri', 'Junco hyemalis', 'Aix sponsa')
keys <- sapply(splist, function(x) name_suggest(x)$key[1], USE.NAMES=FALSE)
## separate requests: use a vector of strings
occ_data(taxonKey = keys, limit=5)
## one request, many instances of same parameter: use semi-colon sep. string
occ_data(taxonKey = paste0(keys, collapse = ";"), limit=5)

# Search using a synonym name
#  Note that you'll see a message printing out that the accepted name will
# be used
occ_data(scientificName = 'Pulsatilla patens', limit=5)

# Search on latitidue and longitude
occ_data(decimalLatitude=40, decimalLongitude=-120, limit = 10)

# Search on a bounding box
## in well known text format
### polygon
occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))',
  limit=20)
### multipolygon
wkt <- 'MULTIPOLYGON(((-123 38, -123 43, -116 43, -116 38, -123 38)),
   ((-97 41, -97 45, -93 45, -93 41, -97 41)))'
occ_data(geometry = gsub("\\n\\\\s+", "", wkt), limit = 20)
### polygon and taxonkey
key <- name_suggest(q='Aesculus hippocastanum')$key[1]
occ_data(taxonKey=key,
 geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))',
 limit=20)
## or using bounding box, converted to WKT internally
occ_data(geometry=c(-125.0,38.4,-121.8,40.9), limit=20)

## you can seaerch on many geometry objects
### separate requests: use a vector of strings
wkts <-
c('POLYGON((-102.2 46.0,-93.9 46.0,-93.9 43.7,-102.2 43.7,-102.2 46.0))',
'POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))')
occ_data(geometry = wkts, limit=20)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(geometry = paste0(wkts, collapse = ";"), limit=20)


# Search on a long WKT string - too long for a GBIF search API request
## By default, a very long WKT string will likely cause a request failure as
## GBIF only handles strings up to about 1500 characters long. You can leave as is, or
##  - Alternatively, you can choose to break up your polygon into many, and do a
##      data request on each piece, and the output is put back together (see below)
##  - Or, 2nd alternatively, you could use the GBIF download API
wkt <- "POLYGON((13.26349675655365 52.53991761181831,18.36115300655365 54.11445544219924,
21.87677800655365 53.80418956368524,24.68927800655365 54.217364774722455,28.20490300655365
54.320018299365124,30.49005925655365 52.85948216284084,34.70880925655365 52.753220564427814,
35.93927800655365 50.46131871049754,39.63068425655365 49.55761261299145,40.86115300655365
46.381388009130845,34.00568425655365 45.279102926537,33.30255925655365 48.636868465271846,
30.13849675655365 49.78513301801265,28.38068425655365 47.2236377039631,29.78693425655365
44.6572866068524,27.67755925655365 42.62220075124676,23.10724675655365 43.77542058000212,
24.51349675655365 47.10412345120368,26.79865300655365 49.55761261299145,23.98615300655365
52.00209943876426,23.63459050655365 49.44345313705238,19.41584050655365 47.580567827212114,
19.59162175655365 44.90682206053508,20.11896550655365 42.36297154876359,22.93146550655365
40.651849782081555,25.56818425655365 39.98171166226459,29.61115300655365 40.78507856230178,
32.95099675655365 40.38459278067577,32.95099675655365 37.37491910393631,26.27130925655365
33.65619609886799,22.05255925655365 36.814081996401605,18.71271550655365 36.1072176729021,
18.53693425655365 39.16878677351903,15.37287175655365 38.346355762190846,15.19709050655365
41.578843777436326,12.56037175655365 41.050735748143424,12.56037175655365 44.02872991212046,
15.19709050655365 45.52594200494078,16.42755925655365 48.05271546733352,17.48224675655365
48.86865641518059,10.62677800655365 47.817178329053135,9.57209050655365 44.154980365192,
8.16584050655365 40.51835445724746,6.05646550655365 36.53210972067291,0.9588092565536499
31.583640057148145,-5.54509699344635 35.68001485298146,-6.77556574344635 40.51835445724746,
-9.41228449344635 38.346355762190846,-12.40056574344635 35.10683619158607,-15.74040949344635
38.07010978950028,-14.68572199344635 41.31532459432774,-11.69744074344635 43.64836179231387,
-8.88494074344635 42.88035509418534,-4.31462824344635 43.52103366008421,-8.35759699344635
47.2236377039631,-8.18181574344635 50.12441989397795,-5.01775324344635 49.55761261299145,
-2.73259699344635 46.25998980446569,-1.67790949344635 44.154980365192,-1.32634699344635
39.30493590580802,2.18927800655365 41.44721797271696,4.47443425655365 43.26556960420879,
2.18927800655365 46.7439668697322,1.83771550655365 50.3492841273576,6.93537175655365
49.671505849335254,5.00177800655365 52.32557322466785,7.81427800655365 51.67627099802223,
7.81427800655365 54.5245591562317,10.97834050655365 51.89375191441792,10.97834050655365
55.43241335888528,13.26349675655365 52.53991761181831))"
wkt <- gsub("\\n", " ", wkt)

#### Default option with large WKT string fails
# res <- occ_data(geometry = wkt)

#### if WKT too long, with 'geom_big=bbox': makes into bounding box
res <- occ_data(geometry = wkt, geom_big = "bbox")
library("rgeos")
library("sp")
wktsp <- readWKT(wkt)
plot(wktsp)
coordinates(res$data) <- ~decimalLongitude+decimalLatitude
points(res$data)

#### Or, use 'geom_big=axe'
(res <- occ_data(geometry = wkt, geom_big = "axe"))
##### manipulate essentially number of polygons that result, so number of requests
###### default geom_size is 40
###### fewer calls
(res <- occ_data(geometry = wkt, geom_big = "axe", geom_size=50))
###### more calls
(res <- occ_data(geometry = wkt, geom_big = "axe", geom_size=30))

# Search on country
occ_data(country='US', limit=20)
isocodes[grep("France", isocodes$name),"code"]
occ_data(country='FR', limit=20)
occ_data(country='DE', limit=20)
### separate requests: use a vector of strings
occ_data(country=c('US','DE'), limit=20)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(country = 'US;DE', limit=20)

# Get only occurrences with lat/long data
occ_data(taxonKey=key, hasCoordinate=TRUE, limit=20)

# Get only occurrences that were recorded as living specimens
occ_data(basisOfRecord="LIVING_SPECIMEN", hasCoordinate=TRUE, limit=20)

# Get occurrences for a particular eventDate
occ_data(taxonKey=key, eventDate="2013", limit=20)
occ_data(taxonKey=key, year="2013", limit=20)
occ_data(taxonKey=key, month="6", limit=20)

# Get occurrences based on depth
key <- name_backbone(name='Salmo salar', kingdom='animals')$speciesKey
occ_data(taxonKey=key, depth=1, limit=20)

# Get occurrences based on elevation
key <- name_backbone(name='Puma concolor', kingdom='animals')$speciesKey
occ_data(taxonKey=key, elevation=50, hasCoordinate=TRUE, limit=20)

# Get occurrences based on institutionCode
occ_data(institutionCode="TLMF", limit=20)
### separate requests: use a vector of strings
occ_data(institutionCode=c("TLMF","ArtDatabanken"), limit=20)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(institutionCode = "TLMF;ArtDatabanken", limit=20)

# Get occurrences based on collectionCode
occ_data(collectionCode="Floristic Databases MV - Higher Plants", limit=20)
### separate requests: use a vector of strings
occ_data(collectionCode=c("Floristic Databases MV - Higher Plants",
  "Artport"), limit = 20)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(collectionCode = "Floristic Databases MV - Higher Plants;Artport",
  limit = 20)

# Get only those occurrences with spatial issues
occ_data(taxonKey=key, hasGeospatialIssue=TRUE, limit=20)

# Search using a query string
occ_data(search="kingfisher", limit=20)
## spell check - only works with the `search` parameter
### spelled correctly - same result as above call
occ_data(search = "kingfisher", limit=20, spellCheck = TRUE)
### spelled incorrectly - stops with suggested spelling
# occ_data(search = "kajsdkla", limit=20, spellCheck = TRUE)
### spelled incorrectly - stops with many suggested spellings
###   and number of results for each
# occ_data(search = "helir", limit=20, spellCheck = TRUE)

# search on repatriated - doesn't work right now
# occ_data(repatriated = "")

# search on phylumKey
occ_data(phylumKey = 7707728, limit = 5)

# search on kingdomKey
occ_data(kingdomKey = 1, limit = 5)

# search on classKey
occ_data(classKey = 216, limit = 5)

# search on orderKey
occ_data(orderKey = 7192402, limit = 5)

# search on familyKey
occ_data(familyKey = 3925, limit = 5)

# search on genusKey
occ_data(genusKey = 1935496, limit = 5)

# search on establishmentMeans
occ_data(establishmentMeans = "INVASIVE", limit = 5)
occ_data(establishmentMeans = "NATIVE", limit = 5)
occ_data(establishmentMeans = "UNCERTAIN", limit = 5)
### separate requests: use a vector of strings
occ_data(establishmentMeans = c("INVASIVE", "NATIVE"), limit = 5)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(establishmentMeans = "INVASIVE;NATIVE", limit = 5)

# search on protocol
occ_data(protocol = "DIGIR", limit = 5)

# search on license
occ_data(license = "CC_BY_4_0", limit = 5)

# search on organismId
occ_data(organismId = "100", limit = 5)

# search on publishingOrg
occ_data(publishingOrg = "28eb1a3f-1c15-4a95-931a-4af90ecb574d", limit = 5)

# search on stateProvince
occ_data(stateProvince = "California", limit = 5)

# search on waterBody
occ_data(waterBody = "pacific ocean", limit = 5)

# search on locality
occ_data(locality = "Trondheim", limit = 5)
### separate requests: use a vector of strings
res <- occ_data(locality = c("Trondheim", "Hovekilen"), limit = 5)
res$Trondheim$data
res$Hovekilen$data
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(locality = "Trondheim;Hovekilen", limit = 5)


# Range queries
## See Detail for parameters that support range queries
occ_data(depth='50,100', limit = 20)
### this is not a range search, but does two searches for each depth
occ_data(depth=c(50,100), limit = 20)

## Range search with year
occ_data(year='1999,2000', limit=20)

## Range search with latitude
occ_data(decimalLatitude='29.59,29.6', limit = 20)

# Search by specimen type status
## Look for possible values of the typeStatus parameter looking at the typestatus dataset
occ_data(typeStatus = 'allotype', limit = 20)$data[,c('name','typeStatus')]

# Search by specimen record number
## This is the record number of the person/group that submitted the data, not GBIF's numbers
## You can see that many different groups have record number 1, so not super helpful
occ_data(recordNumber = 1, limit = 20)$data[,c('name','recordNumber','recordedBy')]

# Search by last time interpreted: Date the record was last modified in GBIF
## The lastInterpreted parameter accepts ISO 8601 format dates, including
## yyyy, yyyy-MM, yyyy-MM-dd, or MM-dd. Range queries are accepted for lastInterpreted
occ_data(lastInterpreted = '2016-04-02', limit = 20)

# Search for occurrences with images
occ_data(mediaType = 'StillImage', limit = 20)
occ_data(mediaType = 'MovingImage', limit = 20)
occ_data(mediaType = 'Sound', limit = 20)

# Search by continent
## One of africa, antarctica, asia, europe, north_america, oceania, or
## south_america
occ_data(continent = 'south_america', limit = 20)$meta
occ_data(continent = 'africa', limit = 20)$meta
occ_data(continent = 'oceania', limit = 20)$meta
occ_data(continent = 'antarctica', limit = 20)$meta
### separate requests: use a vector of strings
occ_data(continent = c('south_america', 'oceania'), limit = 20)
### one request, many instances of same parameter: use semi-colon sep. string
occ_data(continent = 'south_america;oceania', limit = 20)

# Query based on issues - see Details for options
## one issue
x <- occ_data(taxonKey=1, issue='DEPTH_UNLIKELY', limit = 20)
x$data[,c('name','key','decimalLatitude','decimalLongitude','depth')]
## two issues
occ_data(taxonKey=1, issue=c('DEPTH_UNLIKELY','COORDINATE_ROUNDED'), limit = 20)
# Show all records in the Arizona State Lichen Collection that cant be matched to the GBIF
# backbone properly:
occ_data(datasetKey='84c0e1a0-f762-11e1-a439-00145eb45e9a',
   issue=c('TAXON_MATCH_NONE','TAXON_MATCH_HIGHERRANK'), limit = 20)

# Parsing output by issue
(res <- occ_data(geometry='POLYGON((30.1 10.1, 10 20, 20 40, 40 40, 30.1 10.1))', limit = 50))
## what do issues mean, can print whole table, or search for matches
head(gbif_issues())
gbif_issues()[ gbif_issues()$code \%in\% c('cdround','cudc','gass84','txmathi'), ]
## or parse issues in various ways
### remove data rows with certain issue classes
library('magrittr')
res \%>\% occ_issues(gass84)
### split issues into separate columns
res \%>\% occ_issues(mutate = "split")
### expand issues to more descriptive names
res \%>\% occ_issues(mutate = "expand")
### split and expand
res \%>\% occ_issues(mutate = "split_expand")
### split, expand, and remove an issue class
res \%>\% occ_issues(-cudc, mutate = "split_expand")
}
}
\references{
\url{http://www.gbif.org/developer/occurrence#search}
}
\seealso{
\code{\link[=downloads]{downloads()}}, \code{\link[=occ_search]{occ_search()}}
}
