% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sf_as_ee.R
\name{sf_as_ee}
\alias{sf_as_ee}
\title{Convert an sf object to an EE object}
\usage{
sf_as_ee(
  x,
  via = "getInfo",
  assetId = NULL,
  bucket = NULL,
  command_line_tool_path = NULL,
  overwrite = TRUE,
  monitoring = TRUE,
  proj = "EPSG:4326",
  evenOdd = TRUE,
  geodesic = NULL,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{x}{object of class sf, sfc or sfg.}

\item{via}{Character. Upload method for sf objects. Three methods are
implemented: 'getInfo', 'getInfo_to_asset' and 'gcs_to_asset'. See details.}

\item{assetId}{Character. Destination asset ID for the uploaded file. Ignore
if \code{via} argument is "getInfo".}

\item{bucket}{Character. Name of the bucket (GCS) to save intermediate files
(ignore if \code{via} is not defined as "gcs_to_asset").}

\item{command_line_tool_path}{Character. Path to the Earth Engine command line
tool (CLT). If NULL, rgee assumes that CLT is set in the system PATH.
(ignore if \code{via} is not defined as "gcs_to_asset").}

\item{overwrite}{A boolean argument that indicates indicating
whether "filename" should be overwritten. Ignore if \code{via} argument
is "getInfo". By default TRUE.}

\item{monitoring}{Logical. Ignore if via is not set as
\code{getInfo_to_asset} or \code{gcs_to_asset}. If TRUE the exportation task
will be monitored.}

\item{proj}{Integer or character. Coordinate Reference System (CRS) for the
EE object, defaults to "EPSG:4326" (x=longitude, y=latitude).}

\item{evenOdd}{Logical. Ignored if \code{x} is not a Polygon. If TRUE,
polygon interiors will be determined by the even/odd rule, where a point
is inside if it crosses an odd number of edges to reach a point at infinity.
Otherwise polygons use the left-inside rule, where interiors are on the
left side of the shell's edges when walking the vertices in the given order.
If unspecified, defaults to TRUE.}

\item{geodesic}{Logical. Ignored if \code{x} is not a Polygon or LineString.
Whether line segments should be interpreted as spherical geodesics. If
FALSE, indicates that line segments should be interpreted as planar lines
in the specified CRS. If absent, defaults to TRUE if the CRS is geographic
(including the default EPSG:4326), or to FALSE if the CRS is projected.}

\item{quiet}{Logical. Suppress info message.}

\item{...}{\code{\link{ee_utils_create_manifest_table}} arguments might be included.}
}
\value{
When \code{via} is "getInfo" and \code{x} is either an sf or sfc object
with multiple geometries will return an \code{ee$FeatureCollection}. For
single sfc and sfg objects will return an \code{ee$Geometry$...}.

If \code{via} is either "getInfo_to_asset" or "gcs_to_asset" always will
return an \code{ee$FeatureCollection}.
}
\description{
Load an sf object to Earth Engine.
}
\details{
\code{sf_as_ee} supports the upload of \code{sf} objects by three different
options: "getInfo" (default), "getInfo_to_asset", and "gcs_to_asset". \code{getInfo}
transforms sf objects (sfg, sfc, or sf) to GeoJSON (using \code{geojsonio::geojson_json})
and then encrusted them in an HTTP request using the server-side objects that are
implemented in the Earth Engine API (i.e. ee$Geometry$...). If the sf object is too
large (~ >1Mb) is likely to cause bottlenecks since it is a temporary
file that is not saved in your EE Assets (server-side). The second option implemented
is 'getInfo_to_asset'. It is similar to the previous one, with the difference
that after create the server-side object will save it in your Earth Engine
Assets. For dealing with very large spatial objects is preferable to use the
third option 'gcs_to_asset'. This option firstly saves the sf object as a *.shp
file in the /temp directory. Secondly, using the function \code{local_to_gcs}
will move the shapefile from local to Google Cloud Storage. Finally, using
the function \code{gcs_to_ee_table} the ESRI shapefile will be loaded
to their EE Assets. See \href{https://developers.google.com/earth-engine/guides/table_upload/}{Importing
table data} documentation for more details.
}
\examples{
\dontrun{
library(rgee)
library(sf)
ee_Initialize()

# 1. Handling geometry parameters
# Simple
ee_x <- st_read(system.file("shape/nc.shp", package = "sf")) \%>\%
  sf_as_ee()

Map$centerObject(eeObject = ee_x)
Map$addLayer(ee_x)

# Create a right-inside polygon.
toy_poly <- matrix(data = c(-35,-10,-35,10,35,10,35,-10,-35,-10),
                   ncol = 2,
                   byrow = TRUE) \%>\%
  list() \%>\%
  st_polygon()

holePoly <- sf_as_ee(x = toy_poly, evenOdd = FALSE)

# Create an even-odd version of the polygon.
evenOddPoly <- sf_as_ee(toy_poly, evenOdd = TRUE)

# Create a point to test the insideness of the polygon.
pt <- ee$Geometry$Point(c(1.5, 1.5))

# Check insideness with a contains operator.
print(holePoly$contains(pt)$getInfo() \%>\% ee_utils_py_to_r())
print(evenOddPoly$contains(pt)$getInfo() \%>\% ee_utils_py_to_r())

# 2. Upload small geometries to EE asset
assetId <- sprintf("\%s/\%s", ee_get_assethome(), 'toy_poly')
eex <- sf_as_ee(
 x = toy_poly,
 overwrite = TRUE,
 assetId = assetId,
 via = "getInfo_to_asset")
# 3. Upload large geometries to EE asset
ee_Initialize(gcs = TRUE)
assetId <- sprintf("\%s/\%s", ee_get_assethome(), 'toy_poly_gcs')
eex <- sf_as_ee(
  x = toy_poly,
  overwrite = TRUE,
  assetId = assetId,
  bucket = 'rgee_dev',
  monitoring = FALSE,
  via = 'gcs_to_asset'
)
ee_monitoring()
}
}
